/* Mbdofcamera.cc
 * ---------------
 * Xinru Woo,  CS348b
 */


#include "mbdofcamera.h"
#include "geometry.h"
#include "scene.h"
#include "image.h"

#define PI  3.141592654


bool MbDOFCamera::GenerateRay(Float sample[5], Ray & ray) const
{

  /* Crop against window */

  if (sample[0] < scene->image->SampleCropLeft ||
      sample[0] > scene->image->SampleCropRight ||
      sample[1] < scene->image->SampleCropBottom ||
      sample[1] > scene->image->SampleCropTop) return false;
  

  /* Generate pinhole camera eye ray */

  Point Pcamera;
  switch (ProjectionType) {
    case Camera::Orthographic:
      Pcamera = RasterToCamera(Point(sample[0], sample[1], 0));
      ray = Ray(Pcamera, Vector(0, 0, 1));
      break;
    case Camera::Perspective:
      Pcamera = RasterToCamera(Point(sample[0], sample[1], 0));
      ray = Ray(Point(0, 0, 0), Vector(Pcamera.x, Pcamera.y, Pcamera.z));
      ray.D *= invClipHither;
      break;
  }

  // At this point, ray is the pinhole camera generated ray

  Float jit = ShutterStart + ( ShutterEnd - ShutterStart ) * sample[4];   
  // sample[4] is the interpolation between time ShutterStart and ShutterEnd


  /*********************************************************************
    Shutter STRIPE - determine time range for this point's visible
                     integral. Choose random time in this range.
                     change jit to new time interpolation value.
  **********************************************************************/

  if( ShutterType == LRT_STRIPE ) {

    Float t0 = 0;
    Float t1 = 1;

    Point Pscreen = RasterToScreen(Point(sample[0], sample[1], 0));

    // Normalized x, y to the range [0,1]
    Pscreen.x += -ScreenLeft;
    Pscreen.x /= (Float) (ScreenRight - ScreenLeft);
    Pscreen.y += -ScreenBottom;
    Pscreen.y /= (Float) (ScreenTop - ScreenBottom);  
    
    if( StripeDirection == LRT_UP ) {
	t0 = Pscreen.y / ( 1.0 + StripeWidth );
	t1 = ( Pscreen.y + StripeWidth ) / ( 1.0 + StripeWidth );
    }
    else if( StripeDirection == LRT_DOWN ) {
	t0 = 1 - (( Pscreen.y + StripeWidth ) / ( 1.0 + StripeWidth ));
	t1 = 1 - ( Pscreen.y / ( 1.0 + StripeWidth ));
    }
    else if( StripeDirection == LRT_RIGHT ) {
	t0 = Pscreen.x / ( 1.0 + StripeWidth );
	t1 = ( Pscreen.x + StripeWidth ) / ( 1.0 + StripeWidth );
    }
    else if( StripeDirection == LRT_LEFT ) {
	t0 = 1 - (( Pscreen.x + StripeWidth ) / ( 1.0 + StripeWidth ));
	t1 = 1 - ( Pscreen.x / ( 1.0 + StripeWidth ));
    }

    jit = t0 + ( t1 - t0 ) * sample[4];
  }



  /*******************************************************************
    Motion Blur:  If there is no motion blur, CameraToWorld 0 and 1 are
                  the same and these transformations will leave ray the
                  same as it was before. Only Translational motion is 
		  allowed for the positions at time t0 and t1.
   ********************************************************************/

  Ray rayt0 = CameraToWorld[0] (ray);  // position at time t0
  Ray rayt1 = CameraToWorld[1] (ray);  // position at time t1

  ray.O.x = rayt0.O.x * ( 1 - jit ) + rayt1.O.x * jit;
  ray.O.y = rayt0.O.y * ( 1 - jit ) + rayt1.O.y * jit;
  ray.O.z = rayt0.O.z * ( 1 - jit ) + rayt1.O.z * jit;
  
  ray.D.x = rayt0.D.x * ( 1 - jit ) + rayt1.D.x * jit;
  ray.D.y = rayt0.D.y * ( 1 - jit ) + rayt1.D.y * jit;
  ray.D.z = rayt0.D.z * ( 1 - jit ) + rayt1.D.z * jit;



  /*********************************************************************
    Shutter IRIS - determine size of aperature
  **********************************************************************/

  // radius is set to full radius of lens.
  Float radius = 0.5 * ((Float) FocalLength ) / FStop; 
 
  if( ShutterType == LRT_IRIS) {
    Float time = sample[4]; 
    if( IrisRate >= 2 ) {
      if( time < ( 1.0 / IrisRate ) ) 
	radius = IrisRate * time * radius;
      else if( time > (( IrisRate - 1.0 ) / IrisRate ) ) 
	radius = IrisRate * ( 1.0 - time ) * radius; 
    } 
  }


  /**************************************************************
    Depth of Field: only executed if aperature isn't infinity
   **************************************************************/

  if( FStop != RI_INFINITY ) {
    
    /* find point P, focal point for rays */
    Point P;
    Float t = ( FocalDistance - ray.O.z ) / ray.D.z;
    P.x = ray.O.x + ray.D.x * t;
    P.y = ray.O.y + ray.D.y * t;
    P.z = FocalDistance;
    
    // Generate jittered position
    Float r = radius * sqrt( sample[2] );  
    Float theta = 2.0 * PI * sample[3];
    ray.O.x = ray.O.x + r * cos( theta );
    ray.O.y = ray.O.y + r * sin( theta );
    /* ray.O.z stays the same */

    // generate ray from jittered position to P 
    ray.D.x = P.x - ray.O.x;
    ray.D.y = P.y - ray.O.y;
    ray.D.z = P.z - ray.O.z; 
  }

  return true; 
}

