
#include "trimesh.h"

TriangleMesh::TriangleMesh(int nt, int nv, int *vi, Point * P,
						   PrimitiveAttributes * a, SurfaceFunction * sf)
:Primitive(a, sf)
{
	ntris = nt;
	nverts = nv;
	vertexIndex = new int[3 * ntris];
	memcpy(vertexIndex, vi, 3 * ntris * sizeof(int));
	p = new Point[nverts];
	memcpy(p, P, nverts * sizeof(Point));
}

TriangleMesh::~TriangleMesh()
{
	delete[]vertexIndex;
	delete[]p;
}

BBox TriangleMesh::BoundObjectSpace() const
{
	BBox objectSpaceBounds(p[0], p[1]);
	for (int i = 2; i < nverts; i++)
		objectSpaceBounds = Union(objectSpaceBounds, p[i]);
	return objectSpaceBounds;
}

BBox TriangleMesh::BoundWorldSpace() const
{
	BBox worldBounds(attributes->ObjectToWorld(p[0]),
					 attributes->ObjectToWorld(p[1]));
	for (int i = 2; i < nverts; i++) {
		worldBounds = Union(worldBounds, attributes->ObjectToWorld(p[i]));
	}
	return worldBounds;
}

void TriangleMesh::Refine(vector < Primitive * >*refined) const
{
	refined->reserve(ntris);
	for (int i = 0; i < ntris; ++i)
		refined->push_back(new Triangle(this, i));
}

BBox Triangle::BoundObjectSpace() const
{
	BBox objectBounds(mesh->p[mesh->vertexIndex[triNum * 3]],
					  mesh->p[mesh->vertexIndex[triNum * 3 + 1]]);
	return Union(objectBounds, mesh->p[mesh->vertexIndex[triNum * 3 + 2]]);
}

BBox Triangle::BoundWorldSpace() const
{
	Transform o2w = mesh->attributes->ObjectToWorld;
	BBox objectBounds(o2w(mesh->p[mesh->vertexIndex[triNum * 3]]),
					  o2w(mesh->p[mesh->vertexIndex[triNum * 3 + 1]]));
	return Union(objectBounds,
				 o2w(mesh->p[mesh->vertexIndex[triNum * 3 + 2]]));
}


int  
Triangle::RearrangeTriPoints(Point *P0,Point *P1,Point *P2) const
{
  /* For P0 we want the one that is NOT a part of the longest side */
  Vector V10 = *P1 - *P0;
  Vector V21 = *P2 - *P1;
  Vector V20 = *P2 - *P0; 
  if(V10.LengthSquared() > V20.LengthSquared() 
      && V10.LengthSquared() > V21.LengthSquared())
    {
      Point Temp;
      Temp = *P0;
      *P0 = *P2;
      *P2 = Temp;

      V10 = *P1 - *P0;
      V21 = *P2 - *P1;
      V20 = *P2 - *P0; 
    }
  else if(V20.LengthSquared() > V21.LengthSquared() 
      && V20.LengthSquared() > V10.LengthSquared())
    {
      Point Temp;
      Temp = *P0;
      *P0 = *P1;
      *P1 = Temp;

      V10 = *P1 - *P0;
      V21 = *P2 - *P1;
      V20 = *P2 - *P0;
    }

  /* Now that we have found where P0 is to be, we
     potentialy need to switch P1 and P2 */
  Vector VJudge(1,0.5,3);

  if(fabs(Dot(VJudge,V10)) < fabs(Dot(VJudge,V20)))
    {
      Point Temp;
      Temp = *P2;
      *P2 = *P1;
      *P1 = Temp;
      V10 = *P1 - *P0;
      V21 = *P2 - *P1;
      V20 = *P2 - *P0;
    }
  if(Dot(VJudge,V10) < 0)
    return 1;
  return 0;
}

bool Triangle::IntersectClosest(const Ray & ray, Float mint,
								Float * maxt, HitInfo * hit) const
{

	static int triangleTests = 0, triangleHits = 0;
	if (triangleTests == 0)
		StatsRegisterRatio(STATS_DETAILED, "Geometry",
						   "Triangle Ray Intersections", &triangleHits,
						   &triangleTests);

	++triangleTests;

	int *vptr = &(mesh->vertexIndex[3 * triNum]);
	int vertexIndex0 = *vptr++;
	int vertexIndex1 = *vptr++;
	int vertexIndex2 = *vptr++;

	const Point *p = mesh->p;

	Point P0 = p[vertexIndex0];
	Point P1 = p[vertexIndex1];
	Point P2 = p[vertexIndex2];

	int NeedInversion = RearrangeTriPoints(&P0,&P1,&P2);
	Vector E1 = P1 - P0;
	Vector E2 = P2 - P0;
	Vector S_1 = Cross(ray.D, E2);
	Float divisor = Dot(S_1, E1);
	if (divisor == 0.)
		return false;
	Float invDivisor = 1. / divisor;

	Vector T = ray.O - P0;
	Float u = Dot(T, S_1) * invDivisor;
	if (u < 0. || u > 1.0)
		return false;

	Vector S_2 = Cross(T, E1);
	Float v = Dot(ray.D, S_2) * invDivisor;
	if (v < 0 || u + v > 1.0)
		return false;

	Float t = Dot(E2, S_2) * invDivisor;
	if (t < mint || t > *maxt)
		return false;
	*maxt = t;
	++triangleHits;

	if (hit != NULL) {

		Normal N = Normal(Cross(E2, E1));
		N.Normalize();

		/* I need to do a fancy u,v stuff to make sure a smooth 
		   transition in u,v, coords when crossing boundaries
		   between polygons */
		if(NeedInversion)
		  hit->RecordHit(ray(t), N,1- u,1- v, mesh);
		else
		  hit->RecordHit(ray(t), N, u, v, mesh);
		int interpOffsets[3] =
			{ vertexIndex0, vertexIndex1, vertexIndex2 };
		Float interpWeights[3] = { 1. - u - v, u, v };
		InterpolatedPrimData *interpolatedData;
		interpolatedData =
			mesh->surfaceFunction->Interpolate(3,
											   interpOffsets,
											   interpWeights);
		hit->SetInterpolatedData(interpolatedData);

	}
	return true;
}



