
#include "quadrics.h"

Sphere::Sphere(Float rad, Float z0, Float z1, Float tm,
			   PrimitiveAttributes * a, SurfaceFunction * sf)
:	Primitive(a, sf)
{
	radius = rad;
	zmin = Clamp(min(z0, z1), -radius, radius);
	zmax = Clamp(max(z0, z1), -radius, radius);
	thetaMax = Radians(Clamp(tm, 0, 360));
}

BBox Sphere::BoundObjectSpace() const
{
	return BBox(Point(-radius, -radius, zmin),
				Point(radius, radius, zmax));
}

bool Sphere::IntersectClosest(const Ray & ray, Float mint,
							  Float * maxt, HitInfo * hit) const
{
	Point Phit;
	Float u, v;
	Float theta, phi;

	Float A = ray.D.x * ray.D.x + ray.D.y * ray.D.y + ray.D.z * ray.D.z;
	Float B = 2 * (ray.D.x * ray.O.x + ray.D.y * ray.O.y +
				   ray.D.z * ray.O.z);
	Float C = ray.O.x * ray.O.x + ray.O.y * ray.O.y +
		ray.O.z * ray.O.z - radius * radius;

	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.)
		return false;
	Float rootDiscrim = sqrt(discrim);

	Float inv2A = 1. / (2. * A);
	Float t0 = (-B - rootDiscrim) * inv2A;
	if (t0 > *maxt)
		return false;
	Float t1 = (-B + rootDiscrim) * inv2A;
	if (t1 < mint)
		return false;

	Float thit = t0;
	if (t0 < mint)
		thit = t1;
	if (thit > *maxt)
		return false;
	Phit = ray(thit);

	phi = acos(-Phit.z / radius);
	v = phi / M_PI;

	if (v == 0 || v == 1) {
		u = 0;
	} else {
		Float val = Clamp(Phit.x / (radius * sin(phi)), -1, 1);
		theta = acos(val) / (2 * M_PI);
		if (Phit.y > 0)
			u = theta;
		else
			u = 1 - theta;
		u *= (2 * M_PI) / thetaMax;
	}

	if (Phit.z < zmin || Phit.z > zmax || u > 1.) {
		if (thit == t1)
			return false;
		if (t1 > *maxt)
			return false;
		thit = t1;
		Phit = ray(thit);

		phi = acos(-Phit.z / radius);
		v = phi / M_PI;

		if (v == 0 || v == 1) {
			u = 0;
		} else {
			Float val = Clamp(Phit.x / (radius * sin(phi)), -1, 1);
			theta = acos(val) / (2 * M_PI);
			if (Phit.y > 0)
				u = theta;
			else
				u = 1 - theta;
			u *= (2 * M_PI) / thetaMax;
		}

		if (Phit.z < zmin || Phit.z > zmax || u > 1.)
			return false;
	}

	if (hit) {

		Float vmin = acos(-zmin / radius);
		Float vmax = acos(-zmax / radius);
		v = (v * M_PI - vmin) / (vmax - vmin);
		Normal surfNorm = Normal(Phit - Point(0, 0, 0));
		surfNorm.Normalize();
		hit->RecordHit(Phit, surfNorm, u, v, this);

		int interpOffsets[4] = { 0, 1, 2, 3 };
		Float interpWeights[4] =
			{ u * v, (1. - u) * v, u * (1. - v), (1. - u) * (1. - v) };
		InterpolatedPrimData *interpolatedData;
		interpolatedData = surfaceFunction->Interpolate(4,
														interpOffsets,
														interpWeights);

		hit->SetInterpolatedData(interpolatedData);

	}
	*maxt = thit;
	return true;
}

Cylinder::Cylinder(Float rad, Float z0, Float z1, Float tm,
				   PrimitiveAttributes * a, SurfaceFunction * sf)
:	Primitive(a, sf)
{
	radius = rad;
	zmin = min(z0, z1);
	zmax = max(z0, z1);
	thetaMax = Radians(Clamp(tm, 0, 360));
}

BBox Cylinder::BoundObjectSpace() const
{
	Point p1 = Point(-radius, -radius, zmin);
	Point p2 = Point(radius, radius, zmax);
	return BBox(p1, p2);
}

bool Cylinder::IntersectClosest(const Ray & ray, Float mint,
								Float * maxt, HitInfo * hit) const
{
	Point Phit;
	Float u, v;

	Float A = ray.D.x * ray.D.x + ray.D.y * ray.D.y;
	Float B = 2 * (ray.D.x * ray.O.x + ray.D.y * ray.O.y);
	Float C = ray.O.x * ray.O.x + ray.O.y * ray.O.y - radius * radius;

	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.)
		return false;
	Float rootDiscrim = sqrt(discrim);

	Float inv2A = 1. / (2. * A);
	Float t0 = (-B - rootDiscrim) * inv2A;
	if (t0 > *maxt)
		return false;
	Float t1 = (-B + rootDiscrim) * inv2A;
	if (t1 < mint)
		return false;

	Float thit = t0;
	if (t0 < mint)
		thit = t1;
	if (thit > *maxt)
		return false;
	Phit = ray(thit);

	u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
	v = (Phit.z - zmin) / (zmax - zmin);

	if (Phit.z < zmin || Phit.z > zmax || u > 1.) {
		if (thit == t1)
			return false;
		thit = t1;
		if (t1 > *maxt)
			return false;
		Phit = ray(thit);

		u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
		v = (Phit.z - zmin) / (zmax - zmin);

		if (Phit.z < zmin || Phit.z > zmax || u > 1.)
			return false;
	}

	if (hit) {

		Normal surfNorm = Normal(Phit - Point(0, 0, Phit.z));
		surfNorm.Normalize();
		hit->RecordHit(Phit, surfNorm, u, v, this);

	}
	*maxt = thit;
	return true;
}

Cone::Cone(Float ht, Float rad, Float tm,
		   PrimitiveAttributes * a, SurfaceFunction * sf)
:	Primitive(a, sf)
{
	radius = rad;
	height = ht;
	thetaMax = Radians(Clamp(tm, 0, 360));
}

BBox Cone::BoundObjectSpace() const
{
	Point p1 = Point(-radius, -radius, 0);
	Point p2 = Point(radius, radius, height);
	return BBox(p1, p2);
}

bool Cone::IntersectClosest(const Ray & ray, Float mint,
							Float * maxt, HitInfo * hit) const
{
	Point Phit;
	Float u, v;

	Float k = radius / height;
	k = k * k;
	Float A = ray.D.x * ray.D.x +
		ray.D.y * ray.D.y - k * ray.D.z * ray.D.z;
	Float B = 2 * (ray.D.x * ray.O.x +
				   ray.D.y * ray.O.y - k * ray.D.z * (ray.O.z - height));
	Float C = ray.O.x * ray.O.x +
		ray.O.y * ray.O.y - k * (ray.O.z - height) * (ray.O.z - height);


	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.)
		return false;
	Float rootDiscrim = sqrt(discrim);

	Float inv2A = 1. / (2. * A);
	Float t0 = (-B - rootDiscrim) * inv2A;
	if (t0 > *maxt)
		return false;
	Float t1 = (-B + rootDiscrim) * inv2A;
	if (t1 < mint)
		return false;

	Float thit = t0;
	if (t0 < mint)
		thit = t1;
	if (thit > *maxt)
		return false;
	Phit = ray(thit);

	u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
	v = Phit.z / height;

	if (Phit.z < 0 || Phit.z > height || u > 1.) {
		if (thit == t1)
			return false;
		thit = t1;
		if (t1 > *maxt)
			return false;
		Phit = ray(thit);

		u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
		v = Phit.z / height;

		if (Phit.z < 0 || Phit.z > height || u > 1.)
			return false;
	}

	if (hit) {

		Float nTheta = u * thetaMax;
		Normal surfNormal(cos(nTheta) * height, sin(nTheta) * height,
						  radius);
		surfNormal.Normalize();
		hit->RecordHit(Phit, surfNormal, u, v, this);

	}
	*maxt = thit;
	return true;
}

Paraboloid::Paraboloid(Float rad, Float z0, Float z1, Float tm,
					   PrimitiveAttributes * a, SurfaceFunction * sf)
:	Primitive(a, sf)
{
	radius = rad;
	zmin = z0;
	zmax = z1;
	thetaMax = Radians(Clamp(tm, 0, 360));
}

BBox Paraboloid::BoundObjectSpace() const
{
	Point p1 = Point(-radius, -radius, zmin);
	Point p2 = Point(radius, radius, zmax);
	return BBox(p1, p2);
}

bool Paraboloid::IntersectClosest(const Ray & ray, Float mint,
								  Float * maxt, HitInfo * hit) const
{
	Point Phit;
	Float u, v;

	Float k = zmax / (radius * radius);
	Float A = k * (ray.D.x * ray.D.x + ray.D.y * ray.D.y);
	Float B = 2 * k * (ray.D.x * ray.O.x + ray.D.y * ray.O.y) - ray.D.z;
	Float C = k * (ray.O.x * ray.O.x + ray.O.y * ray.O.y) - ray.O.z;

	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.)
		return false;
	Float rootDiscrim = sqrt(discrim);

	Float inv2A = 1. / (2. * A);
	Float t0 = (-B - rootDiscrim) * inv2A;
	if (t0 > *maxt)
		return false;
	Float t1 = (-B + rootDiscrim) * inv2A;
	if (t1 < mint)
		return false;

	Float thit = t0;
	if (t0 < mint)
		thit = t1;
	if (thit > *maxt)
		return false;
	Phit = ray(thit);

	u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / (2. * M_PI);
	v = (Phit.z - zmin) / (zmax - zmin);

	if (Phit.z < zmin || Phit.z > zmax || u * 2 * M_PI > thetaMax) {
		if (thit == t1)
			return false;
		thit = t1;
		if (t1 > *maxt)
			return false;
		Phit = ray(thit);

		u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / (2. * M_PI);
		v = (Phit.z - zmin) / (zmax - zmin);

		if (Phit.z < zmin || Phit.z > zmax || u * 2 * M_PI > thetaMax)
			return false;
	}

	if (hit) {

		u *= (2 * M_PI) / thetaMax;

		Float crossu = atan2(Phit.y, -Phit.x) + M_PI;
		Float crossv = Phit.z / zmax;
		Vector dPdv((radius * cos(crossu)) /
					(2 * sqrt(crossv) * sqrt(zmax)),
					(radius * sin(crossu)) / (2 * sqrt(crossv) *
											  sqrt(zmax)), 1);
		Vector dPdu((-radius * sqrt(crossv) * sin(crossu)) / sqrt(zmax),
					(radius * sqrt(crossv) * cos(crossu)) / sqrt(zmax), 0);
		Normal surfNorm = Normal(Cross(dPdu, dPdv));
		surfNorm.Normalize();
		hit->RecordHit(Phit, surfNorm, u, v, this);

	}
	*maxt = thit;
	return true;
}

Hyperboloid::Hyperboloid(const Point & point1, const Point & point2,
						 Float tm, PrimitiveAttributes * a,
						 SurfaceFunction * sf)
:	Primitive(a, sf)
{
	p1 = point1;
	p2 = point2;
	thetaMax = Radians(Clamp(tm, 0, 360));
	Float rad1 = sqrt(p1.x * p1.x + p1.y * p1.y);
	Float rad2 = sqrt(p2.x * p2.x + p2.y * p2.y);
	rmax = max(rad1, rad2);
	zmin = min(p1.z, p2.z);
	zmax = max(p1.z, p2.z);
}

BBox Hyperboloid::BoundObjectSpace() const
{
	Point p1 = Point(-rmax, -rmax, zmin);
	Point p2 = Point(rmax, rmax, zmax);
	return BBox(p1, p2);
}

bool Hyperboloid::IntersectClosest(const Ray & ray, Float mint,
								   Float * maxt, HitInfo * hit) const
{
	Point Phit;
	Float u, v;

	Float t, a2, c2;
	Float a, c;
	Point pr;

	if (p1.z == p2.z)
		return false;
	t = -p2.z / (p1.z - p2.z);
	pr = Point(t * p1.x + (1 - t) * p2.x,
			   t * p1.y + (1 - t) * p2.y, t * p1.z + (1 - t) * p2.z);
	a2 = pr.x * pr.x + pr.y * pr.y;
	if (p1.x * p1.x + p1.y * p1.y == a2)
		return false;
	c2 = (a2 * p1.z * p1.z) / (p1.x * p1.x + p1.y * p1.y - a2);

	Float A = ray.D.x * ray.D.x / a2 +
		ray.D.y * ray.D.y / a2 - ray.D.z * ray.D.z / c2;
	Float B = 2 * (ray.D.x * ray.O.x / a2 +
				   ray.D.y * ray.O.y / a2 - ray.D.z * ray.O.z / c2);
	Float C = ray.O.x * ray.O.x / a2 +
		ray.O.y * ray.O.y / a2 - ray.O.z * ray.O.z / c2 - 1;

	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.)
		return false;
	Float rootDiscrim = sqrt(discrim);

	Float inv2A = 1. / (2. * A);
	Float t0 = (-B - rootDiscrim) * inv2A;
	if (t0 > *maxt)
		return false;
	Float t1 = (-B + rootDiscrim) * inv2A;
	if (t1 < mint)
		return false;

	Float thit = t0;
	if (t0 < mint)
		thit = t1;
	if (thit > *maxt)
		return false;
	Phit = ray(thit);

	c = sqrt(c2);
	a = sqrt(a2);

	v = (Phit.z - p1.z) / (p2.z - p1.z);

	pr = Point((1 - v) * p1.x + v * p2.x,
			   (1 - v) * p1.y + v * p2.y, (1 - v) * p1.z + v * p2.z);

	u = atan2(Phit.y, Phit.x) - atan2(pr.y, pr.x);
	if (u < 0)
		u += 2 * M_PI;
	u /= 2 * M_PI;

	if (Phit.z < zmin || Phit.z > zmax || u * 2 * M_PI > thetaMax) {
		if (thit == t1)
			return false;
		thit = t1;
		if (t1 > *maxt)
			return false;
		Phit = ray(thit);

		c = sqrt(c2);
		a = sqrt(a2);

		v = (Phit.z - p1.z) / (p2.z - p1.z);

		pr = Point((1 - v) * p1.x + v * p2.x,
				   (1 - v) * p1.y + v * p2.y, (1 - v) * p1.z + v * p2.z);

		u = atan2(Phit.y, Phit.x) - atan2(pr.y, pr.x);
		if (u < 0)
			u += 2 * M_PI;
		u /= 2 * M_PI;

		if (Phit.z < zmin || Phit.z > zmax || u * 2 * M_PI > thetaMax)
			return false;
	}

	if (hit) {

		Float crossu = atan2(Phit.y, Phit.x) + M_PI;
		Float crossv = Phit.z / c;
		Vector dPdu(-a * cosh(crossv) * sin(crossu),
					a * cosh(crossv) * cos(crossu), 0);
		Vector dPdv(a * sinh(crossv) * cos(crossu),
					a * sinh(crossv) * sin(crossu), c * cosh(crossv));
		Normal surfNorm = Normal(Cross(dPdu, dPdv));
		surfNorm.Normalize();
		hit->RecordHit(Phit, surfNorm, u, v, this);

	}
	*maxt = thit;
	return true;
}

Disk::Disk(Float ht, Float r, Float tmax, PrimitiveAttributes * attr,
		   SurfaceFunction * sf)
:	Primitive(attr, sf)
{
	Height = ht;
	Radius = r;
	ThetaMax = Radians(Clamp(tmax, 0, 360));
}

BBox Disk::BoundObjectSpace() const
{
	return BBox(Point(-Radius, -Radius, Height),
				Point(Radius, Radius, Height));
}

bool Disk::IntersectClosest(const Ray & ray, Float mint, Float * maxt,
							HitInfo * hit) const
{

	Float hitt = (Height - ray.O.z) / ray.D.z;
	if (hitt < mint || hitt > *maxt)
		return false;

	Point Phit = ray(hitt);
	Float Dist2 = Phit.x * Phit.x + Phit.y * Phit.y;
	if (Dist2 > Radius * Radius)
		return false;

	Float u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / (2. * M_PI);
	Float v = 1. - (sqrt(Phit.x * Phit.x + Phit.y * Phit.y) / Radius);

	if (u * 2. * M_PI > ThetaMax)
		return false;

	if (hit) {

		u *= (2 * M_PI) / ThetaMax;
		hit->RecordHit(Phit, Normal(0, 0, 1), u, v, this);

		int interpOffsets[4] = { 0, 1, 2, 3 };
		Float interpWeights[4] =
			{ u * v, (1. - u) * v, u * (1. - v), (1. - u) * (1. - v) };
		InterpolatedPrimData *interpolatedData;
		interpolatedData = surfaceFunction->Interpolate(4,
														interpOffsets,
														interpWeights);

		hit->SetInterpolatedData(interpolatedData);

	}
	*maxt = hitt;
	return true;
}
