// LightField.cpp: implementation of the LightField class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "LFView.h"
#include "LightField.h"
#include <math.h>

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

LightField::LightField(CString filename)
{
	samples = NULL;

	FILE *inFile = fopen(filename, "rb");
	if(inFile)
	{
		fread(&header, sizeof(LightFieldHeader), 1, inFile);

		numSamples = header.XWidth * header.YWidth * header.PixelSamples[0] * header.PixelSamples[1];
		samples = new Spectrum[numSamples];
		fread(samples, sizeof(Spectrum), numSamples, inFile);
	}
}

LightField::~LightField()
{
	delete samples;
}

static const float M_PI = 3.14159265358979323846f;

void LightField::GenerateRandomPoint(float sample[2], float maxRadius, float &x, float &y)
{
    float radiusRatio = float(sqrt(sample[0]));
    float radius = radiusRatio * maxRadius;
    float angle = float(sample[1] * 2.0 * M_PI);

    x = float(radius * cos(angle));
    y = float(radius * sin(angle));
}

bool LightField::DoDOF(float xRaster, float yRaster, float xEye, float yEye,
					   float fStop, float focalDistance,
					   int interpolate, int dof, Spectrum &color)
{
	float x, y;

	if(dof)
	{
		float sample[2] = {(float)rand() / RAND_MAX, (float)rand() / RAND_MAX};

		GenerateRandomPoint(sample, header.MinFStop / fStop / 2.0f, x, y);

		float jitterRay = float((header.ZFilmPlane / focalDistance) / (header.FocalLength / (2. * header.MinFStop)));
	
		xRaster -= x * jitterRay / header.xSpan / 2.f;
		yRaster -= y * jitterRay / header.ySpan / 2.f;
	}
	else
	{
		x = y = 0.f;
	}
	
	return Interpolate(xRaster, yRaster, x + 0.5f + xEye, y + 0.5f + yEye, interpolate, color);
}

bool LightField::Interpolate(float xRaster, float yRaster, float xEye, float yEye,
							 int interpolate, Spectrum &color)
{
	yRaster = 1.0f - yRaster;
	yEye = 1.0f - yEye;

	xRaster *= header.XWidth - 1;
	yRaster *= header.YWidth - 1;
	xEye *= header.PixelSamples[0] - 1;
	yEye *= header.PixelSamples[1] - 1;

	int xRasterMin = int(floor(xRaster));
	int yRasterMin = int(floor(yRaster));
	int xEyeMin = int(floor(xEye));
	int yEyeMin = int(floor(yEye));

	if(interpolate)
	{
		float weightXR = 1.0f - (xRaster - xRasterMin);
		float weightYR = 1.0f - (yRaster - yRasterMin);
		float weightXE = 1.0f - (xEye - xEyeMin);
		float weightYE = 1.0f - (yEye - yEyeMin);
		
		Spectrum samples[16];
		
		if(!Lookup(xRasterMin, yRasterMin, xEyeMin, yEyeMin, samples[0]))
			return false;
		if(!Lookup(xRasterMin, yRasterMin+1, xEyeMin, yEyeMin, samples[1]))
			return false;
		if(!Lookup(xRasterMin+1, yRasterMin, xEyeMin, yEyeMin, samples[2]))
			return false;
		if(!Lookup(xRasterMin+1, yRasterMin+1, xEyeMin, yEyeMin, samples[3]))
			return false;
		if(!Lookup(xRasterMin, yRasterMin, xEyeMin, yEyeMin+1, samples[4]))
			return false;
		if(!Lookup(xRasterMin, yRasterMin+1, xEyeMin, yEyeMin+1, samples[5]))
			return false;
		if(!Lookup(xRasterMin+1, yRasterMin, xEyeMin, yEyeMin+1, samples[6]))
			return false;
		if(!Lookup(xRasterMin+1, yRasterMin+1, xEyeMin, yEyeMin+1, samples[7]))
			return false;
		if(!Lookup(xRasterMin, yRasterMin, xEyeMin+1, yEyeMin, samples[8]))
			return false;
		if(!Lookup(xRasterMin, yRasterMin+1, xEyeMin+1, yEyeMin, samples[9]))
			return false;
		if(!Lookup(xRasterMin+1, yRasterMin, xEyeMin+1, yEyeMin, samples[10]))
			return false;
		if(!Lookup(xRasterMin+1, yRasterMin+1, xEyeMin+1, yEyeMin, samples[11]))
			return false;
		if(!Lookup(xRasterMin, yRasterMin, xEyeMin+1, yEyeMin+1, samples[12]))
			return false;
		if(!Lookup(xRasterMin, yRasterMin+1, xEyeMin+1, yEyeMin+1, samples[13]))
			return false;
		if(!Lookup(xRasterMin+1, yRasterMin, xEyeMin+1, yEyeMin+1, samples[14]))
			return false;
		if(!Lookup(xRasterMin+1, yRasterMin+1, xEyeMin+1, yEyeMin+1, samples[15]))
			return false;
		
		Spectrum interp[4];
		
		Bilerp(&samples[0], weightXR, weightYR, interp[0]);
		Bilerp(&samples[4], weightXR, weightYR, interp[1]);
		Bilerp(&samples[8], weightXR, weightYR, interp[2]);
		Bilerp(&samples[12], weightXR, weightYR, interp[3]);
		
		Bilerp(&interp[0], weightXE, weightYE, color);
		
		return true;
	}
	else
	{
		return(Lookup(xRasterMin, yRasterMin, xEyeMin, yEyeMin, color));
	}
}

void LightField::Bilerp(Spectrum *samples, float wx, float wy, Spectrum &color)
{
	color.color[0] = (unsigned char)(samples[0].color[0] * wx * wy + samples[1].color[0] * wx * (1.0f - wy) +
		samples[2].color[0] * (1.0f - wx) * wy + samples[3].color[0] * (1.0f - wx) * (1.0f - wy));
	color.color[1] = (unsigned char)(samples[0].color[1] * wx * wy + samples[1].color[1] * wx * (1.0f - wy) +
		samples[2].color[1] * (1.0f - wx) * wy + samples[3].color[1] * (1.0f - wx) * (1.0f - wy));
	color.color[2] = (unsigned char)(samples[0].color[2] * wx * wy + samples[1].color[2] * wx * (1.0f - wy) +
		samples[2].color[2] * (1.0f - wx) * wy + samples[3].color[2] * (1.0f - wx) * (1.0f - wy));
}

bool LightField::Lookup(int xRaster, int yRaster, int xEye, int yEye, Spectrum &color)
{
	static Spectrum noColor = {0, 0, 0};

	if(xRaster < 0 || yRaster < 0 || xEye < 0 || yEye < 0 ||
		xRaster >= header.XWidth || yRaster >= header.YWidth ||
		xEye >= header.PixelSamples[0] || yEye >= header.PixelSamples[1])
		return false;

	int offset = ((yEye * header.PixelSamples[0] + xEye) * header.YWidth + yRaster) * header.XWidth + xRaster;
	color = samples[offset];
	return true;
}