
#include "lrt.h"
#include "scene.h"
#include "camera.h"
#include "mbdofcamera.h"
#include "primitives.h"
#include "image.h"
#include "transport.h"
#include "accel.h"
#include "sampling.h"

#ifdef USE_MPI
#include <mpi.h>
#include <zlib.h>
#endif

Scene *scene = NULL;

void Scene::Render()
{

	Integrator *integrator = NULL;
	if (IlluminationIntegrator == LRT_COLOR)
		integrator = new ColorIntegrator;
	else if (IlluminationIntegrator == LRT_RAYCAST)
		integrator = new RayCastingIntegrator;
	else if (IlluminationIntegrator == LRT_WHITTED)
		integrator = new WhittedIntegrator;
	else if (IlluminationIntegrator == LRT_MONTECARLO)
		integrator = new MCIntegrator;
	else if (IlluminationIntegrator == LRT_RMAN)
		integrator = new RendermanIntegrator;
	else {
		Severe("Unknown integrator \'%s\'", IlluminationIntegrator);
		return;
	}

	camera->FinalizeOptions();
	sampler->FinalizeValues();
	image->FinalizeValues();

	cerr << "Rendering: ";
	Float sample[5];
	while (sampler->GetNextImageSample(sample)) {

		Ray ray;
		if (!camera->GenerateRay(sample, ray))
			continue;

		static int eyeRaysTraced = 0;
		if (eyeRaysTraced == 0)
			StatsRegisterCounter(STATS_BASIC, "Camera", "Eye Rays Traced",
								 &eyeRaysTraced);
		++eyeRaysTraced;
		if (eyeRaysTraced % 10000 == 0)
			cerr << '+';

		HitInfo hitInfo;
		Float alpha, hitDist = INFINITY;
		Spectrum L =
			integrator->Integrate(ray, &hitInfo, &hitDist, &alpha);

		Float screenz = camera->WorldToScreen(ray(hitDist)).z;
		if (screenz > 1.) {
			L = 0.;
			alpha = 0.;
		}
		Point Praster(sample[0], sample[1], screenz);
		if(image->DisplayMode != LRT_LIGHTFIELD_IMAGE)
			image->AddSample(Praster, L, alpha);
		else
		{
			Point eyeJitter(sample[2], sample[3], 0.);
			((LightFieldImage*)image)->AddLightFieldSample(Praster, eyeJitter, L, alpha);
		}
	}
	image->Write();
	cerr << endl;
}

#ifdef USE_MPI

typedef struct
{
	Point Praster;
	Spectrum L;
	Float alpha;
} SampleData;

MPIScene::MPIScene() : Scene()
{
}

MPIScene::~MPIScene()
{
}

void MPIScene::Render()
{
	Integrator *integrator = NULL;
	if (IlluminationIntegrator == LRT_COLOR)
		integrator = new ColorIntegrator;
	else if (IlluminationIntegrator == LRT_RAYCAST)
		integrator = new RayCastingIntegrator;
	else if (IlluminationIntegrator == LRT_WHITTED)
		integrator = new WhittedIntegrator;
	else if (IlluminationIntegrator == LRT_MONTECARLO)
		integrator = new MCIntegrator;
	else if (IlluminationIntegrator == LRT_RMAN)
		integrator = new RendermanIntegrator;
	else {
		Severe("Unknown integrator \'%s\'", IlluminationIntegrator);
		return;
	}

	camera->FinalizeOptions();
	sampler->FinalizeValues();
	image->FinalizeValues();

	cerr << "Rendering: ";
	
	int thread_id, num_threads;
	
	MPI_Comm_rank(MPI_COMM_WORLD, &thread_id);
	MPI_Comm_size(MPI_COMM_WORLD, &num_threads);

	if(num_threads < 2)
	  Error("Must have more than one thread.\n");
	
	if(thread_id == 0)
	{
		int numTotalSamples = sampler->GetNumSamples();
		int maxSamples = 0;
		int i;
		
		for(i = 1; i < num_threads; i++)
		{
			int startSample = numTotalSamples * (i - 1) / (num_threads - 1);
			int stopSample = numTotalSamples * i / (num_threads - 1);
			int numSamplesToDo = stopSample - startSample;

			if(numSamplesToDo > maxSamples)
				maxSamples = numSamplesToDo;
		}

		SampleData *inData = new SampleData[maxSamples];
		int bufferSize = (int)(maxSamples * sizeof(SampleData) * 1.001 + 12);
		unsigned char *inBuffer = new unsigned char[bufferSize];
		MPI_Status status;
		unsigned long actualBytes, actualSampleBytes, actualSamples;
		
		for(i = 1; i < num_threads; i++)
		{
			MPI_Recv(inBuffer, bufferSize, MPI_BYTE,
				i, MPI_ANY_TAG, MPI_COMM_WORLD, &status);
			MPI_Get_count(&status, MPI_BYTE, (int*)&actualBytes);
			actualSampleBytes = maxSamples * sizeof(SampleData);
			uncompress((Bytef*)inData, &actualSampleBytes, inBuffer, actualBytes);
			actualSamples = status.MPI_TAG;
			
			for(unsigned int j = 0; j < actualSamples; j++)
				image->AddSample(inData[j].Praster, inData[j].L, inData[j].alpha);
		}
		
		image->Write();
	}
	else
	{
		int numTotalSamples = sampler->GetNumSamples();
		int startSample = numTotalSamples * (thread_id - 1) / (num_threads - 1);
		int stopSample = numTotalSamples * thread_id / (num_threads - 1);
		int numSamplesToDo = stopSample - startSample;
		unsigned long bufferSize = (int)(numSamplesToDo * sizeof(SampleData) * 1.001 + 12);
		
		SampleData *outData = new SampleData[numSamplesToDo];
		unsigned char *outBuffer = new unsigned char[bufferSize];
		Float sample[5];

		sampler->SkipSamples(startSample);
		
		for(int i = 0; i < numSamplesToDo; i++)
		{
			sampler->GetNextImageSample(sample);

			Ray ray;
			if (!camera->GenerateRay(sample, ray))
				continue;

			static int eyeRaysTraced = 0;
			if (eyeRaysTraced == 0)
				StatsRegisterCounter(STATS_BASIC, "Camera", "Eye Rays Traced",
								 &eyeRaysTraced);
			++eyeRaysTraced;
			if (eyeRaysTraced % 10000 == 0)
				cerr << '+';

			HitInfo hitInfo;
			Float alpha, hitDist = INFINITY;
			Spectrum L =
				integrator->Integrate(ray, &hitInfo, &hitDist, &alpha);
				
			Float screenz = camera->WorldToScreen(ray(hitDist)).z;
			if (screenz > 1.) {
				L = 0.;
				alpha = 0.;
			}
			Point Praster(sample[0], sample[1], screenz);
			
			outData[i].Praster = Praster;
			outData[i].L = L;
			outData[i].alpha = alpha;
		}

		if(compress(outBuffer, &bufferSize, (Bytef*)outData, numSamplesToDo * sizeof(SampleData)))
		  Error("FUCK!\n");
		
		MPI_Send(outBuffer, bufferSize, MPI_BYTE,
			0, numSamplesToDo, MPI_COMM_WORLD);
	}
	
	cerr << endl;
}
#endif

bool Scene::Intersect(const Ray & ray, Float mint, Float * maxt,
					  HitInfo * hit) const
{
	if (!accelerator)
		return false;
	return accelerator->IntersectClosest(ray, mint, maxt, hit);
}

Scene::Scene()
{

	IlluminationIntegrator = LRT_WHITTED;

	accelerator = NULL;
	camera = new PinholeCamera;
	image = new Image;
	sampler = new JitterSampler;
}

void Scene::AddPrimitives(const vector < Primitive * >&prim)
{
	primitives = prim;
	if (primitives.size() > 0)
		accelerator = new GridAccelerator(primitives);
}

Scene::~Scene()
{
	delete accelerator;
	delete camera;
	delete sampler;
	delete image;
//  for (u_int i = 0; i < primitives.size(); ++i)
//      delete primitives[i];
}

bool Scene::Unoccluded(const Point & p1, const Point & p2) const
{
	if (!accelerator)
		return true;

	static int shadowRayChecks = 0, shadowRayOccluded = 0;
	if (shadowRayChecks == 0)
		StatsRegisterRatio(STATS_DETAILED, "Integration",
						   "Finite Shadow Ray Checks",
						   &shadowRayOccluded, &shadowRayChecks);
	++shadowRayChecks;

	Float tmin = 1e-6;
	Float tmax = 1. - tmin;
	if (accelerator->IntersectClosest(Ray(p1, p2 - p1), tmin, &tmax, NULL)) {
		++shadowRayOccluded;
		return false;
	}

	return true;
}

bool Scene::Unoccluded(const Ray & r) const
{
	if (!accelerator)
		return true;

	static int shadowRayChecks = 0, shadowRayOccluded = 0;
	if (shadowRayChecks == 0)
		StatsRegisterRatio(STATS_DETAILED, "Integration",
						   "Infinite Shadow Ray Checks",
						   &shadowRayOccluded, &shadowRayChecks);
	++shadowRayChecks;

	Float tmin = 1e-6;
	Float tmax = INFINITY;
	Ray ray = r;
	ray.D.Normalize();
	if (accelerator->IntersectClosest(ray, tmin, &tmax, NULL)) {
		++shadowRayOccluded;
		return false;
	}

	return true;
}
