
#include "lrt.h"
#include "scene.h"
#include "sampling.h"
#include "image.h"

Float BoxFilter(Float x, Float y, Float xwidth, Float ywidth)
{
	if (x >= -xwidth * 0.5 && x <= xwidth * 0.5 &&
		y >= -ywidth * 0.5 && y <= ywidth * 0.5)
		return 1.;
	else
		return 0.;
}

Float TriangleFilter(Float x, Float y, Float xwidth, Float ywidth)
{
	if (x >= -xwidth * 0.5 && x <= xwidth * 0.5 &&
		y >= -ywidth * 0.5 && y <= ywidth * 0.5)
		return (xwidth * 0.5 - fabs(x)) * (ywidth * 0.5 - fabs(y));
	else
		return 0.;
}

Float GaussianFilter(Float x, Float y, Float xwidth, Float ywidth)
{
	Float d2 = (x * x + y * y);
	Float d = sqrt(d2);
	Float w2 = 0.5 * (xwidth * xwidth + ywidth * ywidth);
	Float w = sqrt(w2);
	if (d > w)
		return 0.0;
	else
		return exp(-d2) - exp(-w2);
}

Float MitchellFilter(Float x, Float y, Float xwidth, Float ywidth)
{
	x /= xwidth;
	y /= ywidth;
	Float r = sqrt(x * x + y * y);
	if (r > 2.)
		return 0.;
#define B .33333333333333333
#define C .33333333333333333
#define ONE_SIXTH .166666666666666666
	else if (r > 1.) {
		return ((-B - 6. * C) * r * r * r + (6. * B + 30. * C) * r * r +
				(-12. * B - 48. * C) * r + (8. * B + 24. * C)) * ONE_SIXTH;
	} else {
		return ((12. - 9. * B - 6. * C) * r * r * r +
				(-18. + 12. * B + 6. * C) * r * r + (6. -
													 2. * B)) * ONE_SIXTH;
	}
#undef B
#undef C
#undef ONE_SIXTH
}

Sampler::Sampler()
{

	PixelSamples[0] = PixelSamples[1] = 2.;

	JitterSamples = true;

	Filter = RiGaussianFilter;

	FilterXWidth = FilterYWidth = 2.0;

}

JitterSampler::JitterSampler()
{

	XPos = -1;
	YPos = 0;

}

void JitterSampler::FinalizeValues()
{

	XMax = (int) ceil(scene->image->XResolution * PixelSamples[0]);
	YMax = (int) ceil(scene->image->YResolution * PixelSamples[1]);
	DeltaX = 1. / PixelSamples[0];
	DeltaY = 1. / PixelSamples[1];

	Sampler::FinalizeValues();
}

int JitterSampler::GetNumSamples()
{
	return XMax * YMax;
}

void JitterSampler::SkipSamples(int numToSkip)
{
	for(int i = 0; i < numToSkip; i++)
		if(!AdvanceSample())
			return;
}

bool JitterSampler::AdvanceSample()
{
	if (++XPos >= XMax) {
		XPos = 0;
		if (++YPos >= YMax)
			return false;
	}
	
	return true;
}

bool JitterSampler::GetNextImageSample(Float sample[5])
{
	if(!AdvanceSample())
		return false;

	if (JitterSamples) {
		sample[0] = (XPos + RandomFloat()) * DeltaX;
		sample[1] = (YPos + RandomFloat()) * DeltaY;
		sample[2] = RandomFloat();
		sample[3] = RandomFloat();
		sample[4] = RandomFloat();
	} else {
		sample[0] = (XPos + 0.5) * DeltaX;
		sample[1] = (YPos + 0.5) * DeltaY;
		sample[2] = 0.5;
		sample[3] = 0.5;
		sample[4] = 0.5;
	}
	return true;

}

LightFieldSampler::LightFieldSampler() : JitterSampler()
{
	LightFieldSamples[0] = LightFieldSamples[1] = 2;
}

int LightFieldSampler::GetNumSamples()
{
	return XMax * YMax * EyeXMax * EyeYMax;
}

void LightFieldSampler::FinalizeValues()
{
	EyeXMax = (int) ceil(LightFieldSamples[0]);
	EyeYMax = (int) ceil(LightFieldSamples[1]);
	DeltaEyeX = 1. / LightFieldSamples[0];
	DeltaEyeY = 1. / LightFieldSamples[1];

	JitterSampler::FinalizeValues();
}

bool LightFieldSampler::AdvanceSample()
{
	if (++XPos >= XMax) {
		XPos = 0;
		if (++YPos >= YMax) {
			YPos = 0;
			if(++EyeXPos >= EyeXMax) {
				EyeXPos = 0;
				if(++EyeYPos >= EyeYMax)
					return false;
			}
		}
	}
	return true;
}

bool LightFieldSampler::GetNextImageSample(Float sample[5])
{
	if(!AdvanceSample())
		return false;
	
	if (JitterSamples) {
		sample[0] = (XPos + RandomFloat()) * DeltaX;
		sample[1] = (YPos + RandomFloat()) * DeltaY;
		sample[2] = (EyeXPos + RandomFloat()) * DeltaEyeX;
		sample[3] = (EyeYPos + RandomFloat()) * DeltaEyeY;
		sample[4] = RandomFloat();
	} else {
		sample[0] = (XPos + 0.5) * DeltaX;
		sample[1] = (YPos + 0.5) * DeltaY;
		sample[2] = EyeXPos * DeltaEyeX;
		sample[3] = EyeYPos * DeltaEyeY;
		sample[4] = 0.5;
	}
	
	return true;
	
}

GeneralSampler::GeneralSampler()
{
	x0 = 0.;
	y0 = 0.;
	DeltaX = (Float) SQRT_SAMPLE_TABLE_SIZE / PixelSamples[0];
	DeltaY = (Float) SQRT_SAMPLE_TABLE_SIZE / PixelSamples[1];
	tablePos = -1;
}

#include "sampledata.cc"

int GeneralSampler::GetNumSamples()
{
	return -1;
}

void GeneralSampler::SkipSamples(int numToSkip)
{
	return;
}

bool GeneralSampler::GetNextImageSample(Float sample[5])
{

	if (++tablePos >= SAMPLE_TABLE_SIZE) {
		tablePos = 0;
		x0 += DeltaX;
		if (x0 >= scene->image->XResolution) {
			x0 = 0.;
			y0 += DeltaY;
			if (y0 >= scene->image->YResolution)
				return false;
		}
	}

	sample[0] = x0 + DeltaX * sampleTable[tablePos][0];
	sample[1] = y0 + DeltaY * sampleTable[tablePos][1];
	sample[2] = sampleTable[tablePos][2];
	sample[3] = sampleTable[tablePos][3];
	sample[4] = sampleTable[tablePos][4];
	return true;

}
