#include "lrt.h"
#include "scene.h"
#include "camera.h"
#include "mbdofcamera.h"
#include "primitives.h"
#include "image.h"
#include "transport.h"
#include "accel.h"
#include "sampling.h"
#include "quadrics.h"

Scene *scene = NULL;

void Scene::Render()
{

	Integrator *integrator = NULL;
	if (IlluminationIntegrator == LRT_COLOR)
		integrator = new ColorIntegrator;
	else if (IlluminationIntegrator == LRT_RAYCAST)
		integrator = new RayCastingIntegrator;
	else if (IlluminationIntegrator == LRT_WHITTED)
		integrator = new WhittedIntegrator;
	else if (IlluminationIntegrator == LRT_MONTECARLO)
		integrator = new MCIntegrator;
	else if (IlluminationIntegrator == LRT_RMAN)
		integrator = new RendermanIntegrator;
	else {
		Severe("Unknown integrator \'%s\'", IlluminationIntegrator);
		return;
	}

	integrator->Initialize();
	camera->FinalizeOptions();
	sampler->FinalizeValues();
	image->FinalizeValues();

	// Build bounding spheres around caustic objects for efficient sampling

	vector < BBox > primBBoxes;
	vector < float > primSpheres;
	vector < Vector > primSphereOffsets;
	vector < float > primIsSphere;
	vector < float > weights;

	bool foundCausticSurf = false;
	if (IlluminationIntegrator == LRT_MONTECARLO) {

	  BBox allCausticPrimitives = primitives[0]->BoundWorldSpace();
	  int foundIndex = -1;
	  for (int i = 0; i < (int) primitives.size(); i++) {
	    if (primitives[i]->attributes->Surface->generatesCaustics) {
	      foundCausticSurf = true;
	      allCausticPrimitives = primitives[i]->BoundWorldSpace();

	      if (primitives[i]->isSphere) {
		primIsSphere.push_back(primitives[i]->pubrad);
	      } else {
		primIsSphere.push_back(0.);
	      }
	      primBBoxes.push_back(allCausticPrimitives);
	      foundIndex = i;
	      break;
	    }
	  }

	  if (foundCausticSurf) {
	    for (int i = 0; i < (int) primitives.size(); i++) {
	      if (i != foundIndex && primitives[i]->attributes->Surface->generatesCaustics) {
		primBBoxes.push_back(primitives[i]->BoundWorldSpace());
		if (primitives[i]->isSphere) {
		  primIsSphere.push_back(primitives[i]->pubrad);
		} else {
		  primIsSphere.push_back(0.);
		}
	      }
	    }

	    while (primBBoxes.size()) {

	      int last = primBBoxes.size() - 1;

	      if (primIsSphere[last] != 0) {
		primSpheres.push_back(primIsSphere[last]);
	      } else {
		float xDistance = primBBoxes[last].pMax.x - primBBoxes[last].pMin.x;
		float yDistance = primBBoxes[last].pMax.y - primBBoxes[last].pMin.y;
		float zDistance = primBBoxes[last].pMax.z - primBBoxes[last].pMin.z;

		float radius = sqrt(xDistance * xDistance + yDistance * yDistance + zDistance * zDistance) * 0.5;
		primSpheres.push_back(radius);
	      }

	      float xCenter = (primBBoxes[last].pMax.x + primBBoxes[last].pMin.x) * 0.5;
	      float yCenter = (primBBoxes[last].pMax.y + primBBoxes[last].pMin.y) * 0.5;
	      float zCenter = (primBBoxes[last].pMax.z + primBBoxes[last].pMin.z) * 0.5;

	      primBBoxes.pop_back();

	      Vector sphereCenterOffset(xCenter, yCenter, zCenter);

	      primSphereOffsets.push_back(sphereCenterOffset);
	    }

	    float totalw = 0.;
	    float w;
	    for (int i = 0; i < (int) primSpheres.size(); i++) {
	      w = primSpheres[i];
#if 0	      
	      Point P;
	      P.x = primSphereOffsets[i].x;
	      P.y = primSphereOffsets[i].y;
	      P.z = primSphereOffsets[i].z;
	      w /= sqrt(P.x * P.x + P.y * P.y + P.z * P.z);
#endif
	      totalw += w;
	      weights.push_back(w);
	    }
	    for (int i = 0; i < (int) primSpheres.size(); i++) {
	      weights[i] /= totalw;

	    }

	    CmapN = new PhotonMap(this, CausticMap, 100000, &primSpheres,
				  &primSphereOffsets, &weights);
	    CmapN->GeneratePhotonMap();
	  }

	  //GmapN = new PhotonMap(this, GlobalMap, 200000, NULL, NULL, NULL);
	  //GmapN->GeneratePhotonMap();

	  ImapN = new PhotonMap(this, IndirectMap, 300000, NULL, NULL, NULL);
	  ImapN->GeneratePhotonMap();

	  //integrator->SetupPhotonMap(GmapN, 0, 100);
	  if (foundCausticSurf) integrator->SetupPhotonMap(CmapN, 1, 70);
	  integrator->SetupPhotonMap(ImapN, 2, 100);
	}

	cerr << "Rendering: ";
	Float sample[5];
	while (sampler->GetNextImageSample(sample)) {

		Ray ray;
		if (!camera->GenerateRay(sample, ray))
			continue;

		static int eyeRaysTraced = 0;
		if (eyeRaysTraced == 0)
			StatsRegisterCounter(STATS_BASIC, "Camera", "Eye Rays Traced",
								 &eyeRaysTraced);
		++eyeRaysTraced;
		if (eyeRaysTraced % 10000 == 0)
		  cerr << (int) eyeRaysTraced / 10000;
		else if (eyeRaysTraced % 1000 == 0)
		  cerr << '+';

		HitInfo hitInfo;
		Float alpha, hitDist = INFINITY;
		Spectrum L =
			integrator->Integrate(ray, &hitInfo, &hitDist, &alpha);

		Float screenz = camera->WorldToScreen(ray(hitDist)).z;
		if (screenz > 1.) {
			L = 0.;
			alpha = 0.;
		}
		Point Praster(sample[0], sample[1], screenz);
		image->AddSample(Praster, L, alpha);

	}
	image->Write();
	cerr << endl;

	if (IlluminationIntegrator == LRT_MONTECARLO) {
	  //delete GmapN;
	  if (foundCausticSurf) delete CmapN;
	  delete ImapN;
	}
}

bool Scene::Intersect(const Ray & ray, Float mint, Float * maxt,
					  HitInfo * hit) const
{
	if (!accelerator)
		return false;
	return accelerator->IntersectClosest(ray, mint, maxt, hit);
}

Scene::Scene()
{

	IlluminationIntegrator = LRT_WHITTED;

	accelerator = NULL;
	camera = new PinholeCamera;
	image = new Image;
	sampler = new JitterSampler;
}

void Scene::AddPrimitives(const vector < Primitive * >&prim)
{
	primitives = prim;
	if (primitives.size() > 0)
		accelerator = new GridAccelerator(primitives);
}

Scene::~Scene()
{
	delete accelerator;
	delete camera;
	delete sampler;
	delete image;
//  for (u_int i = 0; i < primitives.size(); ++i)
//      delete primitives[i];
}

bool Scene::Unoccluded(const Point & p1, const Point & p2) const
{
	if (!accelerator)
		return true;

	static int shadowRayChecks = 0, shadowRayOccluded = 0;
	if (shadowRayChecks == 0)
		StatsRegisterRatio(STATS_DETAILED, "Integration",
						   "Finite Shadow Ray Checks",
						   &shadowRayOccluded, &shadowRayChecks);
	++shadowRayChecks;

	Float tmin = 1e-6;
	Float tmax = 1. - tmin;
	if (accelerator->IntersectClosest(Ray(p1, p2 - p1), tmin, &tmax, NULL)) {
		++shadowRayOccluded;
		return false;
	}

	return true;
}

bool Scene::Unoccluded(const Ray & r) const
{
	if (!accelerator)
		return true;

	static int shadowRayChecks = 0, shadowRayOccluded = 0;
	if (shadowRayChecks == 0)
		StatsRegisterRatio(STATS_DETAILED, "Integration",
						   "Infinite Shadow Ray Checks",
						   &shadowRayOccluded, &shadowRayChecks);
	++shadowRayChecks;

	Float tmin = 1e-6;
	Float tmax = INFINITY;
	Ray ray = r;
	ray.D.Normalize();
	if (accelerator->IntersectClosest(ray, tmin, &tmax, NULL)) {
		++shadowRayOccluded;
		return false;
	}

	return true;
}
