/*
 * simple homegrown 3 element vector class for position
 * and velocity calculations, plus some util math funcs.
 * If we need more matix math than this we should probably 
 * find a library
 */
#ifndef VEC3F_H_
#define VEC3F_H_

#include <SDL_opengl.h>
#include <SDL.h>
#include <iostream>

#define PI 3.14159265

/* vector of 3 floats, x, y, z */
class Vec3f {
  public:
    /* default ctor */
	Vec3f(): x(0), y(0), z(0) {}
    
    /* construct with x, y, z */
    Vec3f(GLfloat xv, GLfloat yv, GLfloat zv): x(xv), y(yv), z(zv) {}
    
    /* construct with vec */
    Vec3f(GLfloat* vec): x(vec[0]), y(vec[1]), z(vec[2]) {}
    
    /* use c++ default version for these */
    //~Vec3f();
    //Vec3f& operator=(const Vec3f& rhs);
    //Vec3f(const Vec3f& rhs);
    
    /* get x */
    GLfloat getX() const { return x; }
    
    /* set x */
    void setX(GLfloat v) { x = v; }
    
    /* get y */
    GLfloat getY() const { return y; }
    
    /* set y */
    void setY(GLfloat v) { y = v; }

    /* get z */
    GLfloat getZ() const { return z; }
    
    /* set z */
    void setZ(GLfloat v) { z = v; }
    
    /* set all 3 elements */
    void set(GLfloat xv, GLfloat yv, GLfloat zv) { x = xv; y = yv; z = zv; }
    
    /* add another vector to this */
    void add(const Vec3f& rhs) {
        x += rhs.x;
        y += rhs.y;
        z += rhs.z;
    }
    
    /* subtract another vector from this */
    void sub(const Vec3f& rhs) {
        x -= rhs.x;
        y -= rhs.y;
        z -= rhs.z;
    }
    
    /* multiply by scaler */
    void mult(GLfloat s) {
        x *= s;
        y *= s;
        z *= s;
    }
    
    /* divide by scaler */
    void div(GLfloat s) {
        x /= s;
        y /= s;
        z /= s;
    }
    
    /* test whether this vector is non-zero (faster than magnitude) */
    bool nonZero() { return ((x != 0) || (y != 0) || (z != 0)); }
    
    /* get the magnitude(length) of this vector, |v| */
    GLfloat magnitude() const;
    
    /* rotate this vector about Y-axis */
    void rotateY(GLfloat angle);
    
    /* rotate this vector about Z-axis */
    void rotateZ(GLfloat angle);
    
    /* get the angle to the Y axis of this vector, wrt -z axis */
    GLfloat getAngleYaxis();
    
    /* get the angle to this vector */
    GLfloat getAngle(const Vec3f& rhs);
    
    /* trim this vector's magnitude to below mag, keeping same direction */
    void trim(GLfloat mag);
    
    /* normalize this vector's magnitude to 1 */
    void normalize();
    
    /* get the dot product of this vec with another */
    GLfloat dotProduct(const Vec3f& rhs);
    
    /* get the cross product of this vec with another */
    Vec3f crossProduct(const Vec3f& rhs);
    
  private:
    /* xyz data */
    GLfloat x;
    GLfloat y;
    GLfloat z;
};


/* print a Vec3f */
std::ostream& operator<<(std::ostream& s, const Vec3f& v);

    
/* utility helper method for converting degs to rads */
inline GLfloat degToRad(GLfloat degs) {
    /* define pi for private use */
    return degs * (PI / 180.0f);
}

/* utility helper method for converting radians to degs */
inline GLfloat radToDeg(GLfloat rads) {
    /* define pi for private use */
    return rads * (180.0f / PI);
}

#endif /*VEC3F_H_*/
