/*
 * Vec3f impl
 */
 
#include "Vec3f.h"

#include <cmath>
#include <iostream>

using namespace std;

/* get the magnitude(length) of this vector, |v| */
GLfloat Vec3f::magnitude() const {
    return (GLfloat)sqrt((x * x) + (y * y) + (z * z));
}

/* rotate this vector about Y-axis */
void Vec3f::rotateY(GLfloat angle) {
    GLfloat rads = degToRad(angle);
    GLfloat xp = (GLfloat)((x * cos(rads)) + (z * sin(rads)));
    GLfloat zp = (GLfloat)((-x * sin(rads)) + (z * cos(rads)));
    x = xp;
    z = zp;
}

/* rotate this vector about Z-axis */
void Vec3f::rotateZ(GLfloat angle) {
    GLfloat rads = degToRad(angle);
    GLfloat xp = (GLfloat)((x * cos(rads)) - (y * sin(rads)));
    GLfloat yp = (GLfloat)((x * sin(rads)) + (y * cos(rads)));
    x = xp;
    y = yp;
}

/* get the dot product of this vec with another */
GLfloat Vec3f::dotProduct(const Vec3f& rhs) {
    return (x * rhs.x) + (y * rhs.y) + (z * rhs.z);    
}

/* get the cross product of this vec with another */
Vec3f Vec3f::crossProduct(const Vec3f& rhs) {
    return Vec3f((y * rhs.z) - (z * rhs.y),
                 (z * rhs.x) - (x * rhs.z),
                 (x * rhs.y) - (y * rhs.x));   
}

/* get the angle to the Y axis of this vector, wrt -z axis */
GLfloat Vec3f::getAngleYaxis() {
    GLfloat angle = 0;
    Vec3f xzOnly(*this);
    xzOnly.setY(0);
    
    GLfloat mag = xzOnly.magnitude();
    if (mag > 0) {
        double arg = z / mag;
        if (arg <= 1 && arg >= -1) {   
            double rads = acos( arg );
            angle = radToDeg((GLfloat)rads);
            if (x < 0) {
                angle = -angle; 
            }
        }
    }
    return angle;
}
    
/* get the angle to this vector */
GLfloat Vec3f::getAngle(const Vec3f& rhs) {
    GLfloat mag = this->magnitude() * rhs.magnitude();
    if (mag > 0) {
        double arg = this->dotProduct(rhs) / mag;
        if (arg <= 1 && arg >= -1) {   
            double rads = acos( arg );
            return radToDeg((GLfloat)rads);
        }
    }
    return 0;
}


/* trim this vector's magnitude to below mag, keeping same direction */
void Vec3f::trim(GLfloat mag) {
    GLfloat currMag = magnitude();
    if (currMag > mag) {
        div(currMag);
        mult(mag);
    }
}

/* normalize this vector's magnitude to 1 */
void Vec3f::normalize() {
    GLfloat currMag = magnitude();
    div(currMag);
}

/* print a Vec3f */
std::ostream& operator<<(std::ostream& s, const Vec3f& v) {
    s << "(" << v.getX() << ", " << v.getY() << ", " << v.getZ() << ")";
    return s;
}
  

