/*
 * Truck implementation
 */

#include "Truck.h"
#include <iostream>

#include <fstream>
using std::ofstream;

extern bool debugMode;
extern bool musicMode;

using namespace std;


/* construct with reference to game world */
Truck::Truck(GameWorld& gw): GameObject(gw) {
    // set generic GameObject params
    topSpeed = 10;
    setWidth( 3 );
    setHeight( 3 );
    setLength( 5 );

    object = new Mesh("truck");
    scale = 0.00005;
 
    truckR = width/2.0;
    boundSphereDist = (length / 2.0) + truckR + 0.1;
    
    // set boundaries for map for truck
    GLfloat bound = 55;
    mapBoundary.xMax = bound;
    mapBoundary.xMin = -bound;
    mapBoundary.zMax = bound;
    mapBoundary.zMin = -bound;
    
    
    // setup gravel with a force in x
    grav_right = new Gravel(2.0);
    grav_left = new Gravel(-2.0);

    //set gravels initial position where the car is
    grav_right->init_pos = loc;
    grav_left->init_pos = loc;

    //initiate the particle systems
    grav_right->In();
    grav_left->In();

    if (musicMode) {
        // load sound effect
        sound = Mix_LoadWAV("sounds/trucks24.wav");
        if (sound == NULL) {
            cout << "Failed to load sound effect " << Mix_GetError() << endl;
        }
    }
}


/* set the acceleration of this object to point at angleY */
void Truck::accelerate(GLfloat mag) {
    Vec3f newAccel(0, 0, -1); //start with -z dir
    newAccel.rotateY(getAngleY());
    newAccel.mult(mag);
    
    // show some gravel if reversing or starting from stop
    GLfloat oldAccel = accel.magnitude();
    if (oldAccel == 0) {
        if (grav_left->allParticlesDead() && 
            grav_right->allParticlesDead()) 
        {
           grav_left->InitParticles();
           grav_right->InitParticles();
        }
    }
    
    
    accel = newAccel;
    bool forward = mag > 0;
    if (drivingForward != forward) {
        vel.set(0,0,0); // stop first
    }
    setDrivingForward( forward );
    
}


void Truck::CheckSlide()
{
//    //vector that point in the dir of the car
//    Vec3f car_vec(0.0, 0.0, 1.0);
//    //rotate this vector the angleY of the car to get it in the car dir
//    //angle from car = angleY
//    car_vec.rotateY(angleY);
//
//    //compare the two vectors, if 90 degrees betweeen then spray gravel
//    float angle_between = car_vec.getAngle(vel);
//    
//    //check how they are pointing
//    float dotproduct = car_vec.dotProduct(vel);
//
//
//    if(angle_between > 60 && angle_between < 120 && dotproduct <= 0)
//    //if(angle_between > 5 && angle_between < 100 && dotproduct <= 0)
//    {
//       is_particle_left = true;
//       OutFile << "Particles LEFT is true" << endl << endl;
//	   //grav_left->InitParticles();
//    }
//    if(angle_between > 60 && angle_between < 120 && dotproduct >= 0)
//    {
//       is_particle_right = true;
//       OutFile << "Particles RIGHT is true" << endl << endl;
//	   //grav_right->InitParticles();
//    }

}

/* updates truck location  */
void Truck::update(Uint32 timeElapsed) {
    updateLocAndVel(timeElapsed);
}


/* special collison detection for truck */
bool Truck::willCollide(Vec3f truckLoc, Vec3f objLoc, GameObjPtr obj) {
    // approx with a few spheres
    GLfloat objR = obj->getBoundSphereDist();

    vector<Vec3f> bvLocs = getBVLocs(truckLoc);
    for (unsigned int i = 0; i < bvLocs.size(); i++) {
        if (spheresCollide(bvLocs[i], truckR, objLoc, objR)) {  
            return true;
        }
    }
    
    return false;
}

/* get list of BV locations for cd */
vector<Vec3f> Truck::getBVLocs(Vec3f truckLoc) {
    vector<Vec3f> locs;
    Vec3f axis(0, 0, -1 * (length / 2.0));
    axis.rotateY(angleY);
    
    // one for loc
    locs.push_back( Vec3f(truckLoc) );
    
    // spread some out up to length
    for (int i = 2; i <= 10; i += 2) {
        Vec3f tmpA(axis);
        tmpA.mult((GLfloat)i / 10.0);
        
        Vec3f p(truckLoc);
        p.add(tmpA);
        locs.push_back(p);
        
        Vec3f n(truckLoc);
        n.sub(tmpA);
        locs.push_back(n);
    }
    
    return locs;
}

/* set the angle of truck, only if no collisions */
void Truck::setAngleY(GLfloat v) {
	
    GLfloat prev = angleY;
    // set and test for collision
    angleY = v;
    if (detectCollision(loc)) {
        angleY = prev;
    }
}

    
/* draw truck */
void Truck::draw()
{
     // isolate transforms
    glPushMatrix();
    
    // translate to current location
    glTranslatef( loc.getX(), loc.getY(), loc.getZ());

    // draw debugging
    if (debugMode) {
        drawDebugInfo();
        vector<Vec3f> bvLocs = getBVLocs(loc);
        for (vector<Vec3f>::iterator i = bvLocs.begin(); i != bvLocs.end(); i++) {
            Vec3f bvLoc = *i;
            bvLoc.sub(loc);
            drawSphere(bvLoc, truckR);
        }
    }
    
    // rotate to terrain normal
    rotateToNormal();
    
    // rotate to heading
    glRotatef(angleY, 0, 1, 0);
    
    // rotate forward
    glRotatef(180, 0, 1, 0);
    
    // raise up a bit 
    glTranslatef( 0, 1, 0);
    
    // scale this object

    //push, just scale the car, not the gravel
    glPushMatrix();
    glScalef(scale, scale, scale);

    //render truck with textures
    object->render();
    glPopMatrix();

    glPushMatrix();
    GLfloat gy = -0.3;
    GLfloat gz = -1.5;
    glTranslatef( -1.4, gy, gz );
    grav_left->Render();
    glPopMatrix();
    
    glPushMatrix();
    glTranslatef( 1.4, gy, gz );
    grav_right->Render();
    glPopMatrix();

    //pop matrix
    glPopMatrix();
}
