/*
 * Class that represents the initial title screen
 */

#include <SDL_opengl.h>
#include <SDL.h>
#include <SDL_mixer.h>
#include <SDL_ttf.h>
#include <iostream>

#include "TitleScreen.h"
#include "SurfaceUtil.h"
#include "Display.h"

using namespace std;

extern bool musicMode;

/* default constructor */
TitleScreen::TitleScreen(): musicChannel(0) {
    // load audio
    if (musicMode) {
        music = Mix_LoadWAV("sounds/credits.aif");
        if (music == NULL) {
            cout << "Failed to load music audio " << Mix_GetError() << endl;
        }
        startSound = Mix_LoadWAV("sounds/cow.wav");
        if (startSound == NULL) {
            cout << "Failed to load start sound audio " << Mix_GetError() << endl;
        }
    }
    
    // load title screen surfaces
    char* file = "imgs/title-screen-loading.bmp";
    loadingScreen = SDL_LoadBMP(file);
    if (loadingScreen == NULL) {
        cout << "Load TitleScreen Loading Surface Failed! " << file << endl;
    }

    file = "imgs/title-screen-start.bmp";
    startScreen = SDL_LoadBMP(file);
    if (startScreen == NULL) {
        cout << "Load TitleScreen Start Surface Failed! " << file << endl;
    }

    file = "imgs/title-screen-howto.bmp";
    howToScreen = SDL_LoadBMP(file);
    if (howToScreen == NULL) {
        cout << "Load TitleScreen HowTo Surface Failed! " << file << endl;
    }

    file = "imgs/title-screen-quit.bmp";
    quitScreen = SDL_LoadBMP(file);
    if (quitScreen == NULL) {
        cout << "Load TitleScreen Quit Surface Failed! " << file << endl;
    }

    file = "imgs/howto-screen.bmp";
    directionsScreen = SDL_LoadBMP(file);
    if (directionsScreen == NULL) {
        cout << "Load TitleScreen Directions Surface Failed! " << file << endl;
    }

    startGame = false;
    currentMode = NO_MODE;
}

/* destructor */
TitleScreen::~TitleScreen() {
    if (musicMode) {
        // make sure all channels are halted before freeing
        Mix_HaltChannel(-1);
        
        // free audio chunks
        Mix_FreeChunk(music);
        Mix_FreeChunk(startSound);
    }
    
    // free surface
    SDL_FreeSurface(loadingScreen);
    SDL_FreeSurface(startScreen);
    SDL_FreeSurface(howToScreen);
    SDL_FreeSurface(quitScreen);
    SDL_FreeSurface(directionsScreen);
}

/* clear screen to black */
void TitleScreen::clearScreen() {
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glClearDepth(1.0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

/* draw initial loading screen and play audio */
void TitleScreen::drawLoadingScreen() {
    if (musicMode && music != NULL) {
        // start audio
        musicChannel = Mix_PlayChannel(-1, music, -1);
        if (musicChannel == -1) {
            cout << "Failed to play audio " << Mix_GetError() << endl;
        }
    }
    
    // clear screen
    clearScreen();
    
    // draw loading screen surface to OpenGL
    if (loadingScreen != NULL) {
        glWindowPos2f((GLfloat)(Display::WIDTH - loadingScreen->w) / 2.0, 
                      (GLfloat)(Display::HEIGHT - loadingScreen->h) / 2.0);
        glDrawPixels(loadingScreen->w, loadingScreen->h, GL_BGR, GL_UNSIGNED_BYTE, loadingScreen->pixels);
    }
    
    // swap buffers
    SDL_GL_SwapBuffers();
}

/* draw the current screen */
void TitleScreen::drawCurrentScreen() {
    switch (currentMode) {
        case START_MODE:
            drawStartScreen();
            break;
        case HOW_TO_MODE:
            drawHowToScreen();
            break;
        case QUIT_MODE:
            drawQuitScreen();
            break;
        case BACK_MODE:
            drawDirectionsScreen();
            break;
        default:
            drawStartScreen();
            break;
    }
}

/* draw start selection screen */
void TitleScreen::drawStartScreen() {
    // set start mode
    currentMode = START_MODE;
    
    // clear screen
    clearScreen();
    
    // draw start screen surface to OpenGL
    if (startScreen != NULL) {
        glWindowPos2f((GLfloat)(Display::WIDTH - startScreen->w) / 2.0, 
                      (GLfloat)(Display::HEIGHT - startScreen->h) / 2.0);
        glDrawPixels(startScreen->w, startScreen->h, GL_BGR, GL_UNSIGNED_BYTE, startScreen->pixels);
    }
    
    // swap buffers
    SDL_GL_SwapBuffers();
}

/* draw how to play selection screen */
void TitleScreen::drawHowToScreen() {
    // clear screen
    clearScreen();
    
    // draw how to play selection screen surface to OpenGL
    if (howToScreen != NULL) {
        glWindowPos2f((GLfloat)(Display::WIDTH - howToScreen->w) / 2.0, 
                      (GLfloat)(Display::HEIGHT - howToScreen->h) / 2.0);
        glDrawPixels(howToScreen->w, howToScreen->h, GL_BGR, GL_UNSIGNED_BYTE, howToScreen->pixels);
    }
    
    // swap buffers
    SDL_GL_SwapBuffers();
}

/* draw quit selection screen */
void TitleScreen::drawQuitScreen() {
    // clear screen
    clearScreen();
    
    // draw quit selection screen surface to OpenGL
    if (quitScreen != NULL) {
        glWindowPos2f((GLfloat)(Display::WIDTH - quitScreen->w) / 2.0, 
                      (GLfloat)(Display::HEIGHT - quitScreen->h) / 2.0);
        glDrawPixels(quitScreen->w, quitScreen->h, GL_BGR, GL_UNSIGNED_BYTE, quitScreen->pixels);
    }
    
    // swap buffers
    SDL_GL_SwapBuffers();
}

/* draw directions/how to screen */
void TitleScreen::drawDirectionsScreen() {
    // clear screen
    clearScreen();
    
    // draw directions/how to screen surface to OpenGL
    if (directionsScreen != NULL) {
        glWindowPos2f((GLfloat)(Display::WIDTH - directionsScreen->w) / 2.0, 
                      (GLfloat)(Display::HEIGHT - directionsScreen->h) / 2.0);
        glDrawPixels(directionsScreen->w, directionsScreen->h, GL_BGR, GL_UNSIGNED_BYTE, directionsScreen->pixels);
    }
    
    // swap buffers
    SDL_GL_SwapBuffers();
}

/* handle title screen events */
void TitleScreen::handleEvents() {
    SDL_Event event;
    
    // poll for keydown events

    while (!startGame) {
        if (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_QUIT:
                    // stop all audio
                    Mix_HaltChannel(-1);
                    Mix_CloseAudio();
                    // quit sdl
                    SDL_Quit();
                    std::exit(0);
                    break;
                    
                case SDL_KEYDOWN:
                    handleKeyboardEvent(event);
                    break;
                    
                default:
                    break;
            }
        } else {
            SDL_Delay(1);
        }
    }
}

/* handle a keyboard event */
void TitleScreen::handleKeyboardEvent(SDL_Event& event) {
    SDLKey key = event.key.keysym.sym;
    
    // process key
    if (key == SDLK_UP) {
        if (currentMode > START_MODE) {
            --currentMode;
        }
    } else if (key == SDLK_DOWN) {
        if (currentMode < QUIT_MODE) {
            ++currentMode;
        }
    } else {
        // handle the current mode event for any other key
        switch (currentMode) {
            case START_MODE:
                // turn off title screen music and start the game
                if (musicMode) {
                    if (startSound != NULL) {
                        // play start sound audio
                        startSoundChannel = Mix_PlayChannel(-1, startSound, 0);
                        if (startSoundChannel == -1) {
                            cout << "Failed to play start sound audio " << Mix_GetError() << endl;
                        } else {
                            // wait for start sound to finish playing
                            while (Mix_Playing(startSoundChannel));
                        }
                    }
                    
                    // stop title screen music
                    Mix_HaltChannel(getMusicChannel());
                }
                
                // start game
                startGame = true;
                break;
                
            case HOW_TO_MODE:
                // display directions screen
                currentMode = BACK_MODE;
                break;
                
            case QUIT_MODE:
                // stop all audio
                Mix_HaltChannel(-1);
                Mix_CloseAudio();
                // quit sdl
                SDL_Quit();
                std::exit(0);
                break;
                
            case BACK_MODE:
                // go back to how to selection screen
                currentMode = HOW_TO_MODE;
                break;
                
            default:
                // do nothing
                break;
        }
    }
    
    // draw the current screen
    drawCurrentScreen();
}