/*
 * Terrain is a non-updateable GameObject which draws the ground.
 */

#ifndef TERRAIN_H_
#define TERRAIN_H_

#include <SDL_opengl.h>
#include <SDL.h>

#include "GameObject.h"
#include "GameWorld.h"
#include "Patch.h"
#include "HeightMap.h"
#include "QuadTree.h"

#define DEFAULT_TERRAIN_SIZE 150
#define NUM_PATCHES_PER_SIDE 8 // should be a power of 2 so quad tree can subdivide correctly

#define TERRAIN_DARK_BASE_TEXTURE 0
#define TERRAIN_BASE_TEXTURE 1
#define TERRAIN_DETAIL_TEXTURE 2

class QuadTree;
class QuadTreeNode;
typedef GameObject* GameObjPtr;
typedef vector<GameObjPtr> GameObjVec;

/* GameObject that renders as the terrain */
class Terrain: public GameObject {
  public:
    /* construct with reference to world */
    Terrain(GameWorld& gw);

    /* destructor */
    ~Terrain();

    /* load textures */
    void load();

    /* draw the terrain */
    void draw();

    /* get height of terrain at specified location (x and z coordinates) */
    GLfloat getHeight(GLfloat xPos, GLfloat zPos);

    /* gets the dark base texture id for the terrain */
    GLuint getDarkBaseTexture() { return textures[TERRAIN_DARK_BASE_TEXTURE]; }

    /* gets the base texture id for the terrain */
    GLuint getBaseTexture() { return textures[TERRAIN_BASE_TEXTURE]; }

    /* gets the detail texture id for the terrain */
    GLuint getDetailTexture() { return textures[TERRAIN_DETAIL_TEXTURE]; }

  protected:
    /* draw the node */
    void draw(QuadTreeNode node);

    /* reset all patch meshes */
    void resetMesh();

    /* update the node's patch meshes */
    void updateMesh(QuadTreeNode node);

    /* initialize the terrain patches */
    void initPatches();

    // patch quad tree
    QuadTree* patchTree;

    // vector of patch GameObjPtrs
    GameObjVec patchVec;

    // grid of patches that make up this terrain
    Patch* patches[NUM_PATCHES_PER_SIDE][NUM_PATCHES_PER_SIDE];

    // height map for terrain surface heights
    HeightMap* heightMap;

    // textures for the terrain
    GLuint textures[3];
};

#endif /*TERRAIN_H_*/
