/*
 * Terrain implementation.
 */

#include <SDL_opengl.h>
#include <SDL.h>
#include <cmath>
#include <iostream>

#include "Terrain.h"
#include "Frustum.h"
#include "ImageHeightMap.h"

using namespace std;

extern GLint numMultitextures;

/* construct with reference to world */
Terrain::Terrain(GameWorld& gw): GameObject(gw) {
    // set size
    setWidth(DEFAULT_TERRAIN_SIZE);
    setLength(DEFAULT_TERRAIN_SIZE);
    
    // load height map
    heightMap = new ImageHeightMap();
    
    // initialize terrain patches
    initPatches();
}

/* destructor */
Terrain::~Terrain() {
    // free all terrain patches
    for (int i = 0; i < NUM_PATCHES_PER_SIDE; i++) {
        for (int j = 0; j < NUM_PATCHES_PER_SIDE; j++) {
            free(patches[i][j]);
        }
    }
    
    // free quad tree
    free(patchTree);
    
    // free height map
    free(heightMap);
}

/* load textures */
void Terrain::load() {
    // generate textures
    glGenTextures(3, textures);

    // generate dark base texture
    SDL_Surface* darkSurface = SDL_LoadBMP("imgs/terrain-ground-dark.bmp");
    glBindTexture(GL_TEXTURE_2D, textures[TERRAIN_DARK_BASE_TEXTURE]);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, darkSurface->w, darkSurface->h,
              GL_BGR, GL_UNSIGNED_BYTE, darkSurface->pixels);
    SDL_FreeSurface(darkSurface);

    // generate multitextures if they are supported
    if (numMultitextures >= 2) {
        // generate base texture
        SDL_Surface* surface = SDL_LoadBMP("imgs/terrain-ground.bmp");
        glBindTexture(GL_TEXTURE_2D, textures[TERRAIN_BASE_TEXTURE]);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
        gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, surface->w, surface->h,
            GL_BGR, GL_UNSIGNED_BYTE, surface->pixels);
        SDL_FreeSurface(surface);

        // generate detail texture
        SDL_Surface* detailSurface = SDL_LoadBMP("imgs/terrain-detail.bmp");
        //SDL_Surface* detailSurface = SDL_LoadBMP("imgs/Detail.bmp");
        glBindTexture(GL_TEXTURE_2D, textures[TERRAIN_DETAIL_TEXTURE]);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
        gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, detailSurface->w, detailSurface->h,
            GL_BGR, GL_UNSIGNED_BYTE, detailSurface->pixels);
        SDL_FreeSurface(detailSurface);
    }
}

/* draw the terrain */
void Terrain::draw() {
    // isolate transformations specific to the terrain
    glPushMatrix();
    
    // translate to x, y, z
    glTranslatef(getX(), getY(), getZ());
    
    // update the terrain mesh
    resetMesh();
    updateMesh(patchTree->getRoot());
    
    // draw all terrain patches
    draw(patchTree->getRoot());
    
    // return stack to its previous state
    glPopMatrix();   
}

/* get height of terrain at specified location */
GLfloat Terrain::getHeight(GLfloat xPos, GLfloat zPos) {
    return heightMap->getHeight(xPos, zPos);
}

/* draw the node */
void Terrain::draw(QuadTreeNode node) {
    if (world.getViewFrustum()->sphereIntersects(node.getCenter(), node.getBoundSphereDist()) > OUTSIDE_FRUSTUM) {
        if (node.getGameObjs().size() > 0) {
            // if this is a leaf node, draw the patches
            GameObjVec objs = node.getGameObjs();
            for (GameObjVec::size_type i = 0; i < objs.size(); i++) {
                GameObjPtr obj = objs[i];
                obj->draw();
            }
        } else {
            // draw each child node
            QTNVec childNodes = node.getChildNodes();
            for (QTNVec::size_type i = 0; i < childNodes.size(); i++) {
                QuadTreeNode& childNode = childNodes[i];
                draw(childNode);
            }
        }
    }
}

/* reset all patches to pre-split state and link patches together */
void Terrain::resetMesh() {
    // reset all terrain patches
    Patch* patch;
    for (int i = 0; i < NUM_PATCHES_PER_SIDE; i++) {
        for (int j = 0; j < NUM_PATCHES_PER_SIDE; j++) {
            patch = patches[i][j];
            patch->resetMesh();
            patch->updateVariance();
            
            Patch::setNextNodeIndex(0);
            
            // link patches together
            if (j > 0) {
                patch->getRightNode()->leftNeighbor = patches[i][j - 1]->getLeftNode();
            } else {
                patch->getRightNode()->leftNeighbor = NULL;
            }
            if (j < NUM_PATCHES_PER_SIDE - 1) {
                patch->getLeftNode()->leftNeighbor = patches[i][j + 1]->getRightNode();
            } else {
                patch->getLeftNode()->leftNeighbor = NULL;
            }
            if (i > 0) {
                patch->getLeftNode()->rightNeighbor = patches[i - 1][j]->getRightNode();
            } else {
                patch->getLeftNode()->rightNeighbor = NULL;
            }
            if (i < NUM_PATCHES_PER_SIDE - 1) {
                patch->getRightNode()->rightNeighbor = patches[i + 1][j]->getLeftNode();
            } else {
                patch->getRightNode()->rightNeighbor = NULL;
            }
        }
    }
}

/* update the mesh of the terrain */
void Terrain::updateMesh(QuadTreeNode node) {
    if (world.getViewFrustum()->sphereIntersects(node.getCenter(), node.getBoundSphereDist()) > OUTSIDE_FRUSTUM) {
        if (node.getGameObjs().size() > 0) {
            // if this is a leaf node, draw the patches
            GameObjVec objs = node.getGameObjs();
            for (GameObjVec::size_type i = 0; i < objs.size(); i++) {
                GameObjPtr obj = objs[i];
                obj->updateMesh();
            }
        } else {
            // draw each child node
            QTNVec childNodes = node.getChildNodes();
            for (QTNVec::size_type i = 0; i < childNodes.size(); i++) {
                QuadTreeNode& childNode = childNodes[i];
                updateMesh(childNode);
            }
        }
    }
}

/* initialize all terrain patches */
void Terrain::initPatches() {
    Patch* patch;
    for (int i = 0; i < NUM_PATCHES_PER_SIDE; i++) {
        for (int j = 0; j < NUM_PATCHES_PER_SIDE; j++) {
            // create patch
            patch = new Patch(world);
            
            // set patch size
            patch->setWidth(width / NUM_PATCHES_PER_SIDE);
            patch->setLength(length / NUM_PATCHES_PER_SIDE);
            
            // set patch location
            patch->setLocation((getX() + xMin) + (patch->getXMax()) + i * patch->getWidth(),
                getY(), (getZ() + zMin) + (patch->getZMax()) + j * patch->getLength());
            
            // set height map reference
            patch->setHeightMap(heightMap);
            
            // reset patch variances
            patch->updateVariance();
            
            // add patch to array and vector
            patches[i][j] = patch;
            patchVec.push_back(patch);
        }
    }
    
    // max depth of quad tree is log base 2 of the number of patches per side
    unsigned int size = NUM_PATCHES_PER_SIDE;
    unsigned int maxDepth = 0;
    while (size >>= 1) maxDepth++;
    
    // build quad tree
    patchTree = new QuadTree();
    patchTree->build(getX() + xMin, getX() + xMax, getZ() + zMin, getZ() + zMax, patchVec, maxDepth, 0);
}
