/*
 * Skybox implementation.
 */

#include <SDL_opengl.h>
#include <SDL.h>

#include "Skybox.h"
#include "GameWorld.h"
#include "Display.h"

/* construct with reference to world */
Skybox::Skybox(GameWorld& gw): GameObject(gw) {
    width = DEFAULT_SKYBOX_SIZE;
    height = DEFAULT_SKYBOX_SIZE;
    length = DEFAULT_SKYBOX_SIZE;
}

/* destructor */
Skybox::~Skybox() {
}

/* load textures */
void Skybox::load() {
    // SDL surfaces
    SDL_Surface* frontSurface = SDL_LoadBMP("imgs/skybox-front.bmp");
    SDL_Surface* leftSurface = SDL_LoadBMP("imgs/skybox-left.bmp");
    SDL_Surface* rightSurface = SDL_LoadBMP("imgs/skybox-right.bmp");
    SDL_Surface* backSurface = SDL_LoadBMP("imgs/skybox-back.bmp");
    SDL_Surface* upSurface = SDL_LoadBMP("imgs/skybox-up.bmp");
    SDL_Surface* downSurface = SDL_LoadBMP("imgs/skybox-down.bmp");
    
    // front surface
    glGenTextures(6, textures);
    glBindTexture(GL_TEXTURE_2D, textures[FRONT_TEXTURE]);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, frontSurface->w, frontSurface->h,
              GL_BGR, GL_UNSIGNED_BYTE, frontSurface->pixels);
    
    // left surface
    glBindTexture(GL_TEXTURE_2D, textures[LEFT_TEXTURE]);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, leftSurface->w, leftSurface->h,
              GL_BGR, GL_UNSIGNED_BYTE, leftSurface->pixels);
    
    // right surface
    glBindTexture(GL_TEXTURE_2D, textures[RIGHT_TEXTURE]);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, rightSurface->w, rightSurface->h,
              GL_BGR, GL_UNSIGNED_BYTE, rightSurface->pixels);
    
    // back surface
    glBindTexture(GL_TEXTURE_2D, textures[BACK_TEXTURE]);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, backSurface->w, backSurface->h,
              GL_BGR, GL_UNSIGNED_BYTE, backSurface->pixels);
    
    // up surface
    glBindTexture(GL_TEXTURE_2D, textures[UP_TEXTURE]);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, upSurface->w, upSurface->h,
              GL_BGR, GL_UNSIGNED_BYTE, upSurface->pixels);

    // down surface
    glBindTexture(GL_TEXTURE_2D, textures[DOWN_TEXTURE]);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, downSurface->w, downSurface->h,
              GL_BGR, GL_UNSIGNED_BYTE, downSurface->pixels);
    
    // free surfaces
    SDL_FreeSurface(frontSurface);
    SDL_FreeSurface(leftSurface);
    SDL_FreeSurface(rightSurface);
    SDL_FreeSurface(backSurface);
    SDL_FreeSurface(upSurface);
    SDL_FreeSurface(downSurface);
}

/* draw the skybox, assumes this will be the first object drawn */
void Skybox::draw() {
    // isolate transformations specific to the skybox
    glPushMatrix();
    
    // disable lighting and depth buffer since skybox should be the farthest object
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_LIGHTING);
    glEnable(GL_TEXTURE_2D);
    glColor3f(1, 1, 1);
    
    // don't translate skybox with the view to give the illusion of a fixed, infinite horizon
    GLfloat matrix[16];
    glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
    matrix[12] = matrix[13] = matrix[14] = 0.0;
    glLoadMatrixf(matrix);
    
    // draw all skybox walls
    setXMin(getX() - width / 2.0);
    setXMax(getX() + width / 2.0);
    setYMin(getY() - height / 2.0);
    setYMax(getY() + height / 2.0);
    setZMin(getZ() - length / 2.0);
    setZMax(getZ() + length / 2.0);

    // front wall
    glBindTexture(GL_TEXTURE_2D, textures[FRONT_TEXTURE]);
    glBegin(GL_QUADS);
        glTexCoord2f(0.0, 0.0);
        glVertex3f(getXMax(), getYMax(), getZMin());
        glTexCoord2f(1.0, 0.0);
        glVertex3f(getXMin(), getYMax(), getZMin());
        glTexCoord2f(1.0, 1.0);
        glVertex3f(getXMin(), getYMin(), getZMin());
        glTexCoord2f(0.0, 1.0);
        glVertex3f(getXMax(), getYMin(), getZMin());
    glEnd();
    
    // left wall
    glBindTexture(GL_TEXTURE_2D, textures[LEFT_TEXTURE]);
    glBegin(GL_QUADS);
        glTexCoord2f(0.0, 0.0);
        glVertex3f(getXMin(), getYMax(), getZMin());
        glTexCoord2f(1.0, 0.0);
        glVertex3f(getXMin(), getYMax(), getZMax());
        glTexCoord2f(1.0, 1.0);
        glVertex3f(getXMin(), getYMin(), getZMax());
        glTexCoord2f(0.0, 1.0);
        glVertex3f(getXMin(), getYMin(), getZMin());
    glEnd();
    
    // right wall
    glBindTexture(GL_TEXTURE_2D, textures[RIGHT_TEXTURE]);
    glBegin(GL_QUADS);
        glTexCoord2f(0.0, 0.0);
        glVertex3f(getXMax(), getYMax(), getZMax());
        glTexCoord2f(1.0, 0.0);
        glVertex3f(getXMax(), getYMax(), getZMin());
        glTexCoord2f(1.0, 1.0);
        glVertex3f(getXMax(), getYMin(), getZMin());
        glTexCoord2f(0.0, 1.0);
        glVertex3f(getXMax(), getYMin(), getZMax());
    glEnd();
    
    // back wall
    glBindTexture(GL_TEXTURE_2D, textures[BACK_TEXTURE]);
    glBegin(GL_QUADS);
        glTexCoord2f(0.0, 0.0);
        glVertex3f(getXMin(), getYMax(), getZMax());
        glTexCoord2f(1.0, 0.0);
        glVertex3f(getXMax(), getYMax(), getZMax());
        glTexCoord2f(1.0, 1.0);
        glVertex3f(getXMax(), getYMin(), getZMax());
        glTexCoord2f(0.0, 1.0);
        glVertex3f(getXMin(), getYMin(), getZMax());
    glEnd();
    
    // up wall
    glBindTexture(GL_TEXTURE_2D, textures[UP_TEXTURE]);
    glBegin(GL_QUADS);
        glTexCoord2f(0.0, 0.0);
        glVertex3f(getXMax(), getYMax(), getZMax());
        glTexCoord2f(1.0, 0.0);
        glVertex3f(getXMin(), getYMax(), getZMax());
        glTexCoord2f(1.0, 1.0);
        glVertex3f(getXMin(), getYMax(), getZMin());
        glTexCoord2f(0.0, 1.0);
        glVertex3f(getXMax(), getYMax(), getZMin());
    glEnd();
    
    // down wall
    glBindTexture(GL_TEXTURE_2D, textures[DOWN_TEXTURE]);
    glBegin(GL_QUADS);
        glTexCoord2f(1.0, 0.0);
        glVertex3f(getXMin(), getYMin(), getZMin());
        glTexCoord2f(1.0, 1.0);
        glVertex3f(getXMin(), getYMin(), getZMax());
        glTexCoord2f(0.0, 1.0);
        glVertex3f(getXMax(), getYMin(), getZMax());
        glTexCoord2f(0.0, 0.0);
        glVertex3f(getXMax(), getYMin(), getZMin());
    glEnd();
    
    // re-enable lighting and depth buffer so other objects will be drawn over the skybox
    glDisable(GL_TEXTURE_2D);
    glEnable(GL_LIGHTING);
    glEnable(GL_DEPTH_TEST);
    
    // return stack to its previous state
    glPopMatrix();
}
