/*
 * Quad tree that divides and sub-divides 
 * space into 4 squares, placing GameObjects inside
 */

#ifndef QUADTREE_H_
#define QUADTREE_H_

#include <iostream>
#include <vector>
#include <cmath>

#include "GameWorld.h"
#include "GameObject.h"

using namespace std;

typedef GameObject* GameObjPtr;
typedef vector<GameObjPtr> GameObjVec;

/* typedefs for vec of QT nodes */
class QuadTreeNode;
typedef std::vector< QuadTreeNode > QTNVec;

/* struct for one node in tree */
class QuadTreeNode {
  public:
    /* do nothing constructor */
    QuadTreeNode():
        xMin(0), 
        xMax(0), 
        zMin(0), 
        zMax(0), 
        boundSphereDist(0), 
        maxDepth(0), 
        maxObjs(0), 
        depth(0) 
    {
        center.set(0, 0, 0);
    }
  
    /* constructor */
    QuadTreeNode(GLfloat xmin, GLfloat xmax, GLfloat zmin, GLfloat zmax):
        xMin(xmin), 
        xMax(xmax), 
        zMin(zmin), 
        zMax(zmax), 
        maxDepth(0), 
        maxObjs(0), 
        depth(0) 
    {
        GLfloat width = xmax - xmin;
        GLfloat length = zmax - zmin;
        center.set(xmin + (width / 2.0), 0, zmin + (length / 2.0));
        boundSphereDist = sqrt(width * width + length * length);
    }
    
    /* destructor */
    virtual ~QuadTreeNode() {}
  
    /* get child nodes */
    QTNVec getChildNodes() { return childNodes; }
    
    /* get GameObjs contained in this node (if leaf node) */
    GameObjVec getGameObjs() { return gameObjs; }
    
    /* assign GameObjs to this node */
    void assignGameObjs(const GameObjVec& objs);

    /* get minimum x value */
    GLfloat getXMin() { return xMin; }
    
    /* set minimum x value */
    void setXMin(GLfloat v) { xMin = v; }

    /* get maximum x value */
    GLfloat getXMax() { return xMax; }

    /* set maximum x value */
    void setXMax(GLfloat v) { xMax = v; }

    /* get minimum y value */
    GLfloat getZMin() { return zMin; }

    /* set minimum y value */
    void setZMin(GLfloat v) { zMin = v; }

    /* get maximum y value */
    GLfloat getZMax() { return zMax; }

    /* set maximum y value */
    void setZMax(GLfloat v) { zMax = v; }
    
    /* get center location */
    Vec3f getCenter() { return center; }
    
    /* set center location */
    void setCenter(Vec3f v) { center = v; }
    
    /* get bounding sphere distance */
    GLfloat getBoundSphereDist() { return boundSphereDist; }
    
    /* set bounding sphere distance */
    void setBoundSphereDist(GLfloat v) { boundSphereDist = v; }
    
    /* set the max depth limit */
    void setMaxDepth(int v) { maxDepth = v; }
    
    /* set the max objects limit */
    void setMaxObjs(int v) { maxObjs = v; }
    
    /* get the depth of this node */
    int getDepth() { return depth; }
    
    /* set the depth of this node */
    void setDepth(int v) { depth = v; }
    
    /* 
     * subdivide and place the gameObjs, recursively if necessary,
     * depending on max depth or max objs
     */
    void subdivide();
    
    /* whether the GameObj intersects this node, using AABB test */
    bool intersects(const GameObjPtr& obj);
    
  private:
    /* children nodes */
    QTNVec childNodes;
    /* game objects (if this is leaf node) */
    GameObjVec gameObjs;
    
    /* minimum x value */
    GLfloat xMin;
    /* maximum x value */
    GLfloat xMax;
    /* minimum y value */
    GLfloat zMin;
    /* maximum y value */
    GLfloat zMax;
    
    /* center location */
    Vec3f center;
    /* bounding sphere distance */
    GLfloat boundSphereDist;
    
    /* max tree depth limit */
    int maxDepth;
    /* max objects limit */
    int maxObjs;
    /* node depth */
    int depth;
    
    /* helper fn to divide into 4 sub-nodes */
    void createChildNodes();
    
    /* helper fn to distribute objs into nodes, using AABB intersects */
    void distribute(const GameObjVec& objs, QTNVec& nodes);
    
    
  /* print */  
  friend std::ostream& operator<<(std::ostream& s, const QuadTreeNode& n);
};

/* the entire tree */
class QuadTree {
  public:
    /* constructor */
    QuadTree() {}
    
    /* destructor */
    virtual ~QuadTree() {}
    
    /* get the root node */
    QuadTreeNode getRoot() { return root; }
    
    /* 
     * build a quad tree with objects, limited by max depth of tree 
     * and max objects per node, which ever is first, specifying 0
     * for either parameter means ignore (max depth or max objs)
     */
    void build(GLfloat xmin, GLfloat xmax, GLfloat zmin, GLfloat zmax, 
               const GameObjVec& objs, int maxDepth, int maxObjs);
    
    
  private:
    /* the root node */
    QuadTreeNode root;
    
};

/* printint QT stuff*/
std::ostream& operator<<(std::ostream& s, const QuadTreeNode& n);

#endif /*QUADTREE_H_*/
