/*
 * implementation for QuadTree and QuadTreeNode
 */

#include "QuadTree.h"

using namespace std;

/* 
 * build a quad tree with objects, limited by max depth of tree 
 * and max objects per node, which ever is first, specifying 0
 * for either parameter means ignore (max depth or max objs)
 */
void QuadTree::build(GLfloat xmin, GLfloat xmax, GLfloat zmin, GLfloat zmax, 
                     const GameObjVec& objs, int maxDepth, int maxObjs)
{
    root = QuadTreeNode(xmin, xmax, zmin, zmax);
    root.setMaxDepth(maxDepth);
    root.setMaxObjs(maxObjs);
    root.assignGameObjs(objs);
    root.subdivide();
}


/* assign GameObjs to this node */
void QuadTreeNode::assignGameObjs(const GameObjVec& objs) { 
    gameObjs.assign(objs.begin(), objs.end()); 
}

/* 
 * subdivide and place the GameObjs, recursively if necessary,
 * depending on max depth or max objs
 */
void QuadTreeNode::subdivide() {
    // test if stopping condition reached
    if (!(maxObjs > 0 && (int)gameObjs.size() <= maxObjs) &&
        !(maxDepth > 0 && depth >= maxDepth))
    {
        // otherwise, subdivide into 4 nodes, repeat process
        createChildNodes();
        distribute(gameObjs, childNodes);
        // clear own gameobjs
        gameObjs.clear();
        
        // subdivide each child
        for (QTNVec::size_type i = 0; i < childNodes.size(); i++) {
            QuadTreeNode& node = childNodes[i];
            node.setMaxDepth(maxDepth);
            node.setMaxObjs(maxObjs);
            node.setDepth(depth + 1);
            node.subdivide();
        }
    }
}

/* helper fn to divide into 4 sub-nodes */
void QuadTreeNode::createChildNodes() {
    GLfloat xMid = xMin + ((xMax - xMin) / 2.0);
    GLfloat zMid = zMin + ((zMax - zMin) / 2.0);
    childNodes.push_back( QuadTreeNode(xMin, xMid, zMin, zMid) );
    childNodes.push_back( QuadTreeNode(xMin, xMid, zMid, zMax) );
    childNodes.push_back( QuadTreeNode(xMid, xMax, zMid, zMax) );
    childNodes.push_back( QuadTreeNode(xMid, xMax, zMin, zMid) );
}

/* whether the GameObj intersects this node, using AABB test */
bool QuadTreeNode::intersects(const GameObjPtr& obj) {
    GLfloat objXmax = obj->getX() + (obj->getWidth() / 2.0);
    GLfloat objXmin = obj->getX() - (obj->getWidth() / 2.0);
    GLfloat objZmax = obj->getZ() + (obj->getLength() / 2.0);
    GLfloat objZmin = obj->getZ() - (obj->getLength() / 2.0);
    
    if (xMin >= objXmax || objXmin >= xMax) {
        return false;
    } else if (zMin >= objZmax || objZmin >= zMax) {
        return false;
    }
    return true;
}
    

/* helper fn to distribute objs into nodes, using AABB intersects */
void QuadTreeNode::distribute(const GameObjVec& objs, QTNVec& nodes) {
    for (GameObjVec::size_type i = 0; i < objs.size(); i++) {
        GameObjPtr go = objs[i];
        for (QTNVec::size_type j = 0; j < nodes.size(); j++) {
            QuadTreeNode& node = nodes[j];
            if (node.intersects(go)) {
                node.gameObjs.push_back(go);
            }
        }
    }   
}


/* printint QT stuff*/
ostream& operator<<(ostream& s, const QuadTreeNode& n) {
    string depthStr("  ");
    for (int i=0; i < n.depth; i++) {
        s << depthStr;
    }
    s << "[" << n.xMin << ", " << n.xMax << ", " << n.zMin << ", " << n.zMax;
    s << "]" << "(" << n.gameObjs.size() << "," << n.childNodes.size();
    s << "," << n.maxDepth << "," << n.maxObjs << "," << n.depth << ")";
    if (n.gameObjs.size() > 0) {
        s << " GameObjs { ";
        for (int i=0; i < (int)n.gameObjs.size(); i++) {
            GameObjPtr obj = n.gameObjs[i];
            GLfloat objXmax = obj->getX() + (obj->getWidth() / 2.0);
            GLfloat objXmin = obj->getX() - (obj->getWidth() / 2.0);
            GLfloat objZmax = obj->getZ() + (obj->getLength() / 2.0);
            GLfloat objZmin = obj->getZ() - (obj->getLength() / 2.0);
            s << "(";
            s << objXmin << ", ";
            s << objXmax << ", ";
            s << objZmin << ", ";
            s << objZmax << ") ";
        }
        s << "} ";
    }
    s << endl;

    for (int i=0; i < (int)n.childNodes.size(); i++) {
        s << n.childNodes[i];
    }
    return s;
}

