/*
 * Patch is a portion of the terrain.
 */

#ifndef PATCH_H_
#define PATCH_H_

#include <SDL_opengl.h>
#include <SDL.h>

#include "GameObject.h"
#include "HeightMap.h"

#define DEFAULT_PATCH_SIZE 10
#define NODE_POOL_SIZE 25000
#define VARIANCE_SEARCH_DEPTH 1
#define VARIANCE_LIMIT 60

#define MIN(X,Y) (X < Y ? X : Y)
#define MAX(X,Y) (X > Y ? X : Y)

/* node in mesh tree, which defines a piece of the mesh as a square split along its diagonal */
struct MeshTreeNode {
    MeshTreeNode* leftChild;
    MeshTreeNode* rightChild;
    MeshTreeNode* baseNeighbor;
    MeshTreeNode* leftNeighbor;
    MeshTreeNode* rightNeighbor;
};

/* GameObject that renders a patch of the terrain */
class Patch: public GameObject {
  public:
    /* construct with reference to world */
    Patch(GameWorld& gw);

    /* destructor */
    virtual ~Patch();

    /* gets the next free MeshTreeNode from the pool */
    static MeshTreeNode* getNextFreeNode();

    /* returns the index to the next free node */
    static int getNextNodeIndex() { return nextNode; }

    /* sets the index to the next free node to the given value */
    static void setNextNodeIndex(int index) { nextNode = index; }

    /* draw the patch */
    void draw();

    /* draw the given MeshTreeNode, recursively down the tree */
    void drawNode(MeshTreeNode* node, Vec3f left, Vec3f right, Vec3f apex);

    /* split the given MeshTreeNode into two, force splitting neighbors as needed */
    void splitNode(MeshTreeNode* node);

    /* update the variance of the patch, returning the max found in the variance search depth */
    void updateVariance();

    /* returns the node variance (difference between interpolated height and actual height) given the 
     * coordinates of a MeshTreeNode */
    float updateNodeVariance(Vec3f left, Vec3f right, Vec3f apex, int nodePos);

    /* update the mesh of the patch */
    void updateMesh();

    /* update the node given the coordinates of a MeshTreeNode, splitting as needed */
    void updateNode(MeshTreeNode* node, Vec3f left, Vec3f right, Vec3f apex, int nodePos);

    /* reset mesh nodes */
    void resetMesh();

    /* get pointer to left MeshTreeNode */
    MeshTreeNode* getLeftNode() { return &leftNode; }

    /* get pointer to right MeshTreeNode */
    MeshTreeNode* getRightNode() { return &rightNode; }

    /* get height of patch at specified location (x and z coordinates) */
    GLfloat getHeight(GLfloat xPos, GLfloat zPos) { return heightMap->getHeight(xPos, zPos); }

    /* returns whether this patch is visible in the view frustum */
    bool isVisible() { return visible; }

    /* sets whether this patch is visible in the view frustum */
    void setIsVisible(bool v) { visible = v; }

    /* set height map from containing terrain */
    void setHeightMap(HeightMap* hm) { heightMap = hm; }

  private:
    // index to the next free MeshTreeNode in pool
    static int nextNode;

    // pool of available MeshTreeNodes for splitting, used to reduce overhead of 
    // dynamically creating and destroying numerous MeshTreeNode objects
    static MeshTreeNode nodePool[NODE_POOL_SIZE];

    // left MeshTreeNode
    MeshTreeNode leftNode;

    // right MeshTreeNode
    MeshTreeNode rightNode;

    // the variance of this patch's nodes
    float* currentVariance;
    float leftNodeVariance[1 << VARIANCE_SEARCH_DEPTH];
    float rightNodeVariance[1 << VARIANCE_SEARCH_DEPTH];

    // indicates whether this patch is visible
    bool visible;

    // pointer to terrain's heightMap to determine terrain surface heights
    HeightMap* heightMap;

    /* get the texture coordinate for the x axis */
    GLfloat getTextureCoordX(Vec3f center);

    /* get the texture coordinate for the z axis */
    GLfloat getTextureCoordZ(Vec3f center);

    /* get the detail texture coordinate for the x axis */
    GLfloat getDetailTextureCoordX(Vec3f center);

    /* get the detail texture coordinate for the z axis */
    GLfloat getDetailTextureCoordZ(Vec3f center);
};

#endif /*PATCH_H_*/
