/*
 * Patch is a portion of the terrain.
 */

#include <SDL_opengl.h>
#include <SDL.h>
#include <cmath>
#include <iostream>

#include "Patch.h"
#include "GameWorld.h"
#include "Truck.h"

using namespace std;

extern bool debugMode;
extern bool multitextureMode;
extern GLint numMultitextures;
extern PFNGLACTIVETEXTUREARBPROC glActiveTextureARB;
extern PFNGLMULTITEXCOORD2FARBPROC glMultiTexCoord2fARB;

/* static member variables */
int Patch::nextNode;
MeshTreeNode Patch::nodePool[NODE_POOL_SIZE];

/* construct with reference to world */
Patch::Patch(GameWorld& gw): GameObject(gw) {
    setWidth(DEFAULT_PATCH_SIZE);
    setHeight(DEFAULT_PATCH_SIZE);
    setLength(DEFAULT_PATCH_SIZE);
    boundSphereDist = getBoundSphereDist();
}

/* destructor */
Patch::~Patch() {}

/* gets the next free MeshTreeNode from the pool */
MeshTreeNode* Patch::getNextFreeNode() {
    MeshTreeNode* node = NULL;

    if (nextNode < NODE_POOL_SIZE) {
        // get the next available node and increment the index
        node = &(nodePool[nextNode++]);

        // reset the node
        node->leftChild = NULL;
        node->rightChild = NULL;
    }

    return node;
}

/* draw the patch */
void Patch::draw() {
    // isolate transformations specific to the Patch
    glPushMatrix();

    // translate to x, y, z
    glTranslatef(getX(), getY(), getZ());

    Vec3f upperLeft(xMin, getY(), zMin);
    Vec3f upperRight(xMax, getY(), zMin);
    Vec3f lowerLeft(xMin, getY(), zMax);
    Vec3f lowerRight(xMax, getY(), zMax);
    drawNode(&leftNode, lowerRight, upperLeft, lowerLeft);
    drawNode(&rightNode, upperLeft, lowerRight, upperRight);

    // return stack to its previous state
    glPopMatrix();   
}

/* draw the given MeshTreeNode, recursively down the tree */
void Patch::drawNode(MeshTreeNode* node, Vec3f left, Vec3f right, Vec3f apex) {
    if (node->leftChild && node->rightChild) {
        GLfloat centerX = (left.getX() + right.getX()) / 2.0;
        GLfloat centerZ = (left.getZ() + right.getZ()) / 2.0;
        Vec3f center(centerX, getY(), centerZ);
        drawNode(node->leftChild, apex, left, center);
        drawNode(node->rightChild, right, apex, center);
    } else {
        GLfloat leftY = getHeight(getX() + left.getX(), getZ() + left.getZ());
        GLfloat rightY = getHeight(getX() + right.getX(), getZ() + right.getZ());
        GLfloat apexY = getHeight(getX() + apex.getX(), getZ() + apex.getZ());
        glColor3f(1, 1, 1);

        if (debugMode) {
            // draw as mesh
            glDisable(GL_LIGHTING);
            glBegin(GL_LINE_LOOP);
                glVertex3f(left.getX(), leftY, left.getZ());
                glVertex3f(right.getX(), rightY, right.getZ());
                glVertex3f(apex.getX(), apexY, apex.getZ());
            glEnd();
            glEnable(GL_LIGHTING);
        } else {
            // draw with texture
            glDisable(GL_LIGHTING);
            
            // check for multitexture support
            bool hasMultitexture = multitextureMode && glActiveTextureARB != NULL &&
                glMultiTexCoord2fARB != NULL && numMultitextures >= 2;
            
            if (!hasMultitexture) {
                // don't multitexture, use just the dark base texture
                glEnable(GL_TEXTURE_2D);
                glBindTexture(GL_TEXTURE_2D, world.getTerrain()->getDarkBaseTexture());
                glBegin(GL_TRIANGLES);
                    glTexCoord2f(getTextureCoordX(left), getTextureCoordZ(left));
                    glVertex3f(left.getX(), leftY, left.getZ());
                    glTexCoord2f(getTextureCoordX(right), getTextureCoordZ(right));
                    glVertex3f(right.getX(), rightY, right.getZ());
                    glTexCoord2f(getTextureCoordX(apex), getTextureCoordZ(apex));
                    glVertex3f(apex.getX(), apexY, apex.getZ());
                glEnd();
                glDisable(GL_TEXTURE_2D);
            } else {
                // get terrain base texture
                glActiveTextureARB(GL_TEXTURE0_ARB);
                glEnable(GL_TEXTURE_2D);
                glBindTexture(GL_TEXTURE_2D, world.getTerrain()->getBaseTexture());
                glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_EXT);
                glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_RGB_EXT, GL_REPLACE);
                
                // get terrain detail texture
                glActiveTextureARB(GL_TEXTURE1_ARB);
                glEnable(GL_TEXTURE_2D);
                glBindTexture(GL_TEXTURE_2D, world.getTerrain()->getDetailTexture());
                glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_EXT);
                glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_RGB_EXT, GL_MODULATE);
                
                // draw patch with multitextures                             
                glBegin(GL_TRIANGLES);
                    glMultiTexCoord2fARB(GL_TEXTURE0_ARB, getTextureCoordX(left), getTextureCoordZ(left));
                    glMultiTexCoord2fARB(GL_TEXTURE1_ARB, getDetailTextureCoordX(left), getDetailTextureCoordZ(left));
                    glVertex3f(left.getX(), leftY, left.getZ());
                    glMultiTexCoord2fARB(GL_TEXTURE0_ARB, getTextureCoordX(right), getTextureCoordZ(right));
                    glMultiTexCoord2fARB(GL_TEXTURE1_ARB, getDetailTextureCoordX(right), getDetailTextureCoordZ(right));
                    glVertex3f(right.getX(), rightY, right.getZ());
                    glMultiTexCoord2fARB(GL_TEXTURE0_ARB, getTextureCoordX(apex), getTextureCoordZ(apex));
                    glMultiTexCoord2fARB(GL_TEXTURE1_ARB, getDetailTextureCoordX(apex), getDetailTextureCoordZ(apex));
                    glVertex3f(apex.getX(), apexY, apex.getZ());
                glEnd();
                
                // disable texture 1
                glActiveTextureARB(GL_TEXTURE1_ARB);
                glDisable(GL_TEXTURE_2D);
                
                // disable texture 0
                glActiveTextureARB(GL_TEXTURE0_ARB);                
                glDisable(GL_TEXTURE_2D);
                
                // set texture env mode back to modulate
                glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
            }
            
            glEnable(GL_LIGHTING);
        }
    }
}

/* split the given MeshTreeNode into two, force splitting neighbors as needed */
void Patch::splitNode(MeshTreeNode* node) {
    // if the node already has children, no need to split further
    if (node->leftChild && node->rightChild) {
        return;
    }

    // if the base neighbor's base neighbor is not this node, force split the neighbor
    if (node->baseNeighbor && (node->baseNeighbor->baseNeighbor != node)) {
        splitNode(node->baseNeighbor);
    }

    // get child nodes
    node->leftChild = Patch::getNextFreeNode();
    node->rightChild = Patch::getNextFreeNode();

    // if there are no more free nodes, can not split further
    if (!node->leftChild || !node->rightChild) {
        return;
    }

    // update child node's pointers
    node->leftChild->baseNeighbor = node->leftNeighbor;
    node->leftChild->leftNeighbor = node->rightChild;
    node->rightChild->baseNeighbor = node->rightNeighbor;
    node->rightChild->rightNeighbor = node->leftChild;

    // update left neighbor's pointers
    if (node->leftNeighbor) {
        if (node->leftNeighbor->baseNeighbor == node) {
            node->leftNeighbor->baseNeighbor = node->leftChild;
        } else if (node->leftNeighbor->leftNeighbor == node) {
            node->leftNeighbor->leftNeighbor = node->leftChild;
        } else if (node->leftNeighbor->rightNeighbor == node) {
            node->leftNeighbor->rightNeighbor = node->leftChild;
        }
    }

    // update right neighbor's pointers
    if (node->rightNeighbor) {
        if (node->rightNeighbor->baseNeighbor == node) {
            node->rightNeighbor->baseNeighbor = node->rightChild;
        } else if (node->rightNeighbor->rightNeighbor == node) {
            node->rightNeighbor->rightNeighbor = node->rightChild;
        } else if (node->rightNeighbor->leftNeighbor == node) {
            node->rightNeighbor->leftNeighbor = node->rightChild;
        }
    }

    // update base neighbor's pointers
    if (node->baseNeighbor) {
        if (node->baseNeighbor->leftChild && node->baseNeighbor->rightChild) {
            node->baseNeighbor->leftChild->rightNeighbor = node->rightChild;
            node->baseNeighbor->rightChild->leftNeighbor = node->leftChild;
            node->leftChild->rightNeighbor = node->baseNeighbor->rightChild;
            node->rightChild->leftNeighbor = node->baseNeighbor->leftChild;
        } else {
            // base neighbor has no child nodes, force split the base neighbor
            splitNode(node->baseNeighbor);
        }
    } else {
        // no base neighbor
        node->leftChild->rightNeighbor = NULL;
        node->rightChild->leftNeighbor = NULL;
    }
}

/* update the variance of the patch, returning the max found in the variance search depth */
void Patch::updateVariance() {
    Vec3f upperLeft(getX() + xMin, getHeight(getX() + xMin, getZ() + zMin), getZ() + zMin);
    Vec3f upperRight(getX() + xMax, getHeight(getX() + xMax, getZ() + zMin), getZ() + zMin);
    Vec3f lowerLeft(getX() + xMin, getHeight(getX() + xMin, getZ() + zMax), getZ() + zMax);
    Vec3f lowerRight(getX() + xMax, getHeight(getX() + xMax, getZ() + zMax), getZ() + zMax);

    currentVariance = leftNodeVariance;
    updateNodeVariance(lowerRight, upperLeft, lowerLeft, 1);
    currentVariance = rightNodeVariance;
    updateNodeVariance(upperLeft, lowerRight, upperRight, 1);
}

/* update the node variance (difference between interpolated height and actual height) given the 
 * coordinates of a MeshTreeNode */
float Patch::updateNodeVariance(Vec3f left, Vec3f right, Vec3f apex, int nodePos) {
    float variance;

    // get center coordinate between left and right vertices
    GLfloat centerX = (left.getX() + right.getX()) / 2.0;
    GLfloat centerZ = (left.getZ() + right.getZ()) / 2.0;

    // get interpolated height at the center coordinate
    float interpolatedY = (left.getY() + right.getY()) / 2.0;

    // get actual height at the center coordinate
    float actualY = getHeight(centerX, centerZ);
    Vec3f center(centerX, actualY, centerZ);

    // compute variance as the absolute value of the actual height and the interpolated height
    variance = fabs(actualY - interpolatedY);

    // recursively update node variance
    if (fabs(left.getX() - right.getX()) >= 8 || fabs(left.getZ() - right.getZ()) >= 8) {
        variance = MAX(variance, updateNodeVariance(apex, left, center, nodePos << 1));
        variance = MAX(variance, updateNodeVariance(right, apex, center, (nodePos << 1) + 1));
    }

    if (nodePos < (1 << VARIANCE_SEARCH_DEPTH)) {
        currentVariance[nodePos] = 1 + variance;
    }

    return variance;
}

/* update the mesh of the patch */
void Patch::updateMesh() {
    Vec3f upperLeft(getX() + xMin, getY(), getZ() + zMin);
    Vec3f upperRight(getX() + xMax, getY(), getZ() + zMin);
    Vec3f lowerLeft(getX() + xMin, getY(), getZ() + zMax);
    Vec3f lowerRight(getX() + xMax, getY(), getZ() + zMax);

    currentVariance = leftNodeVariance;
    updateNode(&leftNode, lowerRight, upperLeft, lowerLeft, 1);
    currentVariance = rightNodeVariance;
    updateNode(&rightNode, upperLeft, lowerRight, upperRight, 1);
}

/* update the node given the coordinates of a MeshTreeNode, splitting as needed */
void Patch::updateNode(MeshTreeNode* node, Vec3f left, Vec3f right, Vec3f apex, int nodePos) {
    float variance;
    GLfloat centerX = (left.getX() + right.getX()) / 2.0;
    GLfloat centerZ = (left.getZ() + right.getZ()) / 2.0;
    Vec3f center(centerX, getY(), centerZ);
    if (nodePos < (1 << VARIANCE_SEARCH_DEPTH)) {
        GLfloat distanceX = getX() - world.getTruck()->getLocation().getX();
        GLfloat distanceZ = getZ() - world.getTruck()->getLocation().getZ();
        GLfloat distance = fabs(distanceX) + fabs(distanceZ);
        variance = currentVariance[nodePos] * width * 2 - distance;
    }
    if (nodePos >= (1 << VARIANCE_SEARCH_DEPTH) || variance > VARIANCE_LIMIT) {
        splitNode(node);
        if (node->leftChild && node->rightChild && 
            (fabs(left.getX() - right.getX()) >= 3 || fabs(left.getZ() - right.getZ()) >= 3)) {
            updateNode(node->leftChild, apex, left, center, nodePos << 1);
            updateNode(node->rightChild, right, apex, center, (nodePos << 1) + 1);
        }
    }
}

/* reset mesh nodes */
void Patch::resetMesh() {
    // reset left node
    leftNode.baseNeighbor = &rightNode;
    leftNode.leftChild = NULL;
    leftNode.rightChild = NULL;
    leftNode.leftNeighbor = NULL;
    leftNode.rightNeighbor = NULL;

    // reset right node
    rightNode.baseNeighbor = &leftNode;
    rightNode.leftChild = NULL;
    rightNode.rightChild = NULL;
    rightNode.leftNeighbor = NULL;
    rightNode.rightNeighbor = NULL;
}

/* get the texture coordinate for the x axis */
GLfloat Patch::getTextureCoordX(Vec3f center) {
    GLfloat coord = getX() + center.getX() + world.getTerrain()->getWidth() / 2.0;
    coord = coord / world.getTerrain()->getWidth();
    if (coord > 1.0) {
        coord = 1.0;
    } else if (coord < 0.0) {
        coord = 0.0;
    }
    return coord;
}

/* get the texture coordinate for the z axis */
GLfloat Patch::getTextureCoordZ(Vec3f center) {
    GLfloat coord = getZ() + center.getZ() + world.getTerrain()->getLength() / 2.0;
    coord = coord / world.getTerrain()->getLength();
    if (coord > 1.0) {
        coord = 1.0;
    } else if (coord < 0.0) {
        coord = 0.0;
    }
    return coord;
}

/* get the detail texture coordinate for the x axis */
GLfloat Patch::getDetailTextureCoordX(Vec3f center) {
    GLfloat max = getX() + xMax;
    GLfloat min = getX() + xMin;
    GLfloat coord = 1.0 - (max - (getX() + center.getX())) / (max - min);
    if (coord > 1.0) {
        coord = 1.0;
    } else if (coord < 0.0) {
        coord = 0.0;
    }
    return coord;
}

/* get the detail texture coordinate for the z axis */
GLfloat Patch::getDetailTextureCoordZ(Vec3f center) {
    GLfloat max = getZ() + zMax;
    GLfloat min = getZ() + zMin;
    GLfloat coord = 1.0 - (max - (getZ() + center.getZ())) / (max - min);
    if (coord > 1.0) {
        coord = 1.0;
    } else if (coord < 0.0) {
        coord = 0.0;
    }
    return coord;
}
