/*
 * MiniMap impl
 */
#include <iostream>
#include <iomanip>
#include <cstdio>
#include <algorithm>
#include <string>
#include <sstream>
#include <cmath>

#include <SDL_ttf.h>

#include "MiniMap.h"
#include "Display.h"
#include "Corral.h"
#include "SurfaceUtil.h"

using namespace std;

/* constructor */
MiniMap::MiniMap(GameWorld& gw): 
    GameObject(gw), 
    startTime(0), 
    totalTime(600000),
    elapsedTime(0),
    showEnd(false),
    showCowRemoved(false),
    cowTime(0)
{
    
    // set map size and offset
    mapW = 150;
    mapH = 150;
    offsetX = 0;
    offsetY = 0;
    
    // load map surface
    map = SDL_CreateRGBSurface(SDL_SWSURFACE, mapW, mapH, 32,
        SurfaceUtil::R_MASK, SurfaceUtil::G_MASK, SurfaceUtil::B_MASK, SurfaceUtil::A_MASK);
    if (map == NULL) {
        cout << "Create MiniMap Surface Failed! " << SDL_GetError() << endl;
    }
    
    // load icon and text surface
    char* file = "imgs/cow-icon.bmp";
    icon = SDL_LoadBMP(file);
    if (icon == NULL) {
        cout << "Load Icon Failed! " << file << endl;
    } else {
        SDL_SetColorKey(icon, SDL_SRCCOLORKEY, SDL_MapRGBA(icon->format, 0, 0, 255, 255));
        icon = SDL_ConvertSurface(icon, SurfaceUtil::RGBA_SURFACE->format, SDL_SWSURFACE);
        iconW = icon->w;
        iconH = icon->h;
    }
    
    // load start and end splash screens
    endScreen = SDL_CreateRGBSurface(SDL_SWSURFACE, Display::WIDTH, Display::HEIGHT, 32,
        SurfaceUtil::R_MASK, SurfaceUtil::G_MASK, SurfaceUtil::B_MASK, SurfaceUtil::A_MASK);
    if (endScreen == NULL) {
        cout << "Create MiniMap Surface Failed! " << SDL_GetError() << endl;
    }
    
    // cow removed screen
    string cowScr("imgs/cow-removed.bmp");
    cowRemoved = SDL_LoadBMP(cowScr.c_str());
    if (cowRemoved == NULL) {
        cout << "Load Failed! " << cowScr << endl;
    }
    SDL_SetColorKey(cowRemoved, SDL_SRCCOLORKEY, SDL_MapRGBA(cowRemoved->format, 0, 0, 255, 255));
    cowRemoved = SDL_ConvertSurface(cowRemoved, SurfaceUtil::RGBA_SURFACE->format, SDL_SWSURFACE);
    
    // initialize cow message array
    cowMsg[0] = " Success!";
    cowMsg[1] = " Yay!";
    cowMsg[2] = " Hooray!";
    cowMsg[3] = " Great!";
    
    // load fonts
    char* fontFile = "imgs/font.ttf";
    fontSm = TTF_OpenFont(fontFile, 26);
    fontLg = TTF_OpenFont(fontFile, 36);
    if (!fontSm || !fontLg) {
        cout << "Load Font Failed! " << TTF_GetError() << " ";
        cout << fontFile << endl;
    }
    
    // load timer surface
    timerW = 150;
    timerH = 75;
    timer = SDL_CreateRGBSurface(SDL_SWSURFACE, timerW, timerH, 32,
        SurfaceUtil::R_MASK, SurfaceUtil::G_MASK, SurfaceUtil::B_MASK, SurfaceUtil::A_MASK);
    if (timer == NULL) {
        cout << "Create Timer Surface Failed! " << SDL_GetError() << endl;
    }
}

/* destructor */
MiniMap::~MiniMap() {
    SDL_FreeSurface(map);
    SDL_FreeSurface(icon);
    SDL_FreeSurface(timer);
    SDL_FreeSurface(cowRemoved);
    SDL_FreeSurface(endScreen);
    TTF_CloseFont(fontSm);
    TTF_CloseFont(fontLg);
}

/* set show cow removed */
void MiniMap::setShowCowRemoved(bool v) {
    showCowRemoved = v;
    cowTime = SDL_GetTicks();
    randomMsg = rand() % 4;
}

/* draw minimap */
void MiniMap::draw() {
    // enable blending
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    
    // show end screens
    if (showEnd) {
        // clear screen
        SDL_Rect endRect = {0, 0, endScreen->w, endScreen->h};
        SDL_FillRect(endScreen, &endRect, SDL_MapRGBA(endScreen->format, 0, 0, 0, 100));
        
        // thanks text
        ostringstream thanks;
        thanks << "Thanks for playing!";
        
        SDL_Color color = {255, 255, 255};
        SDL_Surface* thanksText = TTF_RenderText_Blended(fontSm, thanks.str().c_str(), color);
        if (thanksText == NULL) {
            cout << "Load Text Failed! " << TTF_GetError() << endl;
        } else {
            SurfaceUtil::reverseY(thanksText);
            SDL_Rect textRect = {endScreen->w / 2 - thanksText->w / 2, 
                                 Display::HEIGHT - 300, 
                                 endScreen->w / 2 + thanksText->w / 2, 
                                 Display::HEIGHT - 300 + thanksText->h};
            SDL_BlitSurface(thanksText, NULL, endScreen, &textRect);
            SDL_FreeSurface(thanksText);
        }
        
        // show some game-over stats?
        // stats text
        Uint32 time = elapsedTime > totalTime ? totalTime : elapsedTime;
        int numCowsRemoved = world.getRemovedCows().size();
        bool gameSuccess = !(world.getCows().size() > 0);
        
        ostringstream stats;
        stats << ((gameSuccess) ? "Success! " : "Try Again! ");
        stats << "You rounded up " << numCowsRemoved;
        if (numCowsRemoved == 1) {
            stats << " cow in ";
        } else {
            stats << " cows in ";
        }
        stats << setfill('0');
        stats << setw(2);
        stats << time / 60000;
        stats << ":";
        stats << setw(2);
        stats << (time / 1000) % 60;
        
        SDL_Surface* statsText = TTF_RenderText_Blended(fontSm, stats.str().c_str(), color);
        if (statsText == NULL) {
            cout << "Load Text Failed! " << TTF_GetError() << endl;
        } else {
            SurfaceUtil::reverseY(statsText);
            SDL_Rect textRect = {endScreen->w / 2 - statsText->w / 2, 
                                 Display::HEIGHT - 100, 
                                 endScreen->w / 2 + statsText->w / 2, 
                                 Display::HEIGHT - 100 + statsText->h};
            SDL_BlitSurface(statsText, NULL, endScreen, &textRect);
            SDL_FreeSurface(statsText);
        }
        
        // "grade" test
        ostringstream grade;
        grade << "Your Grade:  ";
        if (!gameSuccess) {
            grade << "City Slicker";
        } else {
            float timeFrac = (float)time / (float)totalTime;
            if (timeFrac > 0.75) {
                grade << "Master Moover";
            } else if (timeFrac > 0.5) {
                grade << "Matador";
            } else {
                grade << "Lord of the Cows";
            }
        }
        
        SDL_Surface* gradeText = TTF_RenderText_Blended(fontSm, grade.str().c_str(), color);
        if (gradeText == NULL) {
            cout << "Load Text Failed! " << TTF_GetError() << endl;
        } else {
            SurfaceUtil::reverseY(gradeText);
            SDL_Rect textRect = {endScreen->w / 2 - gradeText->w / 2, 
                                 Display::HEIGHT - 200, 
                                 endScreen->w / 2 + gradeText->w / 2, 
                                 Display::HEIGHT - 200 + gradeText->h};
            SDL_BlitSurface(gradeText, NULL, endScreen, &textRect);
            SDL_FreeSurface(gradeText);
        }
        
        // restart text
        ostringstream restart;
        restart << "Press 'Q' to quit or 'R' to restart";
        
        SDL_Surface* restartText = TTF_RenderText_Blended(fontSm, restart.str().c_str(), color);
        if (restartText == NULL) {
            cout << "Load Text Failed! " << TTF_GetError() << endl;
        } else {
            SurfaceUtil::reverseY(restartText);
            SDL_Rect textRect = {endScreen->w / 2 - restartText->w / 2, 
                                 Display::HEIGHT - 400, 
                                 endScreen->w / 2 + restartText->w / 2, 
                                 Display::HEIGHT - 400 + thanksText->h};
            SDL_BlitSurface(restartText, NULL, endScreen, &textRect);
            SDL_FreeSurface(restartText);
        }
        
        glWindowPos2f(0, 0);
        glDrawPixels(endScreen->w, endScreen->h, GL_RGBA, GL_UNSIGNED_BYTE, endScreen->pixels);
    }
    
    // show cow removed, for a short time
    static Uint32 cowDuration = 5 * 1000;
    if (showCowRemoved) {
        if ((SDL_GetTicks() - cowTime) < cowDuration) {
            SDL_Rect cowRect = {cowRemoved->w / 4, 0, cowRemoved->w, cowRemoved->h};
            SDL_FillRect(cowRemoved, &cowRect, SDL_MapRGBA(icon->format, 0, 0, 0, 0));
            
            ostringstream numCows;
            numCows << cowMsg[randomMsg];
            numCows << " Only ";
            numCows << world.getCows().size();
            numCows << " Left!";
            
            SDL_Color color = {0, 162, 225};
            SDL_Surface* text_surface;    
            text_surface = TTF_RenderText_Solid(fontLg, numCows.str().c_str(), color);
            if (text_surface == NULL) {
                cout << "Load Text Failed! " << TTF_GetError() << endl;
            }
            SurfaceUtil::reverseY(text_surface);
            SDL_Rect textRect = {cowRemoved->w / 4, 
                                 cowRemoved->h / 2 - text_surface->h / 2, 
                                 cowRemoved->w, 
                                 cowRemoved->h};
            SDL_BlitSurface(text_surface, NULL, cowRemoved, &textRect);
            SDL_FreeSurface(text_surface);
            
            glWindowPos2f((GLfloat)(Display::WIDTH - cowRemoved->w) / 2.0, 
                          (GLfloat)(Display::HEIGHT - cowRemoved->h) / 2.0);
            glDrawPixels(cowRemoved->w, cowRemoved->h, GL_RGBA, GL_UNSIGNED_BYTE, cowRemoved->pixels);
        } else {
            showCowRemoved = false;
        }
    }
    
    if (map != NULL) {
        // clear map surface
        SDL_Rect mapRect = {0, 0, map->w, map->h};
        SDL_FillRect(map, &mapRect, SDL_MapRGBA(map->format, 0, 0, 0, 100));
        
        // draw mini-map of objs on x-z plane
        // cows
        GameObjVec& cows = world.getCows();
        for (GameObjVec::iterator i = cows.begin(); i != cows.end(); i++) {
            GameObjPtr obj = (*i);
            mapObject(obj, SDL_MapRGBA(map->format, 0, 0, 255, 255));
        }
        GameObjVec& removedCows = world.getRemovedCows();
        for (GameObjVec::iterator i = removedCows.begin(); i != removedCows.end(); i++) {
            GameObjPtr obj = (*i);
            mapObject(obj, SDL_MapRGBA(map->format, 128, 128, 128, 255));
        }
        
        // truck
        Truck* truck = world.getTruck();
        mapObject(GameObjPtr(truck), SDL_MapRGBA(map->format, 255, 0, 0, 255));
        
        // corral
        Corral* corral = world.getCorral();
        GameObjVec fences = corral->getFences();
        for (GameObjVec::iterator i = fences.begin(); i != fences.end(); i++) {
            GameObjPtr obj = (*i);
            mapObject(obj, SDL_MapRGBA(map->format, 0, 255, 0, 255), 2);
        }
        
        // draw map to OpenGL
        glWindowPos2f(Display::WIDTH - mapW - 10, 10);
        glDrawPixels(map->w, map->h, GL_RGBA, GL_UNSIGNED_BYTE, map->pixels);
    }
    
    if (icon != NULL && fontSm != NULL) {
        // clear icon text surface
        SDL_Rect iconRect = {icon->w / 2, 0, icon->w, icon->h};
        SDL_FillRect(icon, &iconRect, SDL_MapRGBA(icon->format, 0, 0, 0, 0));
        
        // num cows text
        ostringstream numCows;
        numCows << " X ";
        numCows << world.getCows().size();
        
        // draw text
        SDL_Color color = {0, 0, 0};
        SDL_Surface* text_surface;    
        text_surface = TTF_RenderText_Solid(fontSm, numCows.str().c_str(), color);
        if (text_surface == NULL) {
            cout << "Load Text Failed! " << TTF_GetError() << endl;
        }
        SurfaceUtil::reverseY(text_surface);
        SDL_Rect textRect = {icon->w / 2, icon->h / 2 - text_surface->h / 2, icon->w, icon->h};
        SDL_BlitSurface(text_surface, NULL, icon, &textRect);
        SDL_FreeSurface(text_surface);
        
        // draw icon and text to OpenGL
        glWindowPos2f(Display::WIDTH - iconW - 10, Display::HEIGHT - iconH - 10);
        glDrawPixels(icon->w, icon->h, GL_RGBA, GL_UNSIGNED_BYTE, icon->pixels);
    }
    
    // calculate time
    Uint32 currTime  = SDL_GetTicks();
    if (world.getGameIsRunning()) {
        elapsedTime = currTime - startTime;
        if (elapsedTime > totalTime) { 
            // time has run out, game over
            elapsedTime = totalTime; // show 0 on clock
            world.gameOver();
        }
    }
    
    if (timer != NULL && fontLg != NULL) {
        // clear timer surface
        SDL_FillRect(timer, NULL, SDL_MapRGBA(icon->format, 0, 0, 0, 0));
        
        // timer text
        ostringstream time;
        
        time << setfill('0');
        time << setw(2);
        time << (totalTime - elapsedTime) / 60000;
        time << ":";
        time << setw(2);
        time << ((totalTime - elapsedTime) / 1000) % 60;
        
        // draw text
        SDL_Color color = {0, 0, 0};
        // if time is running out, show timer in red
        if ((totalTime - elapsedTime) / 60000 < 1) {
            color.r = 255;
        }
        SDL_Surface* text_surface;    
        text_surface = TTF_RenderText_Solid(fontLg, time.str().c_str(), color);
        if (text_surface == NULL) {
            cout << "Load Text Failed! " << TTF_GetError() << endl;
        }
        SurfaceUtil::reverseY(text_surface);
        SDL_Rect textRect = {0, timer->h / 2 - text_surface->h / 2, timer->w, timer->h};
        SDL_BlitSurface(text_surface, NULL, timer, &textRect);
        SDL_FreeSurface(text_surface);
        
        // draw icon and text to OpenGL
        glWindowPos2f(Display::WIDTH - iconW - timerW - 10, Display::HEIGHT - timerH - 10);
        glDrawPixels(timer->w, timer->h, GL_RGBA, GL_UNSIGNED_BYTE, timer->pixels);
    }
    
    glDisable(GL_BLEND);
}

/* draw a spot on the map for an object */
void MiniMap::mapObject(GameObjPtr obj, Uint32 color, int spotSize) {
    int mmX = (int)round((obj->getX()/world.getTerrain()->getWidth()) * mapW);
    int mmY = (int)round((obj->getZ()/world.getTerrain()->getLength()) * mapH);
    mmX = (mapW/2) + mmX + offsetX;
    mmY = (mapH/2) - mmY + offsetY;
    SDL_Rect spot = { mmX, mmY, spotSize, spotSize};
    SDL_FillRect(map, &spot, color);
}
