/*mesh.cpp*/

#include "Mesh.h"
#include <iostream>

extern bool statsMode;

//Constructor
Mesh::Mesh(string name)
{
	nMaterials = 0; 		// mat[0] reserved for default meterial
	nVertices = 0; 
    nNorms = 0;
    nTextureCoords = 0;

	//model name, used to locate files
	modelName = name;

    LoadMesh(getPath(modelName + string(".obj")));
    LoadTex(getPath(modelName + string(".mtl")));
}

//Destructor
Mesh::~Mesh()
{
}

//Load the mesh. We are using vectors, then we dont have to read the file twice, just pushback the data into the vectors.
void Mesh::LoadMesh(string obj_file)
{
    static const int MAX_LINE_WIDTH = 200;
	char	token[MAX_LINE_WIDTH], buf[MAX_LINE_WIDTH];
	GLfloat	vec[3];
	string currentMatl;
    
	//open the file
	FILE    *scene = fopen(obj_file.c_str(),"r");

	if (!scene) 
	{
		cout<< string("Can not open object File \"") << obj_file << "\" !" << endl;
		return;
	}

    if (statsMode) {
	   cout << endl << obj_file << endl;
    }

	while(!feof(scene))
	{
		token[0] = NULL;
		//read in the token
		fscanf(scene,"%s", token);	

		//see what data you are about to read	

		if (!strcmp(token,"g"))
		{
			fscanf(scene,"%s",buf);
		}

		//load the mtl file
		else if (!strcmp(token,"mtllib"))
		{
			// do nothing, mtl file loaded elsewhere
		}

		//set the material
		else if (!strcmp(token,"usemtl"))
		{
			fscanf(scene,"%s",buf);
            currentMatl = string(buf);
		}

		//its a vertex
		else if (!strcmp(token,"v"))
		{
			fscanf(scene,"%f %f %f",&vec[0],&vec[1],&vec[2]);
			vList.push_back(Vec3f(vec));
		}

		//vertex normal (x,y,z)
		else if (!strcmp(token,"vn"))
		{
			fscanf(scene,"%f %f %f",&vec[0],&vec[1],&vec[2]);
			nList.push_back(Vec3f(vec));
		}

		//vertex texture, 2 parameters (u,v)
		else if (!strcmp(token,"vt"))
		{
			fscanf(scene,"%f %f",&vec[0],&vec[1]);
			tList.push_back(Vec3f(vec));
		}

		//faces, Given as f v/vt/vn v/vt/vn v/vt/vn (for 3 sided face)
		else if (!strcmp(token,"f"))
		{
			// read whole line
            fgets(buf,MAX_LINE_WIDTH,scene);
            
            // operate on line
            string line(buf);
            
            // tokenize
            vector<string> verts;
            tokenize(verts, line, " ");
            
            // for each vertex of this face
            Face face;
            for (size_t i=0; i < verts.size(); i++) {
                vector<string> indices;
                split(indices, verts[i], "/");

                Vertex vt(-1,-1,-1);
				vt.v = getIndex(vList.size(), indices[0]);
                if (indices.size() > 1) {
				    vt.t = getIndex(tList.size(), indices[1]);
                }
                if (indices.size() > 2) {
				    vt.n = getIndex(nList.size(), indices[2]);
                }  
                
                face.push_back(vt);
			}
		
			//put Face with vertices into facelist.
            faceMap[currentMatl].push_back(face);
		}

		//reading a comment
		else if (!strcmp(token,"#")) {
			fgets(buf,MAX_LINE_WIDTH,scene);
        }
	}

	//close file
	if (scene) fclose(scene);

	//update number of verts, tris etc.
	nVertices = vList.size();
	nNorms = nList.size();
	nTextureCoords = tList.size();
    if (statsMode) {
        printf("vertex: %d, normal: %d, texture: %d\n",nVertices, nNorms, nTextureCoords);
    }
}

/* 
 * get the actual index into a vector, from a possibly negative 
 * 1-based input index used in obj files
 */
int Mesh::getIndex(size_t vecSize, string indexStr) {
    // empty strings get -1 index
    if (indexStr.empty()) {
        return -1;
    }
    
    // convert to int
    int index = atoi(indexStr.c_str());
    int realIndex = 0;
    if (index < 0) {
        realIndex = (int)vecSize + index;
    } else if (index > 0) {
        realIndex = index - 1;
    }
    if (realIndex >= (int)vecSize || realIndex < 0) {
        cout << "Bad Index! " << realIndex << " " << index << " " << vecSize << endl;
    }
    return realIndex;
}

//Load the mtl file, the materials/texture
void Mesh::LoadTex(string tex_file)
{
    static const int MAX_LINE_WIDTH = 200;
	char	token[MAX_LINE_WIDTH], buf[MAX_LINE_WIDTH];
	float	r,g,b;
	map<string, GLuint> loadedTextures;
    
	//Open file			
	FILE    *texture = fopen(tex_file.c_str(),"r");

	//If not opened, print
	if (!texture) 
	{
		cout << "Can't open material file \"" << tex_file << "\"!" << endl;
		return;
	}

    if (statsMode) {
	   cout << tex_file << endl;
    }

    // current material
    string currentMatl;

	//read through the file
	while(!feof(texture))
	{
		token[0] = NULL;
		//read one line
		fscanf(texture,"%s", token);

		//Connect the material with its id in the map
		if (!strcmp(token,"newmtl"))
		{
			fscanf(texture,"%s",buf);
            currentMatl = string(buf);
		}
		//ambient
		else if (!strcmp(token,"Ka"))
		{
			fscanf(texture,"%f %f %f",&r,&g,&b);
			matlMap[currentMatl].Ka[0] = r;
			matlMap[currentMatl].Ka[1] = g;
			matlMap[currentMatl].Ka[2] = b;
			matlMap[currentMatl].Ka[3] = 1;
		}
		//diffuse
		else if (!strcmp(token,"Kd"))
		{
			fscanf(texture,"%f %f %f",&r,&g,&b);
			matlMap[currentMatl].Kd[0] = r;
			matlMap[currentMatl].Kd[1] = g;
			matlMap[currentMatl].Kd[2] = b;
			matlMap[currentMatl].Kd[3] = 1;
		}
		//specular
		else if (!strcmp(token,"Ks"))
		{
			fscanf(texture,"%f %f %f",&r,&g,&b);
			matlMap[currentMatl].Ks[0] = r;
			matlMap[currentMatl].Ks[1] = g;
			matlMap[currentMatl].Ks[2] = b;
			matlMap[currentMatl].Ks[3] = 1;
		}
		//shininess
		else if (!strcmp(token,"Ns"))
		{
			fscanf(texture,"%f",&r);
			matlMap[currentMatl].Ns = r;
		}
		//if there is a texture
		else if(!strcmp(token,"map_Kd"))
		{
			fscanf(texture,"%s",buf);
			matlMap[currentMatl].map = buf;

            // try to load texture if specified
            string textureFile = getPath(matlMap[currentMatl].map);
            map<string, GLuint>::iterator iter = loadedTextures.find(textureFile);
            if (iter == loadedTextures.end())  {
                // not yet loaded
                GLuint textId = loadTexture(textureFile);
                matlMap[currentMatl].textureId = textId;
                loadedTextures[textureFile] = textId;
            } else {
                // already loaded
                GLuint textId = iter->second;
                if (statsMode) {
                    cout << "Texture exists: " << textureFile << " " << textId << endl;
                }
                matlMap[currentMatl].textureId = textId;
            }
		} else if (!strcmp(token,"#")) {
            // comment
            fgets(buf,MAX_LINE_WIDTH,texture);
        }
	}

    if (statsMode) {
	   cout << "total material: " << matlMap.size() << endl;
    }
    
	//close file
    if (texture)
    {
        fclose(texture);
    }
}

/* utility method to load a texture from file */
GLuint Mesh::loadTexture(string fileName) {
    // try to load file, only bmps for now
    GLuint textureId(0);    
    SDL_Surface* surface = SDL_LoadBMP(fileName.c_str());
    if (surface != NULL) {
        // generate texture
        glGenTextures(1, &textureId);
        glBindTexture(GL_TEXTURE_2D, textureId);
        
        // (re)sets the default texture params
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_LINEAR);
        
        // build mipmaps
        gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, surface->w, surface->h,
                          GL_BGR, GL_UNSIGNED_BYTE, surface->pixels);
        
        // free surface
        SDL_FreeSurface(surface);
        if (statsMode) {
            cout << "Texture Loaded: " << fileName << " Id: " << textureId << endl;
        }
    } else {
        cout << "Texture Loading Failed: " << fileName << endl;
    }
    return textureId;
}


//Render the mesh
void Mesh::render(void)
{   
    if (!glIsList(dspList)) {
        buildDspList();
    }
    glCallList(dspList);
}
 
//Create a displaylist to be drawn later
void Mesh::buildDspList(void)
{
	//Generate a display list
    dspList = glGenLists(1);
    glNewList(dspList,GL_COMPILE);

    for (FaceMap::iterator fmi = faceMap.begin(); fmi != faceMap.end(); fmi++) {
        string matlName = fmi->first;
        FaceList list = fmi->second;
        if (statsMode) {
            cout << "Drawing Matl: " <<  matlName << endl;
            cout << "Faces: " <<  list.size() << endl;
        }
        
        //Get the material properties
        Material matl = matlMap[matlName];
        GLfloat amb[] = {matl.Ka[0], matl.Ka[1], matl.Ka[2] };    
        GLfloat diff[] = {matl.Kd[0], matl.Kd[1], matl.Kd[2] }; 
        GLfloat spec[] = {matl.Ks[0], matl.Ks[1], matl.Ks[2] };   
         
        //Set the material properties
        glMaterialfv(GL_FRONT, GL_AMBIENT, amb);
        glMaterialfv(GL_FRONT, GL_DIFFUSE, diff);
        glMaterialfv(GL_FRONT, GL_SPECULAR, spec);
        glMaterialf( GL_FRONT, GL_SHININESS, matl.Ns);
        
        // bind texture, if exists
        if (glIsTexture(matl.textureId)) {
            glEnable(GL_TEXTURE_2D);
            glBindTexture(GL_TEXTURE_2D, matl.textureId);
        }

        //draw polygons, switch types if needed
        Face firstFace = list.front();
        if (firstFace.size() == 3) {
            glBegin(GL_TRIANGLES);
        } else if (firstFace.size() == 4) {
            glBegin(GL_QUADS);
        } else {
            cout << "Too Many verts/face! " << firstFace.size() << endl;
        }
        for (FaceList::size_type i = 0; i < list.size(); i++) {
            Face face = list[i];
            for (Face::size_type j = 0; j < face.size(); j++) {
                // draw vertex, normal, texture
                Vertex vtx = face[j];
                
                if (vtx.n > -1) { //not all vert have normals
                    Vec3f norm = nList[vtx.n];
                    glNormal3f(norm.getX(), norm.getY(), norm.getZ());
                }
                if (vtx.t > -1) { //not all verts have texture
                    Vec3f text = tList[vtx.t];
                    glTexCoord2f(text.getX(), text.getY() );
                }
                Vec3f vert = vList[vtx.v];
                glVertex3f(vert.getX(), vert.getY(), vert.getZ());
            }
        }
        glEnd();
        
        if(glIsTexture(matl.textureId)) {
            glDisable(GL_TEXTURE_2D);
        }
    }
        
    glEndList(); 
} 

/* return path to file */
string Mesh::getPath(string fileName) {
    static const string MODEL_DIR("models");
    return MODEL_DIR + string("/") + modelName + string("/") + fileName;
}

/* helper method to tokenize a string */
void tokenize(vector<string>& tokens, const string& str, const string& delimiters) {
    // Skip delimiters at beginning.
    string::size_type lastPos = str.find_first_not_of(delimiters, 0);
    // Find first "non-delimiter".
    string::size_type pos     = str.find_first_of(delimiters, lastPos);

    while (string::npos != pos || string::npos != lastPos)
    {
        // Found a token, add it to the vector.
        tokens.push_back(str.substr(lastPos, pos - lastPos));
        // Skip delimiters.  Note the "not_of"
        lastPos = str.find_first_not_of(delimiters, pos);
        // Find next "non-delimiter"
        pos = str.find_first_of(delimiters, lastPos);
    }
}

/* helper method to split a string */
void split(vector<string>& tokens, const string& str, const string& delimiters) {
    // Find first "delimiter".
    string::size_type current = 0;

    while (true) {
        string::size_type pos = str.find_first_of(delimiters, current);
        if (pos == string::npos) {
            tokens.push_back(str.substr(current));
            break;
        } else {
            tokens.push_back(str.substr(current, pos - current));
            current = pos + 1;
        }
    }
}
