/*
 * 
 * Represents the game world, sets up the objects in the game,
 * keeps track of them, updates them, and redraws them as driven
 * by the game loop.  Also handles input events, which mostly 
 * update the world
 * 
 */
 
#ifndef GAMEWORLD_H_
#define GAMEWORLD_H_

#include <SDL_opengl.h>
#include <SDL.h>
#include <SDL_mixer.h>
#include <vector>

#include "GameObject.h"
#include "Terrain.h"
#include "Frustum.h"

using namespace std;

/* define reference counting shared_ptr type to hold GameObjects */
/* using plain pointers now for lack of TR1 support, changes marked with NO_TR1 */
//NO_TR1 typedef tr1::shared_ptr<GameObject> GameObjPtr;
typedef GameObject* GameObjPtr;

/* define vector to GameObjPtrs */
typedef vector<GameObjPtr> GameObjVec;

/* forward decl */
class GameLoop;
class Terrain;
class Corral;
class Truck;
class MiniMap;


/* helper functor to call draw on each GameObject */
class DrawFunctor {
  public:
    /* call draw on GameObject */
    void operator()(GameObjPtr& gameObjPtr) {
        gameObjPtr->draw();
    }
};

/* helper functor to call update on each GameObject */
class UpdateFunctor {
  public:

    /* call update on GameObject */
    void operator()(GameObjPtr& gameObjPtr) {
        gameObjPtr->update(timeElapsed);
    }
    
    /* sets time elapsed */
    void setTimeElapsed(Uint32 t) { timeElapsed = t; }

  private:
    /* stores time elapsed since last update */
    Uint32 timeElapsed;
};


/*
 * Class that represents the game world and all objects in it
 */
class GameWorld {
  public:
    /* constructor */
	GameWorld(unsigned int numCows);
    
    /* destructor */
	~GameWorld();
    
    /* reset the world */
    void reset();
    
    /* update the world according to timeElapsed since last update */
    void updateWorld(Uint32 timeElapsed);
    
    /* redraw the world */
    void drawWorld();
    
    /* handle all the input events in queue */
    void handleInputEvents();
    
    /* sets a handle to the GameLoop */
    void setGameLoop(GameLoop* loop) { gameLoop = loop; }
    
    /* add updatable object to game */
    void addUpdatableGameObj(GameObjPtr go);
    
    /* add object to game, must specify whether updateable and collidable */
    void addGameObj(GameObjPtr go, bool updatable, bool collidable);
    
    /* add a "cow" to the game */
    void addCow(GameObjPtr go);
    
    /* remove a "cow" from the game (i.e. cow is in corral or gone) */
    void removeCow(GameObject* cow);
    
    /* returns the list of cows */
    GameObjVec& getCows() { return cows; }
    
    /* returns the list of cows */
    GameObjVec& getRemovedCows() { return removedCows; }
    
    /* set the "truck" for this game */
    void setTruck(Truck* go);
    
    /* get the "truck" for this game */
    Truck* getTruck() { return truck; }

    /* set the terrain for this game */
    void setTerrain(Terrain* t) { terrain = t; }
    
    /* get the terrain for this game */
    Terrain* getTerrain() { return terrain; }

    /* returns the list of all objects in the world */
    GameObjVec& getAllObjs() { return objects; }
    
    /* returns the list of in the world that need CD */
    GameObjVec& getCollidableObjs() { return collidableObjects; }
    
    /* get the "corral" for this game */
    Corral* getCorral() { return corral; }
    
    /* get the current view frustum */
    Frustum* getViewFrustum() { return frustum; }
    
    /* get whether game is running (started and not ended) */
    bool getGameIsRunning() { return gameIsRunning; }
    
    /* set whether game is running (started and not ended) */
    void setGameIsRunning(bool v) { gameIsRunning = v; }
    
    /* signal game over, either time has run out or success */
    void gameOver();
    
    /* get the minimap */
    MiniMap* getMiniMap() { return miniMap; }
    
    
  private:
    //--------- vars ----------------
    
    /* handle to the running gameloop */
    GameLoop* gameLoop;
    /* functor for GameObject updates */
    UpdateFunctor updateFunctor;
    /* functor for GameObject draws */
    DrawFunctor drawFunctor;
    
    /* vector of all game objects, the "master" list for deletes */
    GameObjVec objects;
    /* vector of game objects that need updating */
    GameObjVec updatableObjects;
    /* vector of game objects that need collision detection */
    GameObjVec collidableObjects;
    /* vector of active cows */
    GameObjVec cows;
    /* vector of removed cows */
    GameObjVec removedCows;
    
    /* the truck, which the camera follows */
    Truck* truck;
    /* the target corral */
    Corral* corral;
    /* pointer to the terrain */
    Terrain* terrain;
    /* pointer to mini-map */
    MiniMap* miniMap;
    
    
    /* the current view frustum, used to check for frustum intersections */
    Frustum* frustum;
    
    /* camera settings, to be able to switch views in game */
    GLfloat cBehind;
    GLfloat cBehindHeight;
    GLfloat cAhead;
    GLfloat cAheadHeight;
    GLfloat cAngle;
    
    /* whether game is running (has started and not ended) */
    bool gameIsRunning;
    
    /* sounds for different keys */
    Mix_Music* hornSound;
    
    
    
    //--------- methods ----------------
    
    /* sets up opengl settings that apply to all objects */
    void setupOpenGL();
    
    /* clears screen */
    void clearScreen();

    /* position the camera (viewing transforms) */
    void positionCamera();

    /* position the light  */
    void positionLight();
    
    /* set up the perspective transforms */
    void setPerspective();
    
    /* handle a keyboard event */
    void GameWorld::handleKeyboardEvent(SDL_Event& event);
    
    /* no one should use assignment operator for this class */
    GameWorld& operator=(const GameWorld& rhs);
    /* no one should use copy constructor for this class */
    GameWorld(const GameWorld& rhs);
    
    /* printing debug */
    friend std::ostream& operator<<(std::ostream& s, const GameWorld& n);
};

/* helper functor to remove cows */
class CowRemovePred {
  public:
    /* pointer to cow being removed */
    GameObject* cowPtr;
    
    /* true if cowPtr == gameObjPtr.get() */
    bool operator()(const GameObjPtr& gameObjPtr) {
        //NO_TR1 return (cowPtr == gameObjPtr.get());
        return (cowPtr == gameObjPtr);
    }
};

/* helper functor to delete pointers */
struct DeleteObject {
    template<typename T>
    void operator()(const T* ptr) {
        delete ptr;
    }
};

/* printing GameWorld */
std::ostream& operator<<(std::ostream& s, const GameWorld& n);

#endif /*GAMEWORLD_H_*/
