/*
 * 
 * Represents the game world, sets up the objects in the game,
 * keeps track of them, updates them, and redraws them as driven
 * by the game loop
 * 
 * 
 */
 
#include <SDL_opengl.h>
#include <SDL.h>
#include <cstdlib>
#include <iostream>
#include <vector>
#include <utility>
#include <algorithm>

#include "GameWorld.h"
#include "GameLoop.h"
#include "Display.h"
#include "Cow.h"
#include "Terrain.h"
#include "Skybox.h"
#include "Truck.h"
#include "Corral.h"
#include "Frustum.h"
#include "House.h"
#include "MiniMap.h"
#include "FractalTree.h"
#include "SurfaceUtil.h"
#include "ImageHeightMap.h"

using namespace std;

extern bool debugMode;
extern bool musicMode;
extern bool statsMode;
extern bool multitextureMode;

/* constructor, sets up the game world */
GameWorld::GameWorld(unsigned int numCows):
    cBehind(-12),
    cBehindHeight(5),
    cAhead(50),
    cAheadHeight(0),
    cAngle(0),
    gameIsRunning(false),
    hornSound(NULL)
{   
    // setup global OpenGL options
    setupOpenGL();
    
    // create a reference to self to pass into GameObjs
    GameWorld& worldRef = (*this);
    
    // create the skybox
    // NOTE: skybox must be the first object rendered since it temporarily turns off
    // depth testing while drawing
    GameObjPtr skybox(new Skybox(worldRef));
    skybox->load();
    addGameObj(skybox, false, false);
    
    // create the terrain
    terrain = new Terrain(worldRef);
    setTerrain(terrain);
    terrain->load();
    terrain->setLocation(0, 0, 0);
    GameObjPtr terrainObj(terrain);
    addGameObj(terrainObj, false, false);
    
    // create truck
    Truck* tmpT(new Truck(worldRef));
    setTruck(tmpT);
    truck->load();
    
    // create some cows, place them at (somewhat)random locations
    vector< pair<int,int> > locs;
    for (int i = 5; i < 20; i += 4) {
        for (int j = -5; j > -20; j -= 4) {
            pair<int,int> p = make_pair(i, j);
            locs.push_back(p);
        }
    }
    random_shuffle(locs.begin(), locs.end());
    
    for (unsigned int i=0; i < numCows; i++) {
        if (i < locs.size()) {
            pair<int,int> p = locs[i];
            GameObjPtr obj(new Cow(worldRef));
            obj->setLocation( p.first, 0, p.second);
            addCow(obj);
        }
    }
    
    // create a corral
    corral = new Corral(worldRef);
    
    // create some trees, place them from tree map
    locs.clear();
    ImageHeightMap treeMap;
    treeMap.loadFromImage("imgs/tree-map.bmp");
    GLfloat bounds = Cow::getCowBounds() - 1;
    for (GLfloat i = -bounds; i < bounds; i++) {
        for (GLfloat j = -bounds; j < bounds; j++) {
            GLfloat h = treeMap.getHeight(i, j);
            if (h < 49 && h > 1) { //tree location found
                locs.push_back(make_pair(i,j));
                if (statsMode) {
                    cout << "Tree Loc found: " << i << " " << j << " " << h << endl;
                }
            }
        }
    }
    if (statsMode) {
        cout << "Tree Locs: " << locs.size() << endl;
    }
    random_shuffle(locs.begin(), locs.end());
    
    for (unsigned int i=0; i < locs.size(); i++) {
        pair<int,int> p = locs[i];
        FractalTree* tree = new FractalTree(worldRef);
        tree->setLocation(p.first, 
                          getTerrain()->getHeight(p.first, p.second), 
                          p.second);
        tree->create();
        addGameObj(tree, false, true);
    }
    
    // create a MiniMap
    miniMap = new MiniMap(worldRef);
    addGameObj(miniMap, false, false);
    
    if (musicMode) {
        // load truck horn sound effect
        hornSound = Mix_LoadMUS("sounds/car_horn.wav");
        if (hornSound == NULL) {
            cout << "Failed to load sound effect " << Mix_GetError() << endl;
        }
    }
}

/* destructor, clean up */
GameWorld::~GameWorld() {
    //NO_TR1 have to delete pointers manually
    //for_each(objects.begin(), objects.end(), DeleteObject());
    
    // free sounds
    if (musicMode) {
        Mix_FreeMusic(hornSound);
    }
}

/* reset the world */
void GameWorld::reset() {
    // reset truck to starting location
    truck->setLocation(0, 0, 0);
    truck->setAngleY(0);
    truck->setVelocity(0,0,0);
    truck->setAcceleration(0,0,0);
    
    // reset camera position
    cBehind = -12;
    cBehindHeight = 5;
    cAhead = 50;
    cAheadHeight = 0;
    cAngle = 0;
    
    // create random locations
    vector< pair<int,int> > locs;
    for (int i = 5; i < 20; i += 4) {
        for (int j = -5; j > -20; j -= 4) {
            pair<int,int> p = make_pair(i, j);
            locs.push_back(p);
        }
    }
    random_shuffle(locs.begin(), locs.end());
    
    // reset removed cows
    for (GameObjVec::iterator i = removedCows.begin(); i != removedCows.end(); i++) {
        GameObjPtr obj = (*i);
        obj->reset();
        addCow(obj);
    }
    removedCows.erase(removedCows.begin(), removedCows.end());
    
    // set cows in random locations
    int loc = 0;
    for (GameObjVec::iterator i = cows.begin(); i != cows.end(); i++) {
        GameObjPtr obj = (*i);
        pair<int,int> p = locs[loc++];
        obj->setLocation(p.first, 0, p.second);
    }
    
    // don't show end screen
    miniMap->setShowEnd(false);
    miniMap->setStartTime(SDL_GetTicks());
    
    // set game to run again
    setGameIsRunning(true);
}

/* signal game over, either time has run out or success */
void GameWorld::gameOver() {
    // show end screen
    miniMap->setShowEnd(true);
    // stop game updating
    gameIsRunning = false;
}

/* sets up opengl settings that apply to all objects */
void GameWorld::setupOpenGL() {
    // enable options
    glEnable(GL_DEPTH_TEST);
    glShadeModel(GL_SMOOTH);
    
    // multisampling AA
    glEnable(GL_MULTISAMPLE);
    
    // cull inside facing polygons
    glEnable(GL_CULL_FACE);
    glCullFace(GL_BACK);
    
    // set up viewport       
    glViewport(0, 0, Display::WIDTH, Display::HEIGHT);
    
    // set up lighting
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    GLfloat ambient[] = { 0.1, 0.1, 0.1, 1.0 }; 
    GLfloat diffuse[] = { 1, 1, 1, 1.0 }; 
    glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
    
    // normalize normals
    glEnable(GL_NORMALIZE);
}

/* update the world according to timeElapsed since last update */
void GameWorld::updateWorld(Uint32 timeElapsed) {
    // update all updatable objects
    updateFunctor.setTimeElapsed(timeElapsed);
    for_each(updatableObjects.begin(), updatableObjects.end(), updateFunctor);
}

/* redraw the world */
void GameWorld::drawWorld() {
    // clear screen
    clearScreen();
    
    // create view frustum
    frustum = new Frustum();
    
    // setup perspective (projective transforms)
    setPerspective();
    
    // position camera (view transforms)
    positionCamera();
    
    // let each object draw themselves (model transforms)
    for_each(objects.begin(), objects.end(), drawFunctor);
    
    // position the light
    positionLight();
    
    // swap buffers (and flush commands)
    SDL_GL_SwapBuffers();
}

/* position the light  */
void GameWorld::positionLight() {
    // just let light be fixed wrt to world
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    GLfloat position[4] = {1, 1, 0.5, 0};
    glLightfv(GL_LIGHT0, GL_POSITION, position);
    glPopMatrix();
}

/* position the camera (viewing transforms) */
void GameWorld::positionCamera() {
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    
    // camera follows a bit behind and above the truck, in direction of travel
    Vec3f eye(0, 0, -1);
    eye.rotateY(truck->getAngleY() + cAngle);
    
    // camera follows velocity a little if non-zero
    GLfloat maxAngle = 20;
    GLfloat velAngle = 0;
    Vec3f truckVel = truck->getVelocity();
    truckVel.setY(0);
    if (truckVel.magnitude() > 0) {
        if (truck->getDrivingForward()) {
            velAngle = eye.getAngle(truckVel);
            if (velAngle > maxAngle) {
                velAngle = maxAngle;
            }
            Vec3f cp = eye.crossProduct(truckVel);
            if (cp.getY() < 0) {
                velAngle = -velAngle;
            }
            eye.rotateY(velAngle);
        }
    }
    
    Vec3f center(eye);
    eye.mult(cBehind); //units behind truck
    eye.add(Vec3f(0, cBehindHeight, 0)); // behind height
    eye.add(truck->getLocation());
        
    center.mult(cAhead); //units ahead of truck
    center.add(Vec3f(0, cAheadHeight, 0)); // ahead height
    center.add(truck->getLocation());
    
    // setup camera
    gluLookAt(eye.getX(), eye.getY(), eye.getZ(),
              center.getX(), center.getY(), center.getZ(),
              0, 1, 0);
    
    // set frustum location
    frustum->setLocation(eye);
    frustum->calculatePlanes();
}

/* set up the perspective transforms */
void GameWorld::setPerspective() {
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    GLfloat fov = 60.0;
    GLfloat near = 0.1;
    GLfloat far = 300.0;
    gluPerspective(fov, (GLfloat)Display::WIDTH / (GLfloat)Display::HEIGHT, 
                   near, far);
}

/* clears screen */
void GameWorld::clearScreen() {
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glClearDepth(1.0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

/* add object to game, must specify whether updateable and collidable */
void GameWorld::addGameObj(GameObjPtr go, bool updatable, bool collidable) {
    objects.push_back(go);
    if (updatable) {
        updatableObjects.push_back(go);
    }
    if (collidable) {
        collidableObjects.push_back(go);
    }
}

/* add a "cow" to the game */
void GameWorld::addCow(GameObjPtr go) {
    cows.push_back(go);
    addGameObj(go, true, true);
}

/* remove a "cow" from the game (i.e. cow is in corral or gone) */
void GameWorld::removeCow(GameObject* cow) {
    static CowRemovePred cowRemover;
    cowRemover.cowPtr = cow;
    // add to removed cows
    removedCows.push_back(cow);
    // remove from active cows
    cows.erase(remove_if(cows.begin(), cows.end(), cowRemover), cows.end());
    
    // see if game over
    if (cows.size() == 0) {
        gameOver();
    } else {
        // show some encouragement
        miniMap->setShowCowRemoved(true);   
    }
}

/* set the "truck" for this game */
void GameWorld::setTruck(Truck* go) {
    truck = go;
    addGameObj(go, true, true);
}

/* handle all the input events in queue */
void GameWorld::handleInputEvents() {
    // stores the current event
    SDL_Event event;
    
    // pull all the events off the queue and handle as necessary
    while (SDL_PollEvent(&event)) {
        switch (event.type) {
            case SDL_QUIT:
                gameLoop->exit();
                break;
        
            case SDL_KEYDOWN:
            case SDL_KEYUP:
                handleKeyboardEvent(event);
                break;

            default:
                break;
        }
    }
}

/* handle a keyboard event */
void GameWorld::handleKeyboardEvent(SDL_Event& event) {
    SDLKey key = event.key.keysym.sym;
    SDLMod mod = event.key.keysym.mod;
    Uint8* keyState = SDL_GetKeyState(NULL);
    
    // truck steering amts, maybe put in Truck class later
    GLfloat accel = 10;
    GLfloat angle = 10;

    // key pressed
    if (event.type == SDL_KEYDOWN) {
        // if game is over, restart if the pressed key is SDLK_r
        if (!gameIsRunning && key == SDLK_r) {
            reset();
        }
        
        // process key
        switch (key) {
            case SDLK_q:
            case SDLK_ESCAPE:
                gameLoop->exit();
                break;
                
            case SDLK_UP:
                truck->accelerate(accel);
                truck->playSound();
                break;
                
            case SDLK_DOWN:
                truck->accelerate(-accel);
                truck->playSound();
                break;
                
            case SDLK_LEFT:
                truck->setAngleY(truck->getAngleY() + angle);
                if (keyState[SDLK_UP]) {
                    truck->accelerate(accel);
                } else if (keyState[SDLK_DOWN]) {
                    truck->accelerate(-accel);
                }
                truck->playSound();
                break;
                
            case SDLK_RIGHT:
		//truck->setNewGravel();
                truck->setAngleY(truck->getAngleY() - angle);
                if (keyState[SDLK_UP]) {
                    truck->accelerate(accel);
                } else if (keyState[SDLK_DOWN]) {
                    truck->accelerate(-accel);
                }
                truck->playSound();
                break;
                
            case SDLK_SPACE: //break
                truck->setVelocity(0,0,0);
                truck->setAcceleration(0,0,0);
                break;
                
            // camera adjustments
            case SDLK_u:
                if (mod & KMOD_SHIFT) {
                    cBehind += 1;
                } else {
                    cBehind -= 1;
                }
                break;

            case SDLK_i:
                if (mod & KMOD_SHIFT) {
                    if (cBehindHeight < 30) {
                        cBehindHeight += 1;
                    }
                } else {
                    //if (cBehindHeight > 1) {
                        cBehindHeight -= 1;
                    //}
                }
                break;
                
            case SDLK_j:
                if (mod & KMOD_SHIFT) {
                    cAhead += 1;
                } else {
                    cAhead -= 1;
                }
                break;
                
            case SDLK_k:
                if (mod & KMOD_SHIFT) {
                    cAheadHeight += 1;
                } else {
                    cAheadHeight -= 1;
                }
                break;
                
            case SDLK_l:
                if (mod & KMOD_SHIFT) {
                    cAngle += 10;
                } else {
                    cAngle -= 10;
                }
                break;
                
    	    case SDLK_s:
                // honk the horn of the car
                Mix_PlayMusic(hornSound, 0);
                
                // stress the cows
                for (GameObjVec::iterator i = cows.begin(); i != cows.end(); i++) {
                    ((Cow*)(*i))->applyHorn();
                }
                for (GameObjVec::iterator i = removedCows.begin(); i != removedCows.end(); i++) {
                    ((Cow*)(*i))->applyHorn();
                }
                
                break;
                
            case SDLK_d:
                // debug
                debugMode = !debugMode;
                break;
            case SDLK_z:
                // stats
                statsMode = !statsMode;
                break;
            case SDLK_t:
                // multitexture
                multitextureMode = !multitextureMode;
                break;
            default:
                break;
        }
    } else if (event.type == SDL_KEYUP) {    
        switch (key) {
            case SDLK_UP:
                truck->setAcceleration(0,0,0);
                break;
                
            case SDLK_DOWN:
                truck->setAcceleration(0,0,0);
                break;
                
            default:
                break;
        }   
    }
}

/* debug printing GameObjVecs */
std::ostream& operator<<(std::ostream& s, const GameObjVec& n) {
    cout << "Size: " << n.size() << endl;
    for (unsigned int i=0; i < n.size(); i++) {
        GameObjPtr g = n[i];
        cout << g << " " << g->getLocation() << endl;
    }
    return s;
}

/* debug printing GameWorld */
std::ostream& operator<<(std::ostream& s, const GameWorld& n) {
    cout << "Cows: " << endl;
    cout << n.cows;
    cout << "RemovedCows: " << endl;
    cout << n.removedCows;   
    return s;
}
