/*
 * Abstract class for an object in the game, like a house or
 * car.  GameObjects should keep track of and update its own state and
 * know how to draw itself.  This base class povides functinality
 * that every object needs, subclass this to provide
 * specialized behavior
 * 
 */
#ifndef GAMEOBJECT_H_
#define GAMEOBJECT_H_

#include <SDL_opengl.h>
#include <SDL.h>
#include <SDL_mixer.h>
#include <vector>

#include "Vec3f.h"

using namespace std;

/* forward decl */
class GameWorld;


/*
 * simple class to store map boundaries, which are not necessarily the same as
 * terrain max and mins
 */
class MapBoundary {
  public:
    /* constructor */
    MapBoundary(): xMax(0), xMin(0), zMax(0), zMin(0) {} 
  
    // the boundary coords
    GLfloat xMax;
    GLfloat xMin;
    GLfloat zMax;
    GLfloat zMin;
};


/* 
 * A renderable object inside the game world, 
 * like a house or car
 */
class GameObject {
  public:
    /* constructor, takes reference to world */
    GameObject(GameWorld& gw);
    
    /* virtual destructor needed for subclassing */
	virtual ~GameObject();
    
    /* updates this object according to time elapsed, by default does nothing */
    virtual void update(Uint32 timeElapsed);
    
    /* 
     * draw this object, subclasses must provide implementation,
     * use glMatrixPush and Pop to isolate transformations, safe
     * to assume MatrixMode is in MODELVIEW
     */
    virtual void draw() = 0;
    
    /* load textures and other things needed to render, by default does nothing */
    virtual void load();
    
    /* update the object's mesh, by default does nothing */
    virtual void updateMesh();
    
    /* reset the object's state, by default does nothing */
    virtual void reset();
    
    /* 
     * check whether this obj can move to the new location, default
     * version checks for map boundaries. subclasses can do special
     * checking
     */
    virtual bool checkNewLoc(Vec3f newLoc);
    
    /* return reference to the world */
    GameWorld& getWorld() { return world; }
    
    /* get x coord of obj */
    GLfloat getX() { return loc.getX(); }
    
    /* get y coord of obj */
    GLfloat getY() { return loc.getY(); }
    
    /* get z coord of obj */
    GLfloat getZ() { return loc.getZ(); }
    
    /* get the object location */
    const Vec3f& getLocation() { return loc; }
    
    /* set location */
    void setLocation(GLfloat xv, GLfloat yv, GLfloat zv) {
        loc.set(xv, yv, zv); 
    }
    
    /* get the velocity */
    Vec3f& getVelocity() { return vel; }
    
    /* set the velocity vector */
    void setVelocity(GLfloat xv, GLfloat yv, GLfloat zv) { 
        vel.set(xv, yv, zv); 
    }
    
    /* get accel */
    const Vec3f& getAcceleration() { return accel; }
    
    /* set the accel vector */
    void setAcceleration(GLfloat xv, GLfloat yv, GLfloat zv) { 
        accel.set(xv, yv, zv); 
    }
        
    /* get angle to Y axis */
    GLfloat getAngleY() { return angleY; }

    /* set angle to Y axis */

    virtual void setAngleY(GLfloat v);

    //bool getParticle(){return particles;}
    //void setParticleFalse(){ particles = false; }

    /* get width of object */
    GLfloat getWidth() { return width; }

    /* get height of object */
    GLfloat getHeight() { return height; }

    /* get length of object */
    GLfloat getLength() { return length; }
    
    /* set the height */
    void setHeight(GLfloat h) { 
        height = h;
        yMin = -height / 2.0;
        yMax = height / 2.0;
    }
    
    /* set the width */
    void setWidth(GLfloat w) {
        width = w;
        xMin = -width / 2.0; 
        xMax = width / 2.0;
    }
        
    /* set the length */
    void setLength(GLfloat l) {
        length = l;
        zMin = -length / 2.0;
        zMax = length / 2.0;
    }

    /* get minimum x value */
    GLfloat getXMin() { return xMin; }
    
    /* set minimum x value */
    void setXMin(GLfloat v) { xMin = v; }

    /* get maximum x value */
    GLfloat getXMax() { return xMax; }

    /* set maximum x value */
    void setXMax(GLfloat v) { xMax = v; }

    /* get minimum y value */
    GLfloat getYMin() { return yMin; }

    /* set minimum y value */
    void setYMin(GLfloat v) { yMin = v; }

    /* get maximum y value */
    GLfloat getYMax() { return yMax; }

    /* set maximum y value */
    void setYMax(GLfloat v) { yMax = v; }

    /* get minimum z value */
    GLfloat getZMin() { return zMin; }

    /* set minimum z value */
    void setZMin(GLfloat v) { zMin = v; }

    /* get maximum z value */
    GLfloat getZMax() { return zMax; }

    /* set maximum z value */
    void setZMax(GLfloat v) { zMax = v; }

    /* get the bounding sphere distance (radius squared) */
    GLfloat getBoundSphereDist();
    
    /* sound effect */
    Mix_Chunk* getSound() { return sound; }
    
    /* sound effect channel */
    int getSoundChannel() { return soundChannel; }
    
    /* play sound effect */
    void playSound();
    
    /* tmp method to draw a box, until we have real models */
    static void drawBox();
    
    /* get the boundaries of the map */
    MapBoundary getMapBoundary() { return mapBoundary; }
    
  protected:
    
    /* reference to the game world */
    GameWorld& world;
    /* orientation of object to y axis */
    GLfloat angleY;
    /* width (x-axis) of object */
    GLfloat width;
    /* height (y-axis) of object */
    GLfloat height;
    /* length (z-axis) of object */
    GLfloat length;
    /* minimum x value */
    GLfloat xMin;
    /* maximum x value */
    GLfloat xMax;
    /* minimum y value */
    GLfloat yMin;
    /* maximum y value */
    GLfloat yMax;
    /* minimum z value */
    GLfloat zMin;
    /* maximum z value */
    GLfloat zMax;
    /* limit for velocity */
    GLfloat topSpeed;
    /* bounding sphere distance */
    GLfloat boundSphereDist;
    
    /* current location */
    Vec3f loc;
    /* current velocity */
    Vec3f vel;
    /* current acceleration */
    Vec3f accel;
    /* current normal to terrain */
    Vec3f normal;
    
    /* sound effect */
    Mix_Chunk* sound;
    
    /* sound effect channel */
    int soundChannel;
   
    /* the boundaries of the map for this object */
    MapBoundary mapBoundary;
    
    /*
     * updates location and velocity according to accel and time elapsed
     * this allows steady motion
     */
    void updateLocAndVel(Uint32 timeElapsedMillis);
    
    /* limits the magnitude of internal velocity vector */
    void limitVelocity();
    
    /* returns wether moving to new location will collide with others */
    bool detectCollision(Vec3f newLoc);
    
    /* get object height and normal according to terrain */
    Vec3f getTerrainHeightAndNormal(GLfloat& objHeight);
    
    /* rotate to match normal (use during draw()) */
    void rotateToNormal();
    
    /* calculate the distance moved, taking terrain into account */
    Vec3f& calcTerrainDist(Vec3f& current, Vec3f& newDist);
    
    /* method to draw debugging info, such as BV and velocity */
    void drawDebugInfo();
        
    /* test whether 2 spheres collide */
    bool spheresCollide(Vec3f loc1, GLfloat r1, Vec3f loc2, GLfloat r2);
    
    /* draw a line sphere, for BV debugging */
    void drawSphere(Vec3f origin, GLfloat r);
    
  private:
    
    /* assignment op not defined for this class */
    GameObject& operator=(const GameObject& rhs);
    /* copy ctor not defined for this class */
    GameObject(const GameObject& rhs);
};

#endif /*GAMEOBJECT_H_*/
