/*
 * Represents an object in the game, like a house or
 * car, keeps track of and updates its own state, knows
 * how to draw itself.  This base class can provide functionality
 * that every object needs, subclass this to provide
 * specialized behavior
 * 
 */
 
#include <cmath>
 
#include <SDL_opengl.h>
#include <SDL.h>
#include <iostream>

#include "GameObject.h"
#include "GameWorld.h"
#include "Truck.h"

using namespace std;

extern bool debugMode;
extern bool musicMode;

/* constructor, takes reference to world */
GameObject::GameObject(GameWorld& gw): 
    world(gw),
    angleY(0),
    width(0), height(0), length(0),
    xMin(0), xMax(0), yMin(0), yMax(0), zMin(0), zMax(0),
    topSpeed(0),
    boundSphereDist(0),
    normal(0, 1, 0),
    sound(NULL),
    soundChannel(-1) {}

/* virtual destructor, needed for subclassing */
GameObject::~GameObject() {
    if (musicMode && soundChannel != -1) {
        Mix_HaltChannel(soundChannel);
        Mix_FreeChunk(sound);
    }
}

/* updates this object according to time elapsed, by default does nothing  */
void GameObject::update(Uint32 timeElapsed) {}

/* default do nothing impl of load */
void GameObject::load() {}

/* default do nothing impl of updateMesh */
void GameObject::updateMesh() {}

/* reset the object's state, by default does nothing */
void GameObject::reset() {}

/* set angle to Y axis */
void GameObject::setAngleY(GLfloat v) { angleY = v; }

/*
 *  updates location and velocity according to accel and time elapsed
 * this allows steady motion
 */
void GameObject::updateLocAndVel(Uint32 timeElapsedMillis) {
    // coast to stop decel rate
    static const GLfloat DECEL_RATE = 20;
    
    // conver to seconds for easier vel and accel values
    GLfloat timeElapsed = timeElapsedMillis / 1000.0f;
    
    // calc distance moved
    Vec3f distMoved(vel);
    distMoved.mult(timeElapsed); // v*t
    Vec3f distFromAccel(accel);
    distFromAccel.mult(timeElapsed * timeElapsed * 0.5f); // .5*a*t^2
    distMoved.add(distFromAccel);
    
    // get the new location, taking into account the terrain
    Vec3f newLoc(loc);
    GLfloat newHeight = 0;
    Vec3f newNormal = getTerrainHeightAndNormal(newHeight);
    newLoc.add(distMoved);
    newLoc.setY(newHeight);
    
    // do Collision Detection, so we don't move into other objs
    bool willCollide = detectCollision(newLoc);
    // do any other special checks to make sure it's ok to move to newLoc
    bool newLocOK = checkNewLoc(newLoc);
    
    if (willCollide || !newLocOK) { 
        // stop if we will collide or new location is not acceptable to obj
        accel.set(0,0,0);
        vel.set(0,0,0);
    } else { 
        // ok to move, update stuff
        
        // update location
        loc = newLoc;
        
        // update normal to match terrain
        normal = newNormal;
        
        // update velocity with accel
        if (accel.magnitude() != 0) {
            Vec3f newVel(accel);
            newVel.mult(timeElapsed); // a*t
            vel.add(newVel);
        } else {
            // coast to stop if no accel
            Vec3f newVel(vel);
            newVel.mult(-1);
            newVel.trim(DECEL_RATE);
            newVel.mult(timeElapsed); // a*t
            vel.add(newVel);
        }
        
        // limit new velocity
        limitVelocity();
    }
}

/* play sound effect */
void GameObject::playSound() {
    if (musicMode && sound != NULL) {
        if (soundChannel != -1) {
            // play sound only if channel is not currently playing
            if (Mix_Playing(soundChannel) == 0) {
                Mix_PlayChannel(soundChannel, sound, 0);
            }
        } else {
            // play sound effect
            soundChannel = Mix_PlayChannel(-1, sound, 0);
            if (soundChannel == -1) {
                cout << "Failed to play sound effect " << Mix_GetError() << endl;
            }
        }
    }
}

/* 
 * check whether this obj can move to the new location, default
 * version checks for map boundaries. subclasses can do special
 * checking
 */
bool GameObject::checkNewLoc(Vec3f newLoc) {
    return  (newLoc.getX() > getMapBoundary().xMin &&
             newLoc.getX() < getMapBoundary().xMax &&
             newLoc.getZ() > getMapBoundary().zMin &&
             newLoc.getZ() < getMapBoundary().zMax);
}

/* calculate the distance moved, taking terrain into account */
Vec3f& GameObject::calcTerrainDist(Vec3f& current, Vec3f& newDist) {
    // rotate newDist to line up with terrain 
    GLfloat newDistLen = newDist.magnitude();
    GLfloat destHeight = world.getTerrain()->getHeight(loc.getX() + newDist.getX(), 
                                                       loc.getZ() + newDist.getZ());
    GLfloat currentHeight = world.getTerrain()->getHeight(loc.getX(), loc.getZ());
    newDist.setY(destHeight - currentHeight);
    newDist.trim(newDistLen);
    return newDist;
}

/* rotate to match normal (use during draw()) */
void GameObject::rotateToNormal() {
    // get angle between Y-axis and normal
    Vec3f yaxis(0, 1, 0);
    GLfloat angle = yaxis.getAngle(normal);
    // get axis of rotation by cross product
    Vec3f rotAxis = yaxis.crossProduct(normal);
    // rotate in opengl
    glRotatef(angle, rotAxis.getX(), rotAxis.getY(), rotAxis.getZ());
}

/* get object height and normal according to terrain */
Vec3f GameObject::getTerrainHeightAndNormal(GLfloat& objHeight) {
    // find vec in dir of AngleY
    Vec3f a(0, 0, -1 * getBoundSphereDist());
    a.rotateY(angleY);
    GLfloat heightA = world.getTerrain()->getHeight(loc.getX() + a.getX(), 
                                                    loc.getZ() + a.getZ());
    Vec3f aneg(a);
    aneg.mult(-1);
    GLfloat heightAneg = world.getTerrain()->getHeight(loc.getX() + aneg.getX(), 
                                                       loc.getZ() + aneg.getZ());                                                
    // figure out obj height
    objHeight = (heightA + heightAneg) / 2.0f;
    
    heightA -= objHeight;
    a.setY(heightA);
    
    Vec3f b(0, 0, -1 * getBoundSphereDist());
    b.rotateY(angleY - 90);
    GLfloat heightB = world.getTerrain()->getHeight(loc.getX() + b.getX(),
                                                    loc.getZ() + b.getZ());
    heightB -= objHeight;
    b.setY(heightB);
    
    // figure out the normal, so draw() can rotate appropriately
    return b.crossProduct(a);
        
}

/* returns wether moving to new location will collide with others */
bool GameObject::detectCollision(Vec3f newLoc) {
    // do CD against all objs for now
    GameObjVec neighbors = world.getCollidableObjs();
    
    GameObjVec::iterator i;
    for (i = neighbors.begin(); i != neighbors.end(); i++) {
        GameObjPtr obj = (*i);
        if (obj != this) { // don't include self
            // sphere/sphere test
            bool collide = spheresCollide(obj->getLocation(), 
                                          obj->getBoundSphereDist(),
                                          newLoc,
                                          this->getBoundSphereDist());
            if (collide) {
                Truck* truck = (Truck*)world.getTruck();
                // special CD for the truck
                if (this == truck) {
                    if (truck->willCollide(newLoc, obj->getLocation(), obj)) {
                        // play object's sound effect
                        obj->playSound();
                        return true;
                    }
                } else if (obj == truck) {
                    if (truck->willCollide(truck->getLocation(), newLoc, this)) {
                        return true;
                    }
                } else {
                    // other obj this coarse test is fine
                    return true;
                }
            }
        }
    }
    return false;    
}

/* test whether 2 spheres collide */
bool GameObject::spheresCollide(Vec3f loc1, GLfloat r1, Vec3f loc2, GLfloat r2) {
    loc1.setY(0); // flatten 
    loc2.setY(0);
    Vec3f dist(loc1);
    dist.sub(loc2);
    GLfloat safeDist = r1 + r2;
    return (dist.magnitude() < safeDist);
}

/* limits the magnitude of internal velocity vector */
void GameObject::limitVelocity() {
    vel.trim(topSpeed);
}

/* get the bounding sphere distance (radius squared) */
GLfloat GameObject::getBoundSphereDist() {
    if (boundSphereDist == 0) {
        // bounding sphere distance may not have been calculated yet
        Vec3f corner(width / 2.0f, height / 2.0f, length / 2.0f);
        boundSphereDist = corner.magnitude();
    }
    return boundSphereDist;
}

/* method to draw debugging info, such as BV and velocity */
void GameObject::drawDebugInfo() {
    if (debugMode) {
        // draw vel
        glDisable(GL_LIGHTING);
        glBegin(GL_LINES);
            glColor3f(1, 0, 0);
            glVertex3f(0, 0, 0);
            Vec3f a(vel);
            a.mult(10);
            glVertex3f(a.getX(), a.getY(), a.getZ());
            
            // draw accel
            glVertex3f(0,0,0);
            a = accel;
            a.mult(10);
            glVertex3f(a.getX(), a.getY(), a.getZ());
            
            // draw normal
            glVertex3f(0, 0, 0);
            a = normal;
            a.mult(10);
            glVertex3f(a.getX(), a.getY(), a.getZ());
        glEnd();
        glEnable(GL_LIGHTING);
        
        // draw BV if not truck
        if (this != world.getTruck()) {
            glColor3f(0, 0, 1);
            drawSphere(Vec3f(), getBoundSphereDist());
        }
    }
}

/* draw a line sphere, for BV debugging */
void GameObject::drawSphere(Vec3f origin, GLfloat r) {
    glPushMatrix();
    glDisable(GL_LIGHTING);
    glTranslatef( origin.getX(), origin.getY(), origin.getZ());
    for (int j=0; j < 180; j += 30) {
        glPushMatrix();
        glRotatef(j, 0, 0, -1);
        glBegin(GL_LINE_LOOP);
        for (int i=0; i < 360; i += 10) {
            GLfloat x = r * sin(degToRad(i));
            GLfloat z = r * cos(degToRad(i));
            glVertex3f(x, 0, z);
        }
        glEnd();
        glPopMatrix();
    }
    glEnable(GL_LIGHTING);
    glPopMatrix();
}

/* tmp method to draw a box, until we have real models */
void GameObject::drawBox() {
    // isolate transformations specific to this box
    glPushMatrix();
    
    // move box above ground
    glTranslatef( 0, 1, 0);
    
    // mostly copied from demo, draw box at origin
    glBegin(GL_QUADS);
        glColor3f(1, 1, 1);
        GLfloat white[] = { 1.0, 1.0, 1.0, 1.0 };
        glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, white);
        glNormal3f( 0,  0,  1);
        glVertex3f(-1, -1,  1);
        glVertex3f( 1, -1,  1);
        glVertex3f( 1,  1,  1);
        glVertex3f(-1,  1,  1);
        
        GLfloat blue[] = { 0.4, 0.4, 1.0, 1.0 };
        glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, blue);
        glNormal3f( 0,  0, -1);
        glVertex3f(-1, -1, -1);
        glVertex3f(-1,  1, -1);
        glVertex3f( 1,  1, -1);
        glVertex3f( 1, -1, -1);
        
        GLfloat green[] = { 0.4, 1.0, 0.4, 1.0 };
        glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, green);
        glNormal3f( 1,  0,  0);
        glVertex3f( 1, -1, -1);
        glVertex3f( 1,  1, -1);
        glVertex3f( 1,  1,  1);
        glVertex3f( 1, -1,  1);

        glNormal3f(-1,  0,  0);
        glVertex3f(-1, -1, -1);
        glVertex3f(-1, -1,  1);
        glVertex3f(-1,  1,  1);
        glVertex3f(-1,  1, -1);
        
        GLfloat red[] = { 1.0, 0.4, 0.4, 1.0 };
        glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, red);
        glNormal3f( 0,  1,  0);
        glVertex3f(-1,  1, -1);
        glVertex3f(-1,  1,  1);
        glVertex3f( 1,  1,  1);
        glVertex3f( 1,  1, -1);
        
        glNormal3f( 0, -1,  0);
        glVertex3f(-1, -1, -1);
        glVertex3f(-1, -1,  1);
        glVertex3f( 1, -1,  1);
        glVertex3f( 1, -1, -1);
    glEnd();
    
    // return stack to its previous state
    glPopMatrix(); 
}
