/*
 * Represents the game loop, constant loop of updating
 * game world and handling input events, with frame 
 * throttling as necessary
 */
#include <SDL.h>
#include <cstdlib>
#include <iostream>

#include <SDL_ttf.h>
#include <SDL_mixer.h>

#include "GameLoop.h"

extern bool musicMode;
extern bool statsMode;

/* construct with reference to a game world */
GameLoop::GameLoop(GameWorld& gw): 
    gameWorld(gw), 
    endLoop(false),
    music(NULL),
    musicChannel(-1) {
    if (musicMode) {
        // load audio
        music = Mix_LoadWAV("sounds/music.aif");
        if (music == NULL) {
            cout << "Failed to load audio " << Mix_GetError() << endl;
        }
    }    
}

/* destructor */
GameLoop::~GameLoop() {
    if (musicMode) {
        // free music chunk
        Mix_FreeChunk(music);
    }
}

/* start the loop, does not return until program ends */
void GameLoop::start() {
    if (musicMode && music != NULL) {
        if (musicChannel != -1) {
            // play audio only if channel is not currently playing
            if (Mix_Playing(musicChannel) == 0) {
                Mix_PlayChannel(musicChannel, music, 0);
            }
        } else {
            // start playing audio
            musicChannel = Mix_PlayChannel(-1, music, -1);
            if (musicChannel == -1) {
                cout << "Failed to play audio " << Mix_GetError() << endl;
            }
        }
    }
    
    // keep track of some performance stats
    Uint32 lastStatOutput(0);
    Uint32 updTime(0);
    Uint32 updCnt(0);
    Uint32 drawTime(0);
    Uint32 drawCnt(0);
    Uint32 droppedFrames(0);
    
    // init lastTime, nextTime to now at start of loop
    Uint32 nextTime = SDL_GetTicks();
    Uint32 lastTime = SDL_GetTicks();
    bool updateWorld = false;
    
    while (!endLoop) {
        // get current time;
        Uint32 currentTime = SDL_GetTicks();
        
        // process all input events in queue
        gameWorld.handleInputEvents();
       
        // update world
        if (updateWorld) {
            if (gameWorld.getGameIsRunning()) {
                Uint32 start = SDL_GetTicks(); //keep some stats
                gameWorld.updateWorld(currentTime - lastTime);
                updTime += SDL_GetTicks() - start;
                updCnt++;
                
                // flip update flag, note last time updated
                updateWorld = false;
            }
            lastTime = currentTime;
        }
            
        
        // see if it's time to redraw world
        if (currentTime > nextTime) {
            
            // yes, render the frame
            Uint32 start = SDL_GetTicks(); //keep some stats
            gameWorld.drawWorld();
            drawTime += SDL_GetTicks() - start;
            drawCnt++;
            
            // set time for next frame
            nextTime += TIME_PER_FRAME;
    
            // drop frames if we get too far behind
            if (nextTime < currentTime) {
                // reset nextTime
                nextTime = currentTime + TIME_PER_FRAME;
                droppedFrames++;
            }
            
            // flag that we need to update world again
            updateWorld = true;
            
            // log any GL errors just in case
            GLenum errCode;
            if ((errCode = glGetError()) != GL_NO_ERROR) {
                cout << "OpenGL Error! " << gluErrorString(errCode) << endl;
            }
            
        } else {
            // let other threads run before looping again?
            SDL_Delay(1);
        }
        
        // print some stats in statsMode, to gauge performance
        if (statsMode) {
            static Uint32 statsInterval = 5 * 1000;
            if (SDL_GetTicks() - lastStatOutput > statsInterval) {
                cout << "Avg Upd Time: " << (float)updTime / (float)updCnt;
                cout << " Avg Draw Time: " << (float)drawTime / (float)drawCnt;
                cout << " Frames " << drawCnt;
                cout << " Dropped " << droppedFrames << endl;
                // reset counters
                lastStatOutput = SDL_GetTicks();
                updTime = 0;
                updCnt = 0;
                drawTime = 0;
                drawCnt = 0;
                droppedFrames = 0;
            }
        }
    }
}

/* end the loop and exit program */
void GameLoop::exit() {
    endLoop = true;
    if (musicMode) {
        // quit audio
        Mix_HaltChannel(-1);
        Mix_CloseAudio();
    }
    // quit fonts
    TTF_Quit();
    // quit sdl
    SDL_Quit();
    // exit here?
    std::exit(0);
}

/* storage for static contants */
const Uint32 GameLoop::TIME_PER_FRAME;
