/*
 * Represents the view frustum, defined by the six clipping planes:
 * left, right, top, bottom, near far. This is used to determine whether
 * objects intersect the current view frustum.
 */
 
#ifndef FRUSTUM_H_
#define FRUSTUM_H_

#include <SDL_opengl.h>
#include <SDL.h>

#include "Vec3f.h"
#include "GameObject.h"

#define FRUSTUM_OFFSET 10

struct Plane {
    float a, b, c, d;
};

enum ClippingPlane { LEFT_PLANE, RIGHT_PLANE, TOP_PLANE, BOTTOM_PLANE, NEAR_PLANE, FAR_PLANE, NOT_PLANE };
enum FrustumPosition { OUTSIDE_FRUSTUM = -1, INTERSECT_FRUSTUM = 0, INSIDE_FRUSTUM = 1 };

/* view frustum */
class Frustum {
  public:
    /* default constructor */
	Frustum();
    
    /* destructor */
    ~Frustum();
    
    /* calculates the clipping planes of the frustum from the current transformation matrices */
    void calculatePlanes();
    
    /* normalizes the clipping plane coefficients to unit distance */
    void normalizePlanes();
    
    /* normalize specified clipping plane */
    void normalizePlane(ClippingPlane plane);
    
    /* set the location */
    void setLocation(Vec3f v) { location = v; }
    
    /* get the location */
    Vec3f getLocation() { return location; }

    /* get left clipping plane */
    Plane getLeftPlane() { return planes[LEFT_PLANE]; }
    
    /* get right clipping plane */
    Plane getRightPlane() { return planes[RIGHT_PLANE]; }
    
    /* get top clipping plane */
    Plane getTopPlane() { return planes[TOP_PLANE]; }
    
    /* get bottom clipping plane */
    Plane getBottomPlane() { return planes[BOTTOM_PLANE]; }
    
    /* get near clipping plane */
    Plane getNearPlane() { return planes[NEAR_PLANE]; }
    
    /* get far clipping plane */
    Plane getFarPlane() { return planes[FAR_PLANE]; }

    /* returns the distance from the given point to the specified clipping plane */
    float pointDistanceToPlane(Vec3f point, ClippingPlane plane);

    /* returns the position of the given point relative to the specified clipping plane */
    FrustumPosition pointPositionToPlane(Vec3f point, ClippingPlane plane);

    /* returns the position of the bounding sphere with the given center and distance
     * (squared radius) relative to this frustum */
    FrustumPosition sphereIntersects(Vec3f center, GLfloat distance);

    /* returns the position of the given GameObject relative to this frustum */
    FrustumPosition objectIntersects(GameObject* obj);
    
  private:
    /* clipping planes */
    Plane planes[6];
    
    /* location of the frustum (camera's view position) */
    Vec3f location;
};

// helper function to iterate through clipping planes
inline ClippingPlane& operator++(ClippingPlane& p) {
    p = ClippingPlane(p + 1);
    return p;
}

#endif /*FRUSTUM_H_*/
