/*
 * Frustum implementation represents the view frustum, defined by the six
 * clipping planes: left, right, top, bottom, near far. This is used to
 * determine whether objects intersect the current view frustum.
 */

#include <cmath>
#include <iostream>

#include "Frustum.h"

using namespace std;
    
/* default constructor */
Frustum::Frustum() {}

/* destructor */
Frustum::~Frustum() {}

/* calculates the clipping planes of the frustum from the current transformation matrices */
void Frustum::calculatePlanes() {
    // isolate transformations
    glPushMatrix();
    
    // get combined modelview and projection matrix
    GLfloat modelview[16];
    GLfloat projection[16];
    GLfloat frustum[16];
    
    glGetFloatv(GL_MODELVIEW_MATRIX, modelview);
    glGetFloatv(GL_PROJECTION_MATRIX, projection);
    glLoadMatrixf(projection);
    glMultMatrixf(modelview);
    glGetFloatv(GL_MODELVIEW_MATRIX, frustum);
    
    // left clipping plane
    planes[LEFT_PLANE].a = frustum[3] + frustum[0];
    planes[LEFT_PLANE].b = frustum[7] + frustum[4];
    planes[LEFT_PLANE].c = frustum[11] + frustum[8];
    planes[LEFT_PLANE].d = frustum[15] + frustum[12];
    
    // right clipping plane
    planes[RIGHT_PLANE].a = frustum[3] - frustum[0];
    planes[RIGHT_PLANE].b = frustum[7] - frustum[4];
    planes[RIGHT_PLANE].c = frustum[11] - frustum[8];
    planes[RIGHT_PLANE].d = frustum[15] - frustum[12];
    
    // top clipping plane
    planes[TOP_PLANE].a = frustum[3] - frustum[1];
    planes[TOP_PLANE].b = frustum[7] - frustum[5];
    planes[TOP_PLANE].c = frustum[11] - frustum[9];
    planes[TOP_PLANE].d = frustum[15] - frustum[13];
    
    // bottom clipping plane
    planes[BOTTOM_PLANE].a = frustum[3] + frustum[1];
    planes[BOTTOM_PLANE].b = frustum[7] + frustum[5];
    planes[BOTTOM_PLANE].c = frustum[11] + frustum[9];
    planes[BOTTOM_PLANE].d = frustum[15] + frustum[13];
    
    // near clipping plane
    planes[NEAR_PLANE].a = frustum[3] + frustum[2];
    planes[NEAR_PLANE].b = frustum[7] + frustum[6];
    planes[NEAR_PLANE].c = frustum[11] + frustum[10];
    planes[NEAR_PLANE].d = frustum[15] + frustum[14];
    
    // far clipping plane
    planes[FAR_PLANE].a = frustum[3] - frustum[2];
    planes[FAR_PLANE].b = frustum[7] - frustum[6];
    planes[FAR_PLANE].c = frustum[11] - frustum[10];
    planes[FAR_PLANE].d = frustum[15] - frustum[14];
    
    // normalize planes
    normalizePlanes();
    
    // return stack to its previous state
    glPopMatrix();
}

/* normalizes the clipping plane coefficients to unit distance */
void Frustum::normalizePlanes() {
    // normalize each clipping plane
    for (ClippingPlane p = LEFT_PLANE; p <= FAR_PLANE; ++p) {
        normalizePlane(p);
    }
}

/* normalize specified clipping plane */
void Frustum::normalizePlane(ClippingPlane plane) {
    Plane p = planes[plane];
    // divide each plane coefficient by the length (magnitude)
    float length = sqrt(p.a * p.a + p.b * p.b + p.c * p.c);
    if (length != 0) {
        p.a /= length;
        p.b /= length;
        p.c /= length;
        p.d /= length;
    }
}

/* returns the distance from the given point to the specified clipping plane */
float Frustum::pointDistanceToPlane(Vec3f point, ClippingPlane plane) {
    Plane p = planes[plane];
    return p.a * point.getX() + p.b * point.getY() + p.c * point.getZ() + p.d;
}

/* returns the position of the given point relative to the specified clipping plane */
FrustumPosition Frustum::pointPositionToPlane(Vec3f point, ClippingPlane plane) {
    float d = pointDistanceToPlane(point, plane);
    if (d < 0) {
        // if distance from point to plane is negative, point lies outside plane
        return OUTSIDE_FRUSTUM;
    } else if (d > 0) {
        // if distance from point to plane is positive, point lies inside plane
        return INSIDE_FRUSTUM;
    } else {
        // otherwise the point intersects the plane
        return INTERSECT_FRUSTUM;
    }
}

/* returns the position of the bounding sphere with the given center and distance
 * relative to this frustum */
FrustumPosition Frustum::sphereIntersects(Vec3f center, GLfloat distance) {
    //return INSIDE_FRUSTUM;
    float d;
    for (ClippingPlane p = LEFT_PLANE; p <= FAR_PLANE; ++p) {
        // calculate distance from sphere center to each plane
        d = pointDistanceToPlane(center, p);
        
        if (d < -distance) {
            // if the distance is less than the negative radius of the sphere, the sphere is outside
            return OUTSIDE_FRUSTUM;
        } else if (fabs(d) < distance) {
            // if the distance is between the absolute value of the radius, the sphere is intersecting
            return INTERSECT_FRUSTUM;
        }
    }
    
    // the sphere is neither outside or intersecting the frustum, so it is fully inside
    return INSIDE_FRUSTUM;
}

/* returns the position of the given GameObject relative to this frustum */
FrustumPosition Frustum::objectIntersects(GameObject* obj) {
    // test whether the object's bounding sphere intersects this frustum
    FrustumPosition pos = sphereIntersects(obj->getLocation(), obj->getBoundSphereDist() + FRUSTUM_OFFSET);
    return pos;
}
