/*
 * Classes to generate realistic trees
 * using fractal methods
 * 
 */
#ifndef FRACTALTREE_H_
#define FRACTALTREE_H_

#include <vector>
#include <iostream>

#include <SDL_opengl.h>
#include <SDL.h>

#include "GameObject.h"
#include "Vec3f.h"


/* a vector of FT nodes */
class FTNode;
typedef std::vector< FTNode* > FTNodeVec;

/* a geometric transformation spec for stems or branches */
class Transform {
  public:
    /* constructor */
    Transform(): 
        rotZ(0),
        rotZDev(0),
        rotY(0),
        rotYDev(0),
        scale(0),
        scaleDev(0),
        taper(0) {}
    
    /* destructor */
    virtual ~Transform() {}
  
    /* apply transform to FTNode, using parent as guide */
    void apply(FTNode* parent, FTNode* node);
  
    /* get a random float between 0 and 1 */
    static GLfloat random();
  
    /* get a random sign, +-1 */
    static GLfloat randSign();
    
    /* get a random value of mean +- random()*deviation */
    static GLfloat randVal(GLfloat mean, GLfloat dev);
  
    //--------- public fields -----------------
    /* rotation */
    GLfloat rotZ;
    GLfloat rotZDev;
    GLfloat rotY;
    GLfloat rotYDev;
    /* scale */
    GLfloat scale;
    GLfloat scaleDev;
    /* radius taper */
    GLfloat taper;
    
};

class FractalTree;

/* a single node in the tree */
class FTNode {
  public:
    /* the branch number for "stems" */
    static const int STEM_NUM = -1;
    
    /* constructor */
    FTNode(FractalTree* t, 
           FTNode* p, 
           Transform& sTrans, 
           Transform& bTrans,
           int lev, 
           int bNum);
    
    /* destructor */
    virtual ~FTNode();
    
    /* get the parent node */
    FTNode* getParent() { return parent; }
    
    /* get the child nodes */
    FTNodeVec& getChildNodes() { return childNodes; }
    
    /* get the start point of node branch */
    Vec3f& getStart() { return start; }
    
    /* get the start point of node branch */
    void setStart(Vec3f v) { start = v; }
    
    /* get the end point of node branch */
    Vec3f& getEnd() { return end; }
    
    /* get the end point of node branch */
    void setEnd(Vec3f v) { end = v; }
    
    /* get the stem transform */
    Transform& getStemTrans() { return stemTrans; }
    
    /* get the branch transform */
    Transform& getBranchTrans() { return branchTrans; }
    
    /* get the recursion level */
    int getLevel() { return level; }
    
    /* get the branch number */
    int getBranchNum() { return branchNum; }
    
    /* grow this node, recursively until minBranchSize reached */
    void grow();
    
    /* get the size of this node */
    GLfloat FTNode::size();
    
    /* get the vector segment (end-start) that represents this node */
    Vec3f& getSegment() { return segment; }
    
    /* recalc the vector segment (end-start) that represents this node */
    void recalcSegment();
    
    /* get the radius at end of this node */
    GLfloat getRadius() { return radius; }
    
    /* get the parent ending radius */
    GLfloat getParentRadius() const;
    
    /* get if this node is a leaf */
    bool getIsLeaf() { return isLeaf; }
    
  private:
    /* ptr to tree */
    FractalTree* tree;
    /* ptr to parent node */
    FTNode* parent;
    /* children nodes */
    FTNodeVec childNodes;
    /* current start point of this branch */
    Vec3f start;
    /* current end point of this branch */
    Vec3f end;
    /* branch segment, end - start */
    Vec3f segment;
    /* is node a leaf */
    bool isLeaf;
    /* transfrom for the stem */
    Transform stemTrans;
    /* transform for the branch */
    Transform branchTrans;
    /* recursive level */
    int level;
    /* branch number */
    int branchNum;
    /* radius at node end */
    GLfloat radius;
    
  /* printing  */  
  friend std::ostream& operator<<(std::ostream& s, const FTNode& n);
  
};


/* An entire tree, to be rendered in the game */
class FractalTree: public GameObject {
  public:
    /* constructor */
	FractalTree(GameWorld& gw);
    
    /* destructor */
	virtual ~FractalTree();
    
    /* draw this tree */
    virtual void draw();
    
    /* get minimum branch size */
    GLfloat getMinBranchSize() { return minBranchSize; }
    
    /* get the number of branches */
    int getNumBranches() { return numBranches; }
    
    /* create the tree */
    void create();
    
    /* get root node of tree */
    FTNode* getRoot() { return root; }
    
    /* build display list to render this tree */
    GLuint buildDisplayList();
    
    /* total segments (nodes) in tree */
    int getTotalNodes() { return totalNodes; }
    
    /* increment totalNodes */
    void incrementTotalNodes() { totalNodes++; }
    
    /* render this node and child nodes, branches only */
    void renderBranches(FTNode* node, int& count);
    
    /* render this node and child nodes, leaves only */
    void renderLeaves(FTNode* node, int& count);
    
    /* adjust the extent (max size) of this tree */
    void adjustMaxExtent(Vec3f vert);
    
    /* get the radius of max extent */
    GLfloat getMaxExtentRadius();
    
    /* get the current display lod level */
    GLuint getDisplayLod();
    
    /* get a shared leaf texture */
    static GLuint getLeafTexture();
    
    /* get a shared bark texture */
    static GLuint getBarkTexture();
    
    
  private:
    /* the root node */
    FTNode* root;
    /* the minimum branch size before it turns leaf */
    GLfloat minBranchSize;
    /* the number of branches */
    int numBranches;
    /* total segments (nodes) in tree */
    int totalNodes;
    /* display list id */
    vector<GLuint> displayLists;
    /* root radius */
    GLfloat rootRadius;
    /* max extent of entire tree*/
    Vec3f maxExtent;
    /* lod level */
    int lodLevel;
    
  friend class FTNode;
};

/* printing Tree */
std::ostream& operator<<(std::ostream& s, const FTNode& n);


#endif /*FRACTALTREE_H_*/
