/*
 * implementation of FractalTree classes
 */
#include <iostream>
#include <string>
#include <cmath>

#include "FractalTree.h"
#include "GameWorld.h"
#include "Mesh.h"
#include "Frustum.h"

using namespace std;

extern bool debugMode;
extern bool statsMode;

/* get a random float between 0 and 1 */
GLfloat Transform::random() { 
    return (GLfloat)((double)std::rand() / (double)RAND_MAX); 
}

/* get a random sign, +-1 */
GLfloat Transform::randSign() {
    return (random() > 0.5) ? 1 : -1;
}

/* get a random value of mean +- random()*deviation */
GLfloat Transform::randVal(GLfloat mean, GLfloat dev) {
    return mean + (randSign() * random() * dev);
}

/* apply transform to FTNode, using parent as guide */
void Transform::apply(FTNode* parent, FTNode* node) {
    if (parent != NULL) {
        // set start point
        if (node->getBranchNum() == FTNode::STEM_NUM) {
            // start at parent endpoint if stem
            node->setStart(parent->getEnd());
        } else { 
            // start branch somewhere
            Vec3f branchLoc(parent->getEnd());
            node->setStart(branchLoc);
        }
        
        // copy the parent segment to transfrom
        Vec3f mySegment(parent->getSegment());
        
        // scale
        mySegment.mult( randVal(scale, scaleDev) );
            
        // rotate z
        mySegment.rotateZ( randVal( rotZ, rotZDev) );
        
        // rotate y
        mySegment.rotateY( randVal( rotY, rotYDev) );
        
        // set node endpoint, recalc node segement
        node->setEnd(node->getStart());
        node->getEnd().add( mySegment );
        node->recalcSegment();
    }
}

/* constructor */
FTNode::FTNode(FractalTree* t, FTNode* p, 
               Transform& sTrans, Transform& bTrans,
               int lev, int bNum): 
    tree(t), 
    parent(p),
    isLeaf(false),
    stemTrans(sTrans),
    branchTrans(bTrans),
    level(lev),
    branchNum(bNum),
    radius(0)
{
    // keep track of total nodes created
    tree->incrementTotalNodes();
}

/* destructor */
FTNode::~FTNode() {
    // delete children nodes
    for_each(childNodes.begin(), childNodes.end(), DeleteObject());
}

/* the branch number for "stems" */
const int FTNode::STEM_NUM;

/* grow this node, recursively until minBranchSize reached */
void FTNode::grow() {
    if (size() < tree->getMinBranchSize()) {
        // turn this node into leaf if too small
        isLeaf = true;
    } else {
        // create new stem
        FTNode* newStem = 
            new FTNode(tree, this, stemTrans, branchTrans, level + 1, STEM_NUM);
        stemTrans.apply(this, newStem);
        newStem->grow();
        childNodes.push_back(newStem);
        
        // create new branches
        for (int i=0; i < tree->getNumBranches(); i++) {
            FTNode* newBranch = 
                new FTNode(tree, this, stemTrans, branchTrans, level + 1, i);
            branchTrans.apply(this, newBranch);
            newBranch->grow();
            childNodes.push_back(newBranch);
        }
    }
}

/* get the size of this node */
GLfloat FTNode::size() {
    return segment.magnitude();
}

/* helper fn to lin interp */
GLfloat interp(GLfloat a, GLfloat b, GLfloat fraction) {
    return a * (1 - fraction) + b * fraction;
}

/* recalc the vector segment (end-start) that represents this node */
void FTNode::recalcSegment() {
    segment = end;
    segment.sub(start);
    // recalc radius too
    GLfloat taper = stemTrans.taper;
    if (getBranchNum() != STEM_NUM) {
        taper = branchTrans.taper;
    }
    GLfloat pr = getParentRadius();
    radius = pr * taper; //interp(pr, taper * pr, size());
}

/* get the parent ending radius */
GLfloat FTNode::getParentRadius() const {
    GLfloat rv = (parent == NULL) ? tree->rootRadius : parent->getRadius();
    return rv;
}


/* render leaves this node and child nodes */
void FractalTree::renderLeaves(FTNode* node, int& count) {
    // draw node
    if (node->getIsLeaf()) {
        count++; //keep a count of how many drawn
        // draw leaf, simple diamond for now
        // width/len ratio
        GLfloat widthRatio = 0.4;
        GLfloat leafScale = 2.5;
        
        // calc segment
        Vec3f segment(node->getSegment());
        segment.mult(leafScale); // make leaves bigger
        Vec3f end(node->getStart());
        end.add(segment);
        
        Vec3f mid(segment);
        mid.mult(0.5);
        mid.add(node->getStart());
        
        Vec3f midRight(segment);
        midRight.setY(0); // flatten
        midRight.mult(widthRatio);
        midRight.rotateY(segment.getAngleYaxis() - 90);
        midRight.add(mid);
        
        Vec3f midLeft(segment);
        midLeft.setY(0); // flatten
        midLeft.mult(widthRatio);
        midLeft.rotateY(segment.getAngleYaxis() + 90);
        midLeft.add(mid);
        
        Vec3f v0(end);
        Vec3f v1(midLeft);
        Vec3f v2(node->getStart());
        Vec3f v3(midRight);
        
        Vec3f cv(midRight);
        cv.sub(mid);
        Vec3f n0 = mid.crossProduct(segment);
        
        // draw points in opengl
        glNormal3f(n0.getX(), n0.getY(), n0.getZ());
        
        glTexCoord2f(0, 0);
        glVertex3f(v0.getX(), v0.getY(), v0.getZ());
        
        glTexCoord2f(1, 0);
        glVertex3f(v1.getX(), v1.getY(), v1.getZ());
        
        glTexCoord2f(1, 1);
        glVertex3f(v2.getX(), v2.getY(), v2.getZ());
        
        glTexCoord2f(0, 1);
        glVertex3f(v3.getX(), v3.getY(), v3.getZ());
        
        // draw reverse side
        n0.mult(-1);
        glNormal3f(n0.getX(), n0.getY(), n0.getZ());
        
        glTexCoord2f(0, 0);
        glVertex3f(v0.getX(), v0.getY(), v0.getZ());
        
        glTexCoord2f(1, 0);
        glVertex3f(v3.getX(), v3.getY(), v3.getZ());
        
        glTexCoord2f(1, 1);
        glVertex3f(v2.getX(), v2.getY(), v2.getZ());
        
        glTexCoord2f(0, 1);
        glVertex3f(v1.getX(), v1.getY(), v1.getZ());
        
        //adjust max extent
        adjustMaxExtent(v0);
        adjustMaxExtent(v1);
        adjustMaxExtent(v2);
        adjustMaxExtent(v3);
    }
    
    // draw children
    FTNodeVec& childNodes = node->getChildNodes();
    for (FTNodeVec::size_type i = 0; i < childNodes.size(); i++) {
        renderLeaves(childNodes[i], count);
    }
}

/* render branches for this node and child nodes */
void FractalTree::renderBranches(FTNode* node, int& count) {
    // lod stuff
    GLfloat minSize = 0;
    int sides = 5; // polygons to approx cyl
    if (lodLevel > 0) {
        minSize = 0.15;
        sides = 4;
    }
    if (lodLevel > 1) {
        minSize = 0.5;
        sides = 3;
    }
    
    // draw node
    if (!node->getIsLeaf() && node->size() > minSize) {
        count++; //keep a count of how many drawn
        // draw branch
        GLfloat r01 = node->getParentRadius();
        GLfloat r23 = node->getRadius();
        
    
        int angle = (360 / sides);
        for (int i = 0; i < 360; i += angle) {
            GLfloat x, z;
            
            x = r01 * cos(degToRad(i));
            z = -r01 * sin(degToRad(i));
            Vec3f v0(x, 0, z);
            Vec3f n0(x, 0, z);
            
            x = r01 * cos(degToRad(i + angle));
            z = -r01 * sin(degToRad(i + angle));
            Vec3f v1(x, 0, z);
            Vec3f n1(x, 0, z);
            
            x = r23 * cos(degToRad(i + angle));
            z = -r23 * sin(degToRad(i + angle));
            Vec3f v2(x, 0, z);
            Vec3f n2(x, 0, z);
            
            x = r23 * cos(degToRad(i));
            z = -r23 * sin(degToRad(i));
            Vec3f v3(x, 0, z);
            Vec3f n3(x, 0, z);
            
            // transform points as nec
            v0.add(node->getStart());
            v1.add(node->getStart());
            v2.add(node->getEnd());
            v3.add(node->getEnd());
            n0.normalize();
            n1.normalize();
            n2.normalize();
            n3.normalize();
            
            // draw points in opengl
            glTexCoord2f((GLfloat)i / 360.0, 0);
            glNormal3f(n0.getX(), n0.getY(), n0.getZ());
            glVertex3f(v0.getX(), v0.getY(), v0.getZ());
            
            glTexCoord2f((GLfloat)(i + angle) / 360.0, 0);
            glNormal3f(n1.getX(), n1.getY(), n1.getZ());
            glVertex3f(v1.getX(), v1.getY(), v1.getZ());
            
            glTexCoord2f((GLfloat)(i + angle) / 360.0, 1);
            glNormal3f(n2.getX(), n2.getY(), n2.getZ());
            glVertex3f(v2.getX(), v2.getY(), v2.getZ());
            
            glTexCoord2f((GLfloat)i / 360.0, 1);
            glNormal3f(n3.getX(), n3.getY(), n3.getZ());
            glVertex3f(v3.getX(), v3.getY(), v3.getZ());
            
            //adjust max extent
            adjustMaxExtent(v0);
            adjustMaxExtent(v1);
            adjustMaxExtent(v2);
            adjustMaxExtent(v3);
        }
    }
    
    // draw children
    FTNodeVec& childNodes = node->getChildNodes();
    for (FTNodeVec::size_type i = 0; i < childNodes.size(); i++) {
        renderBranches(childNodes[i], count);
    }
}

/* adjust the extent (max size) of this tree */
void FractalTree::adjustMaxExtent(Vec3f vert) {
    Vec3f newExtent(vert);
    newExtent.sub(loc);
    if (newExtent.magnitude() > maxExtent.magnitude()) {
        maxExtent = newExtent;
    }    
}

/* get the radius of max extent */
GLfloat FractalTree::getMaxExtentRadius() {
//    return sqrt((maxExtent.getX() * maxExtent.getX()) + 
//                (maxExtent.getZ() * maxExtent.getZ()));
    return maxExtent.magnitude();
}

/* build display list to render this tree */
GLuint FractalTree::buildDisplayList() {
    int totBranches = 0, totLeaves = 0 ; //keep some stats
    GLuint displayListId = glGenLists(1);
    glNewList(displayListId, GL_COMPILE_AND_EXECUTE);
    
    GLfloat color[] = {1, 1, 1}; 
    glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
    glEnable(GL_TEXTURE_2D);
    
    // render all branches
    glBindTexture(GL_TEXTURE_2D, getBarkTexture());
    glBegin(GL_QUADS);
    renderBranches(root, totBranches);
    glEnd();
    
    // render all leaves
    glBindTexture(GL_TEXTURE_2D, getLeafTexture());
    glBegin(GL_QUADS);
    renderLeaves(root, totLeaves);
    glEnd();
    glDisable(GL_TEXTURE_2D);
    
    glEndList();
    
    if (statsMode) {
        cout << this << " Branches: " << totBranches << " Leaves: " << totLeaves << endl;
        //cout << "MaxExtent: " << maxExtent.magnitude() << " " << getMaxExtentRadius() << endl;
    }
    return displayListId;
}

/* draw this tree */
void FractalTree::draw() {
    // debug info
    if (debugMode) {
        glPushMatrix();
        glTranslatef( loc.getX(), loc.getY(), loc.getZ());
        drawDebugInfo();
        glPopMatrix();
        // draw extent sphere
        glColor3f(0, 1, 0);
        drawSphere(loc, getMaxExtentRadius());
    }

    if (world.getViewFrustum()->sphereIntersects(getLocation(), getMaxExtentRadius()) > OUTSIDE_FRUSTUM) {
        glCallList(getDisplayLod());
    }
        
}

/* get the current display lod level */
GLuint FractalTree::getDisplayLod() {
    Frustum* frustum = world.getViewFrustum();    
    Vec3f eyeLoc(frustum->getLocation());
    eyeLoc.sub(loc);
    GLfloat dist = eyeLoc.magnitude();
    
    unsigned int level = 0;
    if (dist < 23) {
        level = 0;
    } else if (dist < 33) {
        level = 1;
    } else {
        level = 2;
    }
    
    if (level > displayLists.size() - 1) {
        level = displayLists.size() - 1;
    }
    return displayLists[level];
}

/* create the tree */
void FractalTree::create() {
    // test tree
    minBranchSize = 0.1;
    numBranches = 2;
    rootRadius = 0.15;
    
    Transform stemT;
    stemT.rotZ = 0;
    stemT.rotZDev = 60;
    stemT.rotY = 0;
    stemT.rotYDev = 60;
    stemT.scale = .8;
    stemT.scaleDev = 0;
    stemT.taper = .8;
    
    Transform branchT;
    branchT.rotZ = 0;
    branchT.rotZDev = 60;
    branchT.rotY = 0;
    branchT.rotYDev = 180;
    branchT.scale = 0.6;
    branchT.scaleDev = 0.1;
    branchT.taper = .6;
    
    GLfloat startHeight = loc.getY() + Transform::randVal(1.5, 0.5);
 
    root = new FTNode(this, NULL, stemT, branchT, 0, FTNode::STEM_NUM);
    root->setStart(Vec3f(loc.getX(), loc.getY(), loc.getZ()));
    root->setEnd(Vec3f(loc.getX(), startHeight, loc.getZ()));
    root->recalcSegment();
    root->grow();
    
    // build LOD models
    lodLevel = 0;
    displayLists.push_back( buildDisplayList() );
    lodLevel = 1;
    displayLists.push_back( buildDisplayList() );
    lodLevel = 2;
    displayLists.push_back( buildDisplayList() );
}

/* get a shared leaf texture */
GLuint FractalTree::getLeafTexture() {
    static GLuint leafTextId(0);
    if (leafTextId == 0) {
        string textureFile = string("imgs/tree/leaf.bmp");
        leafTextId = Mesh::loadTexture(textureFile);
    }
    return leafTextId;
}

/* get a shared bark texture */
GLuint FractalTree::getBarkTexture() {
    static GLuint barkTextId(0);
    if (barkTextId == 0) {
        string textureFile = string("imgs/tree/bark.bmp");
        barkTextId = Mesh::loadTexture(textureFile);
    }
    return barkTextId;
}

/* constructor */
FractalTree::FractalTree(GameWorld& gw): 
    GameObject(gw),
    root(NULL),
    minBranchSize(0), 
    numBranches(0),
    totalNodes(0),
    rootRadius(0),
    lodLevel(0)
{
    // set some approx. dims for collision det
    setWidth( 1 );
    setHeight( 1 );
    setLength( 1 );
}

/* destructor */
FractalTree::~FractalTree() {
    delete root;
}

/* printing Tree */
ostream& operator<<(ostream& s, const FTNode& n) {
    
    string depthStr("  ");
    for (int i=0; i < n.level; i++) {
        s << depthStr;
    }
    s << "[" << n.level << " " << n.branchNum << " ";
    s << n.start << " " << n.end;
    Vec3f v(n.end);
    v.sub(n.start);
    s << " " << v.magnitude() << " " << n.radius << " " << n.isLeaf << "]";
    s << " " << n.childNodes.size() << " " << n.getParentRadius();
    s << endl;
    
    for (unsigned int j = 0; j < n.childNodes.size(); j++) {
        //s << n.childNodes[j];
    }
    return s;
}
