/*
 * Class to read in a fractally generated Height Map for terrain
 * generation
 */

#include <SDL_opengl.h>
#include <SDL.h>
#include <cmath>
#include <cstdlib>

#include "FractalHeightMap.h"

using namespace std;

/* default constructor, initialize members */
FractalHeightMap::FractalHeightMap() : offset(DEFAULT_OFFSET), octaves(DEFAULT_OCTAVES),
        roughness(DEFAULT_ROUGHNESS), lacunarity(DEFAULT_LACUNARITY) {
    initFrequencyWeights();
}

/* destructor */
FractalHeightMap::~FractalHeightMap() {
    delete[] frequencyWeights;
}

/* get the fractal value with the given inputs */
GLfloat FractalHeightMap::getHeight(GLfloat xPos, GLfloat zPos) {
    GLfloat result;
    float signal, weight, remainder;
    result = (noise(xPos, zPos) + offset) * frequencyWeights[0];
    weight = result;
    xPos *= lacunarity;
    zPos *= lacunarity;

    // sum weighted noise values
    for (int i = 1; i < octaves; i++) {
        if (weight > 1.0) weight = 1.0;
        signal = (noise(xPos, zPos) + offset) * frequencyWeights[i];
        result += weight * signal;
        weight *= signal;
        xPos *= lacunarity;
        zPos *= lacunarity;
    }

    // take care of remainder part of octaves
    remainder = octaves - (int)octaves;
    if (remainder > 0.0) {
        result += remainder * noise(xPos, zPos) * frequencyWeights[(int)octaves];
    }

    return result;
}

/* initialize the frequency weight map used to generate fractal */
void FractalHeightMap::initFrequencyWeights() {
    // initialize the frequency weight map array
    frequencyWeights = new float[(int)octaves + 1];

    // compute weight for each frequency
    float frequency = 1.0;
    for (int i = 0; i <= octaves; i++) {        
        frequencyWeights[i] = powf(frequency, -roughness);
        frequency += lacunarity;
    }
}

/* returns a random value with the given inputs */
float FractalHeightMap::randomValue(float x, float z) {
    int n = x + z * 57;
    return (1.0 - ((n * (n * n * 15731 + 789221) + 1376312589) & 0x7fffffff) / 1073741824.0);
}

/* returns a smoothed, weighted sum of random values given the inputs */
float FractalHeightMap::smooth(float x, float z) {
    return (randomValue(x - 1.0, z - 1.0) + randomValue(x + 1.0, z - 1.0) +
            randomValue(x - 1.0, z + 1.0) + randomValue(x + 1.0, z + 1.0)) / 16.0 +
           (randomValue(x - 1.0, z) + randomValue(x + 1.0, z) +
            randomValue(x, z - 1.0) + randomValue(x, z + 1.0)) / 8.0 +
           randomValue(x, z) / 4.0;
}

/* returns the linear interpolation between the given inputs with the fractional weight */
float FractalHeightMap::lerp(float x, float z, float fraction) {
    return x * (1 - fraction) + z * fraction;
}

/* generates a noise value given the inputs */
float FractalHeightMap::noise(float x, float z) {
    int intX = (int)x;
    int intZ = (int)z;
    float fractionX = x - intX;
    float fractionZ = z - intZ;

    float v1 = smooth(intX, intZ);
    float v2 = smooth(intX + 1.0, intZ);
    float v3 = smooth(intX, intZ + 1.0);
    float v4 = smooth(intX + 1.0, intZ + 1.0);

    return lerp(lerp(v1, v2, fractionX), lerp(v3, v4, fractionX), fractionZ);
}
