//
// Cow implementation
//
#include <iostream>
#include <cmath>
#include <string>
#include <vector>
#include <algorithm>
#include <SDL_mixer.h>

#include "Cow.h"
#include "Terrain.h"
#include "Corral.h"
#include "Frustum.h"
#include "Truck.h"

using namespace std;

extern bool musicMode;

/* construct with game world, setup boids parameters */
Cow::Cow(GameWorld& gw): 
    GameObject(gw),
    maxAccel( 2 ),
    minDist( 2 ),
    ruleCenterWeight( 1.0f / 100.0f ),
    ruleCollisionWeight( 1 ),
    ruleTruckWeight( 3 ),
    ruleMatchWeight( 1.0f / 8.0f ),
    ruleInboundsWeight( 2 ),
    ruleHornWeight( 5 ),
    removedFromGame(false),
    prevAngleY( 0 ),
    maxRotateAngle( 5 ),
    hornStart( 0 ),
    hornRange( 20 ),
    hornDuration( 1000 )
{
    // set generic GameObject params
    topSpeed = 5;
    
    // set sizes
    setWidth(2);
    setLength(2.5);
    setHeight(2);
    
    // set boundaries for map for cows
    GLfloat bound = getCowBounds();
    mapBoundary.xMax = bound;
    mapBoundary.xMin = -bound;
    mapBoundary.zMax = bound;
    mapBoundary.zMin = -bound;

    mesh = getCowMesh();

    if (musicMode) {
        // load a random moo
        vector<string> files;
        files.push_back(string("sounds/cow.wav"));
        files.push_back(string("sounds/cow2.wav"));
        files.push_back(string("sounds/cow3.wav"));
        files.push_back(string("sounds/cow4.wav"));
        random_shuffle(files.begin(), files.end());
        
        sound = Mix_LoadWAV(files.front().c_str());
        if (sound == NULL) {
            cout << "Failed to load sound effect " << files.front().c_str();
            cout << " " << Mix_GetError() << endl;
        }
    }
}

/* get the cow bounds */
GLfloat Cow::getCowBounds() {
    static GLfloat bounds = 40;
    return bounds;
}

/* get a shared mesh for all cows */
Mesh* Cow::getCowMesh() {
    static Mesh* cow = new Mesh("cow");
    return cow;
}

/* updates this object according to time elapsed, do boids here */
void Cow::update(Uint32 timeElapsed) {
    // figure out combined acceleration from Boid behavior
    accel.set(0,0,0);
    accumulate( accel, ruleHorn(), maxAccel);
    accumulate( accel, ruleAvoidCollision(), maxAccel);
    accumulate( accel, ruleStayInbounds(), maxAccel);
    accumulate( accel, ruleAvoidTruck(), maxAccel);
    accumulate( accel, ruleCenterFlock(), maxAccel);
    accumulate( accel, ruleMatchVelocity(), maxAccel);
    
    // update vel and loc
    updateLocAndVel(timeElapsed);
    
    // update heading (angleY)
    if (vel.nonZero()) {
        setAngleY(vel.getAngleYaxis());
    }
    
    // test if we're in the corral, stop being part of the herd if so
    //NO_TR1 Corral* corral = (Corral*)world.getCorral().get();
    Corral* corral = world.getCorral();
    if (!removedFromGame && corral->isInCorral(loc)) {
        world.removeCow(this);
        // stop following herd
        ruleCenterWeight = 0;
        ruleMatchWeight = 0;
        removedFromGame = true;
    }
}

/* reset the cow's state */
void Cow::reset() {
    removedFromGame = false;
    ruleCenterWeight = 1.0f / 100.0f;
    ruleMatchWeight = 1.0f / 8.0f;
}

/*  add inside corral checking to keep from leaving corral */
bool Cow::checkNewLoc(Vec3f newLoc) {
    bool insideBounds = GameObject::checkNewLoc(newLoc);
    if (!insideBounds) {
        return false;
    } else if (removedFromGame) {
        // stay in corral
        return world.getCorral()->isInCorral(newLoc);
    }
    return true;
}

/* accumulate into vector, with limit max */
void Cow::accumulate(Vec3f& target, Vec3f addAmt, GLfloat maxMag) {
    GLfloat targetMag = target.magnitude();
    if (targetMag < maxMag) {
        GLfloat addMag = addAmt.magnitude();
        if ((targetMag + addMag) > maxMag) {
            addAmt.trim(maxMag - targetMag);
        }
        target.add(addAmt);
    }
}

/* velocity rule for staying in bounds of world */
Vec3f Cow::ruleStayInbounds() {
    Vec3f inbounds;
    if (loc.getX() - getBoundSphereDist() < mapBoundary.xMin + minDist) {
        inbounds.setX(1);
    } else if (loc.getX() + getBoundSphereDist() > mapBoundary.xMax - minDist) {
        inbounds.setX(-1);
    }
    if (loc.getZ() - getBoundSphereDist() < mapBoundary.zMin + minDist) {
        inbounds.setZ(1);
    } else if (loc.getZ() + getBoundSphereDist() > mapBoundary.zMax - minDist) {
        inbounds.setZ(-1);
    }
    inbounds.mult( ruleInboundsWeight );
    return inbounds;
}

/* velocity rule for avoiding truck */
Vec3f Cow::ruleAvoidTruck() {
    Vec3f truckDist(world.getTruck()->getLocation());
    truckDist.sub(this->getLocation());
    GLfloat safeDist = world.getTruck()->getBoundSphereDist() + getBoundSphereDist() + minDist;
    if (truckDist.magnitude() < safeDist) {
        truckDist.mult( -1 );
    } else {
        truckDist.set(0, 0, 0); //no effect
    }
    
    // mult by weight factor
    truckDist.mult( ruleTruckWeight );
    return truckDist;    
}

/* signal that truck has sounded horn */
void Cow::applyHorn() {
    hornStart = SDL_GetTicks();
}

/* velocity rule for truck horn */
Vec3f Cow::ruleHorn() {
    Uint32 current = SDL_GetTicks();
    Vec3f truckDist;
    if (hornStart != 0 && (current - hornStart) < hornDuration) {
        truckDist = this->getLocation();
        truckDist.sub(world.getTruck()->getLocation());
        GLfloat mag = truckDist.magnitude();
        if (mag > hornRange) {
            truckDist.set(0, 0, 0); //no effect
        } else {
            // max accel in direction, zero prev accel
            accel.set(0,0,0);
            truckDist.normalize();
            truckDist.mult(ruleHornWeight);
        }
    }
    return truckDist;    
}

/* velocity rule for going towards center of neighbors */
Vec3f Cow::ruleCenterFlock() {
    Vec3f center;
    GameObjVec neighbors = getNearbyCows();
    if (neighbors.size() > 1) { // need at least 2 cows
        GameObjVec::iterator i;
        for (i = neighbors.begin(); i != neighbors.end(); i++) {
            GameObjPtr obj = (*i);
            //NO_TR1 if (obj.get() != this) {
            if (obj != this) {
                center.add( obj->getLocation() );
            }
        }
        center.div(neighbors.size() - 1);
        
        // get diff to current loc
        center.sub(this->getLocation());
        center.mult( ruleCenterWeight ); // weight factor
    }
    return center;
}
    

/* velocity rule for avoiding collision */
Vec3f Cow::ruleAvoidCollision() {
    Vec3f away;
    GameObjVec neighbors = getNearbyObjs();
    GameObjVec::iterator i;
    for (i = neighbors.begin(); i != neighbors.end(); i++) {
        GameObjPtr obj = (*i);
        // don't include self
        if (obj != this) {
            Vec3f dist(obj->getLocation());
            dist.sub( loc );
            GLfloat safeDist = obj->getBoundSphereDist() + getBoundSphereDist() + minDist;
            if (dist.magnitude() < safeDist) {
                away.sub( dist );
            }
        }
    }
    away.mult( ruleCollisionWeight ); //weight factor
    return away;
}

/* velocity rule for matchig velocity of neighbors */
Vec3f Cow::ruleMatchVelocity() {
    Vec3f match;
    GameObjVec neighbors = getNearbyCows();
    if (neighbors.size() > 1) { // need at least 2 cows
        GameObjVec::iterator i;
        for (i = neighbors.begin(); i != neighbors.end(); i++) {
            GameObjPtr obj = (*i);
            //NO_TR1 if (obj.get() != this) {
            if (obj != this) {
                match.add(obj->getVelocity());
            }
        }
        match.div(neighbors.size() - 1);
        
        // get diff to current vel
        match.sub(this->getVelocity());
        match.mult( ruleMatchWeight ); // weight factor
    }
    return match;
}

/* get list of cows in the vicinity, that will affect boids behavior */
GameObjVec& Cow::getNearbyCows() {
    // just returns all cows right now
    // will need some area based formula later
    return world.getCows();
}

/* get list of objects to avoid collision from */
GameObjVec Cow::getNearbyObjs() {
    // just returns all objs in world right now
    // need more efficient CD later
    return world.getCollidableObjs();
}

    
/* rotate this cow to match velocity vector */
void Cow::rotateToVel(){
    // rotate to match heading 
    GLfloat newAngle = getAngleY();
    // but don't rotate too much at a time?
//        if (abs(newAngle - prevAngleY) > maxRotateAngle) {
//            newAngle = prevAngleY + 
//                ((newAngle > prevAngleY) ? maxRotateAngle : -maxRotateAngle);
//        }
    glRotatef(newAngle, 0, 1, 0);
    prevAngleY = newAngle;
}

/* draw a cow at current location */
void Cow::draw() {    
    if (world.getViewFrustum()->objectIntersects(this) > OUTSIDE_FRUSTUM) {
        // isolate transformations specific to this box
        glPushMatrix();
        
        // translate to x, y, z
        glTranslatef( loc.getX(), loc.getY(), loc.getZ());

        // draw vel/normal for debugging
        drawDebugInfo();
        
        // rotate to terrain normal
        rotateToNormal();
        
        // rotate to match velocity dir
        rotateToVel();
        
        // move up off the ground a bit
        glTranslatef(0, 1.2, -0.5);
        
        // render mesh
        mesh->render();
		
        // return stack to its previous state
        glPopMatrix();  
    }
}
