#include "World.h"
#include "Tunnel.h"

// static initialization code
Planet ** World::planets = new Planet*[NUMBER_OF_PLANETS];
int * World::tunnelConnections = new int[20*NUMBER_OF_PLANETS];

struct sample {
    Uint8 *data;
    Uint32 dpos;
    Uint32 dlen;
    Uint8 volume;
} sounds[NUMBER_OF_SOUNDS_P];

SDL_AudioCVT * cvt[NUMBER_OF_SOUNDS];

/** this code from http://www.libsdl.org/intro/usingsound.html */
void mixaudio(void *unused, Uint8 *stream, int len)
{
    int i;
    Uint32 amount;

    for ( i=0; i<NUMBER_OF_SOUNDS_P; ++i ) {
        amount = (sounds[i].dlen-sounds[i].dpos);
        if ( (int)amount > len ) {
            amount = len;
        }
        SDL_MixAudio(stream, &sounds[i].data[sounds[i].dpos], amount, sounds[i].volume);
        sounds[i].dpos += amount;
    }
}

void World::loadSound(char * file, int i)
{
    SDL_AudioSpec wave;
    Uint8 *data;
    Uint32 dlen;

    cvt[i] = new SDL_AudioCVT();
    /* Load the sound file and convert it to 16-bit stereo at 22kHz */
    if ( SDL_LoadWAV(file, &wave, &data, &dlen) == NULL ) {
        fprintf(stderr, "Couldn't load %s: %s\n", file, SDL_GetError());
        return;
    }
    SDL_BuildAudioCVT(cvt[i], wave.format, wave.channels, wave.freq,
                            AUDIO_S16,   2,             22050);
    cvt[i]->buf = (Uint8 *) malloc(dlen*cvt[i]->len_mult);
    memcpy(cvt[i]->buf, data, dlen);
    cvt[i]->len = dlen;
    SDL_ConvertAudio(cvt[i]);
    SDL_FreeWAV(data);
}

void World::loadAllSounds()
{
    this->loadSound("Sounds/mineexplosion.wav", SOUND_EXPLOSION_MINE);
    this->loadSound("Sounds/missileexplosion.wav", SOUND_EXPLOSION_MISSILE);
    this->loadSound("Sounds/spacejump.wav", SOUND_SPACE_JUMP);
    this->loadSound("Sounds/driveby.wav", SOUND_DRIVEBY);
    this->loadSound("Sounds/missilelaunch.wav", SOUND_MISSILE_LAUNCH);
    this->loadSound("Sounds/subping.wav", SOUND_AUTODESTROY);
    this->loadSound("Sounds/splooge.wav", SOUND_SMOKE);
    this->loadSound("Sounds/spike.wav", SOUND_SPIKE);
    this->loadSound("Sounds/acceleration.wav", SOUND_ACCELERATION);
    this->loadSound("Sounds/boing.wav", SOUND_BOING);
    //this->loadSound("Sounds/boing2.wav", SOUND_BOING2);
}

/** this code from http://www.libsdl.org/intro/usingsound.html */
int World::playSound(int i, int volume)
{
    if (this->mute) return -1;

    int index;
    //SDL_AudioSpec wave;
    //Uint8 *data;
    //Uint32 dlen;
    //SDL_AudioCVT cvt;

    /* Look for an empty (or finished) sound slot */
    for ( index=0; index<NUMBER_OF_SOUNDS_P; ++index ) {
        if ( sounds[index].dpos == sounds[index].dlen ) {
            break;
        }
    }
    if ( index == NUMBER_OF_SOUNDS_P )
        return -1;

    /* Put the sound data in the slot (it starts playing immediately) */
    //if ( sounds[index].data ) {
    //    free(sounds[index].data);
    //}
    SDL_LockAudio();
    sounds[index].data = cvt[i]->buf;
    sounds[index].dlen = cvt[i]->len_cvt;
    sounds[index].dpos = 0;
    sounds[index].volume = volume;
    SDL_UnlockAudio();
    
    return index;
}

void World::killSound(int s)
{
    sounds[s].dpos = sounds[s].dlen;
}

World::World(int numberOfSpheres)
{
    if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO) != 0)
	    fprintf(stderr, "Unable to initialize SDL: %s\n", SDL_GetError());
    
    this->video_flags = SDL_OPENGL | SDL_GL_DOUBLEBUFFER | SDL_FULLSCREEN;
//SDL_RESIZABLE;

    this->width = 1000;
    this->height = 750;

    SDL_WM_SetCaption(GAME_TITLE, NULL);
    SDL_SetVideoMode(this->width, this->height, 32, this->video_flags);

    glViewport(0, 0, this->width, this->height);

    SDL_ShowCursor( SDL_DISABLE );

    /* Set 16-bit stereo audio at 22Khz */
    fmt.freq = 22050;
    fmt.format = AUDIO_S16;
    fmt.channels = 2;
    fmt.samples = 512;        /* A good value for games */
    fmt.callback = mixaudio;
    fmt.userdata = NULL;
    this->mute = false;

    /* Open the audio device and start playing sound! */
    if ( SDL_OpenAudio(&fmt, NULL) < 0 ) {
        fprintf(stderr, "Unable to open audio: %s\n", SDL_GetError());
        exit(1);
    }
    SDL_PauseAudio(0);

    initLight();

    this->spheres = new Sphere*[numberOfSpheres];
    this->numberOfSpheres = 0; // increased as spheres are added

    this->n = NUMBER_OF_EXTRUSIONS;
    this->maxN = NUMBER_OF_EXTRUSIONS + 1;

    this->p = new float[3];
    this->t = new float[3];
    this->v = new float[3];
    this->eye = new float[3];
    this->front = new float[3];
    this->top = new float[3];
    this->mapTop = new float[3];

    for (int i=0; i<3; i++)
    {
        this->p[i] = 0;
        this->t[i] = 0;
        this->v[i] = 0;
        this->eye[i] = 0;
        this->front[i] = 0;
        this->top[i] = 0;
        this->mapTop[i] = 0;
    }
    this->mapTop[0] = 1;
    this->top[2] = 1;

    this->dp = 0;
    this->maxdp = PLAYERS_MAX_SPEED_NORMAL;
    this->dt = 0.08f;
    this->mv_forward=0;
    this->mv_backward=0;
    this->mv_left=0;
    this->mv_right=0;
    this->mv_shift=0;
    this->radar_on = 1;
    this->die = false;
    this->currentFace = 18;
    this->closeTop = DISTANCE_UPON_CAMERA;
    this->closeBehind = DISTANCE_BEHIND_CAMERA;
    this->lookingDown = false;
    this->onTunnel = false;
    this->pauseGame = false;
    this->gameIsOver = 0;

    this->loadInitialTextures();

    this->timeOffset = 1;
    this->deltaTimeOffset = 1;
    this->timeOfLatestAutodestroy = 0;

    this->playing_shift_sound = -1;

    this->onWelcomeScreen = true;

    for (int km=0; km<NUMBER_OF_MINES; km++)
        this->mines[km] = new Mine();

    this->wire = false;

    this->numberOfAliveTargets = NUMBER_OF_TARGETS;
    this->score = 0;
}

World::~World(void)
{
}

void World::addSphere(Sphere * s)
{
    this->spheres[this->numberOfSpheres] = s;
    this->numberOfSpheres++;
}

void World::initLight(void)
{
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    //glEnable(GL_LIGHT2);

    LIGHT0_position[0]=30;
    LIGHT0_position[1]=1;
    LIGHT0_position[2]=10;
    LIGHT0_position[3]=1;
    GLfloat LIGHT0_color[4] = {1.0f, 1.0f, 1.0f, 1.f};
    GLfloat LIGHT0_color2[4] = {0.3f, 0.3f, 0.2f, 1.f};
    glLightfv(GL_LIGHT0, GL_POSITION, LIGHT0_position);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, LIGHT0_color);
    glLightfv(GL_LIGHT0, GL_AMBIENT, LIGHT0_color2);

    GLfloat LIGHT1_position[4] = {0, 0, 0.1f, 1};
    GLfloat LIGHT1_color[4] = {1, 1, 1, 0};
    glLightfv(GL_LIGHT1, GL_POSITION, LIGHT1_position);
    glLightfv(GL_LIGHT1, GL_AMBIENT, LIGHT1_color);

    //GLfloat LIGHT2_color[4] = {0.6f, 0.6f, 0.6f, 1};
    //glLightfv(GL_LIGHT2, GL_POSITION, LIGHT1_position);
    //glLightfv(GL_LIGHT2, GL_AMBIENT, LIGHT2_color);

    glLightfv(GL_LIGHT3, GL_AMBIENT, LIGHT1_color);
}

int World::selectAutodestroyTarget()
{
    int i0 = SDL_GetTicks() % NUMBER_OF_TARGETS;
    int bestCandidate = 0;
    bool isAlive = false;
    for (int i=0; i<NUMBER_OF_TARGETS; i++)
    {
        int j = (i+i0) % NUMBER_OF_TARGETS;
        if (this->targets[j]->isAlive)
        {
            if (this->targets[i]->planet == this->currentPlanet)
            {
                if (!isAlive)
                {
                    bestCandidate = j;
                    isAlive = true;
                }
            }
            else
                return j;        
        }
    }
    return bestCandidate;
}

void World::autodestroyTarget()
{
    Target * t = this->targets[this->selectAutodestroyTarget()];
    t->smoke(this);
    this->playSound(SOUND_AUTODESTROY, MY_SDL_MIX_MAXVOLUME);
    t->autoDestroyed = true;
}

void World::update()
{
    if (this->gameIsOver) this->gameIsOver--;

    if (mv_forward)
    {
        if (this->dp < this->maxdp) this->dp += 0.0005f;
        if (this->dp < 0) this->dp += 0.0005f;
        if (this->dp > this->maxdp) this->dp -= 0.0005f;
    }
    else if (mv_backward)
    {
        if (this->dp > -this->maxdp) this->dp -= 0.0005f;
        if (this->dp > 0) this->dp -= 0.0005f;
        if (this->dp < -this->maxdp) this->dp += 0.0005f;
    }
    else // decelerate
    {
        if (this->dp > 0) this->dp -= 0.0005f;
        else if (this->dp < 0) this->dp += 0.0005f;
    }

    if (this->dp < 0.00001 && this->dp > -0.00001) 
        this->dp = 0;
    else
        for (int i=0; i<3; i++) 
            this->p[i] += this->dp*this->t[i];

    Planet::tunnels[Planet::getCurrentFace(this->p, this->currentFace)].projectOnTunnelD(this);

    v[0] = this->top[1]*t[2]-this->top[2]*t[1];
    v[1] = this->top[2]*t[0]-this->top[0]*t[2];
    v[2] = this->top[0]*t[1]-this->top[1]*t[0];

    if (mv_left + mv_right)
    {
        if (mv_left)
            for (int i=0; i<3; i++) t[i] += dt * v[i];
        else // if (mv_right)
            for (int i=0; i<3; i++) t[i] -= dt * v[i];
    }

    float norm = sqrt((float) t[0]*t[0]+t[1]*t[1]+t[2]*t[2]);
    for (int i=0; i<3; i++)
    {
        t[i] /= norm;
        this->eye[i] = p[i] + (this->closeTop*this->top[i] - this->closeBehind*t[i]);
        this->front[i] = p[i] + DISTANCE_BEHIND_VIEW_CENTER * t[i];
    }

    for (int i=0; i<NUMBER_OF_TARGETS; i++)
    {
        if (this->targets[i]->isAlive)
        {
            this->targets[i]->update(this);
            if (this->targets[i]->isAtImpact(this))
                if (this->targets[i]->spikiness == INITIAL_SPIKINESS)
                {
                    this->player->health -= 10;
                    if (this->player->health < 0)
                    {
                        this->player->health = 0;
                        this->gameIsOver = 10;
                    }
                    this->playSound(SOUND_SPIKE, MY_SDL_MIX_MAXVOLUME);
                    this->targets[i]->explode(this);
                }
                else
                {
                    this->playSound(SOUND_SMOKE, MY_SDL_MIX_MAXVOLUME);
                    this->score += 20;
                    this->targets[i]->smoke(this);
                }
        }
    }

    for (int i=0; i<NUMBER_OF_MINES; i++)
    {
        if (this->mines[i]->isAlive)
        {
            this->mines[i]->update(this);
            if (this->mines[i]->isAtImpact(this))
            {
                this->player->health -= 7;
                if (this->player->health < 0) 
                {
                    this->player->health = 0;
                    this->gameIsOver = 10;
                }
                this->mines[i]->explode(this);
            }
        }
    }

    for (int i=0; i<NUMBER_OF_MISSILES; i++)
    {
        if (this->player->missile[i]->isAlive)
        {
            this->player->missile[i]->update(this);
            if (this->player->missile[i]->isAtImpact(this))
                this->player->missile[i]->explode(this);
        }
    }

    float rImpact = this->player->radius + SNAKE_RADIUS; rImpact *= rImpact;
    for (int i=0; i<NUMBER_OF_SNAKES; i++)
    {
        Snake * snake = this->snakes[i];
        snake->update(this);

        // move player to the referentiel of the snake.
        float p1[3];
        if (this->currentPlanet == snake->planet1)
            for (int i=0; i<3; i++)
                p1[i] = this->p[i];
        else if (this->currentPlanet == snake->planet2)
            for (int i=0; i<3; i++)
                p1[i] = this->p[i] - snake->tr[i];
        else continue;

        if (snake->detectHeadImpact(p1, 3*rImpact))
        {
            if (this->dp >= 0)
            {
                if (this->dp < PLAYERS_MIN_SPEED_AFTER_BOUNCE)
                    this->dp = PLAYERS_MIN_SPEED_AFTER_BOUNCE;
            }
            else
            {
                if (this->dp > -PLAYERS_MIN_SPEED_AFTER_BOUNCE)
                    this->dp = -PLAYERS_MIN_SPEED_AFTER_BOUNCE;
            }

            this->dp = - this->dp;

            // h: w->p - the center of this snake's head
            Snake::Ring * ring = snake->rings + ((snake->firstRing/SNAKE_SPEED) % SNAKE_LENGTH);
            float h[3], n=0;
            for (int i=0; i<3; i++)
            {
                h[i] = this->p[i] - (ring->p[i] + 3 * SNAKE_HEAD_LENGTH * ring->t[i]);
                n += h[i] * h[i];
            }
            n = this->dp / sqrtf(n);
            for (int i=0; i<3; i++)
            {
                this->p[i] -= n * h[i];
            }

            this->playSound(SOUND_BOING, MY_SDL_MIX_MAXVOLUME);

            break;
        }

        if (!(snake->detectImpact(p1, rImpact, this->currentPlanet)))
            continue;

        if (this->dp >= 0)
        {
            if (this->dp < PLAYERS_MIN_SPEED_AFTER_BOUNCE)
                this->dp = PLAYERS_MIN_SPEED_AFTER_BOUNCE;
        }
        else
        {
            if (this->dp > -PLAYERS_MIN_SPEED_AFTER_BOUNCE)
                this->dp = -PLAYERS_MIN_SPEED_AFTER_BOUNCE;
        }

        this->dp = - this->dp;
        for (int i=0; i<3; i++)
        {
            this->p[i] += this->dp * this->t[i];
        }

        this->playSound(SOUND_BOING, MY_SDL_MIX_MAXVOLUME);

        break;
    }

    int t = (int)(SDL_GetTicks() - this->timeOffset);
    if (t - this->timeOfLatestAutodestroy > TIME_LAPS_BETWEEN_TARGET_AUTODESTROY)
    {
        this->timeOfLatestAutodestroy = t;
        this->autodestroyTarget();
    }

    this->player->update(this);
}

//void World::printSnakes()
//{
//    for (int i=0; i<NUMBER_OF_SNAKES; i++)
//    {
//        cout << "Snake " << i << " is on planet1 " << this->snakes[i]->planet1
//            << " and planet2 " << this->snakes[i]->planet2 << endl;
//    }
//}

void World::exitPause()
{
    this->pauseGame = false;
    this->onWelcomeScreen = false;
    this->timeOffset += this->deltaTimeOffset + SDL_GetTicks();
    if (mv_forward && mv_shift && this->playing_shift_sound == -1)
        playing_shift_sound = this->playSound(SOUND_ACCELERATION, MY_SDL_MIX_MAXVOLUME);
}

int World::handleEvents() 
{
    SDL_Event event;
	while (SDL_PollEvent(&event))
    {
	    switch (event.type)
        {
	    case SDL_QUIT:
		    SDL_Quit();
		    return 0;
	    case SDL_KEYDOWN:
            if (this->pauseGame)
            {
                if (this->gameIsOver == 1)
                {
		            switch (event.key.keysym.sym)
                    {
		            case SDLK_ESCAPE:
		                SDL_Quit();
		                return 0;
                    default:
                        break;
                    }
                }
                else
                {
		            switch (event.key.keysym.sym)
                    {
                    case SDLK_LCTRL:
                    case SDLK_RCTRL:
                        this->exitPause();
		                this->player->shoot(this);
		                break;
                    case SDLK_SPACE:
                        this->exitPause();
                        for (int i=0; i<NUMBER_OF_MISSILES; i++)
                            this->player->missile[i]->explode(this);
                        break;
                    case SDLK_UP:
                        this->exitPause();
                        mv_forward = 1;
                        break;
                    case SDLK_DOWN:
                        this->exitPause();
                        mv_backward = 1;
                        break;
                    case SDLK_LEFT:
                        this->exitPause();
                        mv_left = 1;
                        break;
                    case SDLK_RIGHT:
                        this->exitPause();
                        mv_right = 1;
                        break;
                    case SDLK_LSHIFT:
                    case SDLK_RSHIFT:
                        this->exitPause();
                        mv_shift++;
                        this->maxdp = PLAYERS_MAX_SPEED_SHIFT;
                        break;
                    case SDLK_r:
                        this->exitPause();
                        this->radar_on = ! this->radar_on;
                        break;
                    default:
                        this->exitPause();
		                break;
                    }
                }
            }
            else
            {
		        switch (event.key.keysym.sym)
                {
		        case SDLK_ESCAPE:
		            SDL_Quit();
		            return 0;
                case SDLK_LCTRL:
                case SDLK_RCTRL:
		            this->player->shoot(this);
		            break;
                case SDLK_SPACE:
                    for (int i=0; i<NUMBER_OF_MISSILES; i++)
                        this->player->missile[i]->explode(this);
                    break;
                case SDLK_UP:
                    if (mv_shift && this->playing_shift_sound == -1) 
                        playing_shift_sound = this->playSound(SOUND_ACCELERATION, MY_SDL_MIX_MAXVOLUME);
                    mv_forward = 1;
                    break;
                case SDLK_DOWN:
                    mv_backward = 1;
                    break;
                case SDLK_LEFT:
                    mv_left = 1;
                    break;
                case SDLK_RIGHT:
                    mv_right = 1;
                    break;
                case SDLK_LSHIFT:
                case SDLK_RSHIFT:
                    if (mv_forward && this->playing_shift_sound == -1) 
                        playing_shift_sound = this->playSound(SOUND_ACCELERATION, MY_SDL_MIX_MAXVOLUME);
                    mv_shift++;
                    this->maxdp = PLAYERS_MAX_SPEED_SHIFT;
                    break;
                //case SDLK_1:
                //    n = (n+1) % this->maxN;
                //    break;
                case SDLK_r:
                    this->radar_on = ! this->radar_on;
                    break;
                case SDLK_m:
                    this->wire = ! this->wire;
                    break;
                //case SDLK_i:
                //    this->printSnakes();
                //    break;
                case SDLK_s:
                    for (int i=0; i<NUMBER_OF_SOUNDS_P; i++)
                        sounds[i].dpos = sounds[i].dlen;
                    this->mute = ! this->mute;
                    break;
                //case SDLK_o:
                //    this->gameIsOver=10;
                //    break;
                case SDLK_p:
                    this->deltaTimeOffset = - (int) SDL_GetTicks();
                    this->pauseGame = true;
                    this->timeOffset += this->deltaTimeOffset + SDL_GetTicks();
                    this->deltaTimeOffset = - (int) SDL_GetTicks();
                    this->draw();
                    this->drawPauseScreen();
                    SDL_GL_SwapBuffers();
                    this->timeOffset += this->deltaTimeOffset + SDL_GetTicks();
                    this->deltaTimeOffset = - (int) SDL_GetTicks();
                    this->draw();
                    this->drawPauseScreen();
                    SDL_GL_SwapBuffers();
                    if (this->playing_shift_sound != -1) 
                    {
                        sounds[this->playing_shift_sound].dpos = sounds[this->playing_shift_sound].dlen;
                        this->playing_shift_sound = -1;
                    }
                    break;
                default:
		            break;
		        }
            }
		    break;
        case SDL_KEYUP:
		    switch (event.key.keysym.sym)
            {
            case SDLK_UP:
                mv_forward = 0;
                if (this->playing_shift_sound != -1) 
                {
                    sounds[this->playing_shift_sound].dpos = sounds[this->playing_shift_sound].dlen;
                    this->playing_shift_sound = -1;
                }
                break;
            case SDLK_DOWN:
                mv_backward = 0;
                break;
            case SDLK_LEFT:
                mv_left = 0;
                break;
            case SDLK_RIGHT:
                mv_right = 0;
                break;
            case SDLK_LSHIFT:
            case SDLK_RSHIFT:
                if (this->playing_shift_sound != -1) 
                {
                    sounds[this->playing_shift_sound].dpos = sounds[this->playing_shift_sound].dlen;
                    this->playing_shift_sound = -1;
                }
                mv_shift--;
                if (!mv_shift)
                    this->maxdp = PLAYERS_MAX_SPEED_NORMAL;
                break;
		    default:
		        break;
            }
            break;
        case SDL_VIDEORESIZE:
            this->height = event.resize.h;
            this->width = event.resize.w;
            //cout << "vidresize " << height << " " << width << endl;
            SDL_SetVideoMode(event.resize.w, event.resize.h, 32, this->video_flags);
            this->loadInitialTextures();
            this->loadAdditionalTextures();
            this->initLight();
            if (this->pauseGame)
            {
                if (this->onWelcomeScreen)
                {
                    this->draw();
                    this->draw();
                    this->drawWelcomeScreen();
                    SDL_GL_SwapBuffers();
                    this->draw();
                    this->drawWelcomeScreen();
                    SDL_GL_SwapBuffers();
                }
                else
                {
                    this->draw();
                    this->draw();
                    this->drawPauseScreen();
                    SDL_GL_SwapBuffers();
                    this->draw();
                    this->drawPauseScreen();
                    SDL_GL_SwapBuffers();
                }
            }
            break;
	    default:
    		break;
	    }
	}
    return 1;
}

void World::loadInitialTextures()
{
    Sphere::loadTexture("Textures/welcome.bmp", &this->welcomeTextureId);
    Sphere::loadTexture("Textures/white.bmp", &this->whiteTextureId);
    Sphere::loadTexture("Textures/sphere.bmp", &this->smallSphereTextureId);
    char filename[15] = "Textures/i.bmp";
    for (int i=0; i<10; i++)
    {
        filename[9] = '0' + i;
        Sphere::loadTexture(filename, &(this->digitTextureId[i]));
    }
    Sphere::loadTexture("Textures/c.bmp", &(this->digitTextureId[10]));
    Sphere::loadTexture("Textures/cylinder.bmp", &(this->cylinderTextureId));

    Sphere::loadTexture("Textures/sky.bmp", &this->skyTextureId);
    Sphere::loadTexture("Textures/ground.bmp", & this->planetTextureId1);
    Sphere::loadTexture("Textures/planet.bmp", & this->planetTextureId2);
    Sphere::loadTexture("Textures/snake.bmp", &this->snakeTextureId);
    Sphere::loadTexture("Textures/snake_distant.bmp", &this->snakeDistantTextureId);
}

void World::loadAdditionalTextures()
{
    Sphere::loadTexture("Textures/pause.bmp", &(this->pauseTextureId));
    Sphere::loadTexture("Textures/fire.bmp", &this->fireTextureId);
    Sphere::loadTexture("Textures/smoke.bmp", &this->smokeTextureId);
    Sphere::loadTexture("Textures/smiley.bmp", &this->smileyTextureId);
}

void World::drawPauseScreen()
{
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    //gluPerspective(45, 4.0/3, 0.1, 100);
    float Y = ((float)(3*this->width))/this->height;
    glOrtho(-Y, Y, -3, 3, -10, 10);

    gluLookAt(0, 0, 1, 0, 0, 0, 0, 1, 0);

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();

    glClear(GL_DEPTH_BUFFER_BIT);

    int logs = 0;
    for (int s = this->score; s>0; s/=10) logs++;
    if (!logs) logs = 1;

    float Y2 = -Y + 0.22f * logs + 0.2f;

    glEnable(GL_BLEND);
    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glColor4f(0.5f, 1.0f, 0.0f, 0.75f);

    glBindTexture(GL_TEXTURE_2D, this->pauseTextureId);

    glBegin(GL_QUADS);
    glTexCoord2f(0, 1);
    glVertex3f(Y2, 2.5f, -0.5);
    glTexCoord2f(0, 0);
    glVertex3f(Y2, 2.9f, -0.5);
    glTexCoord2f(1, 0);
    glVertex3f(Y2+2.5f, 2.9f, -0.5);
    glTexCoord2f(1, 1);
    glVertex3f(Y2+2.5f, 2.5f, -0.5);
    glEnd();

    //SDL_GL_SwapBuffers();	

    //glDisable(GL_BLEND);

    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
}

void World::drawWelcomeScreen()
{
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    //gluPerspective(45, 4.0/3, 0.1, 100);
    glOrtho(-((float)(3*this->width))/this->height, ((float)(3*this->width))/this->height, -3, 3, -10, 10);

    gluLookAt(0, 0, 1, 0, 0, 0, 0, 1, 0);

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();

    glClear(GL_DEPTH_BUFFER_BIT);

    glEnable(GL_BLEND);
    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    
    glColor4f(0.5f, 1.0f, 0.0f, 0.9f);

    glBindTexture(GL_TEXTURE_2D, this->welcomeTextureId);

    glBegin(GL_QUADS);
    glTexCoord2f(0, 1);
    glVertex3f(-2.8f, -2, -0.5f);
    glTexCoord2f(0, 0);
    glVertex3f(-2.8f, 2.4f, -0.5f);
    glTexCoord2f(1, 0);
    glVertex3f(2.8f, 2.4f, -0.5f);
    glTexCoord2f(1, 1);
    glVertex3f(2.8f, -2, -0.5f);
    glEnd();

    //SDL_GL_SwapBuffers();	

    glDisable(GL_BLEND);

    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
}

void World::gameOver()
{
    this->gameIsOver = 0;

    int nTarg = 0;
    for (int i=0; i<NUMBER_OF_TARGETS; i++)
        if (!(this->targets[i]->isAlive || this->targets[i]->autoDestroyed))
            nTarg++;

    int sTime = 0;
    if (this->numberOfAliveTargets == 0)
    {
        sTime = 480 - ((int)(SDL_GetTicks() - this->timeOffset) / 1000);
        if (sTime < 0) sTime = 0;
    }

    this->deltaTimeOffset = - (int) SDL_GetTicks();
    Sphere::loadTexture("Textures/gameover.bmp", &(this->gameOverTextureId));
    this->timeOffset += this->deltaTimeOffset + SDL_GetTicks();
    this->deltaTimeOffset = - (int) SDL_GetTicks();
    this->draw();
    this->drawGameOverScreen(sTime + this->score, this->score, nTarg, sTime);
    SDL_GL_SwapBuffers();
    this->timeOffset += this->deltaTimeOffset + SDL_GetTicks();
    this->deltaTimeOffset = - (int) SDL_GetTicks();
    this->draw();
    this->drawGameOverScreen(sTime + this->score, this->score, nTarg, sTime);
    SDL_GL_SwapBuffers();

    if (this->playing_shift_sound != -1) 
    {
        sounds[this->playing_shift_sound].dpos = sounds[this->playing_shift_sound].dlen;
        this->playing_shift_sound = -1;
    }

    this->pauseGame = true;
}

void World::drawGameOverScreen(int s1, int s2, int s3, int s4)
{
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    glOrtho(-((float)(3*this->width))/this->height, ((float)(3*this->width))/this->height, -3, 3, -10, 10);

    gluLookAt(0, 0, 1, 0, 0, 0, 0, 1, 0);

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();

    glClear(GL_DEPTH_BUFFER_BIT);

    glEnable(GL_BLEND);
    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glColor4f(0.5f, 1, 0, 0.8f);

    glDisable(GL_CULL_FACE);

    Map::drawNumber(s1, this, 0.9f, 0.33f, 0.3f, 0.56f);

    Map::drawNumber(s2, this, -1.7f, -0.315f, 0.3f, 0.56f);

    Map::drawNumber(s3, this, -0.23f, -0.315f, 0.3f, 0.56f);

    Map::drawNumber(s4, this, -1.24f, -1.589f, 0.3f, 0.56f);

    glBindTexture(GL_TEXTURE_2D, this->gameOverTextureId);

    glBegin(GL_QUADS);
    glTexCoord2f(0, 1);
    glVertex3f(-3, -2, -0.5f);
    glTexCoord2f(0, 0);
    glVertex3f(-3, 2.4f, -0.5f);
    glTexCoord2f(1, 0);
    glVertex3f(3, 2.4f, -0.5f);
    glTexCoord2f(1, 1);
    glVertex3f(3, -2, -0.5f);
    glEnd();

    glDisable(GL_BLEND);

    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
}

void World::draw()
{
    if (this->gameIsOver == 1) 
    {
        this->gameOver();
        this->gameIsOver = 1;
        return;
    }

    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glEnable(GL_DEPTH_TEST);
    glDepthRange (0.01, 1.0);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(45, ((float)this->width)/this->height, 0.001, 100);

    gluLookAt(this->eye[0], this->eye[1], this->eye[2],
        this->front[0], this->front[1], this->front[2], 
        this->top[0], this->top[1], this->top[2]);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    glLightfv(GL_LIGHT0, GL_POSITION, LIGHT0_position);
    glColor4f(1.0f, 1.0f, 1.0f, 0.0f);

    for (int i=0; i<this->numberOfSpheres; i++)
        this->spheres[i]->draw(this);

    this->planets[this->currentPlanet]->drawCurrentPlanet(this);

    //SDL_GL_SwapBuffers();	
}

