#include "World.h"
#include "Tunnel.h"

float * Snake::trigo = NULL;
float * Snake::textureCoordonates = NULL;
float Snake::dl = SNAKE_RING_LENGTH / (6.28318531f * SNAKE_RADIUS * SNAKE_SPEED);
float * Snake::cosalpha = new float[SNAKE_PULSE];

Snake::Snake(World * w, float x, float y, float z, float tx, float ty, float tz, int pl)
{
    if (! Snake::trigo)
    {
        Snake::precomputeTrigoValues(SNAKE_TESSELIZATION);
        Snake::precomputeTextureCoords(SNAKE_TESSELIZATION);
    }

    this->planet1 = pl;
    this->planet2 = pl;

    float p0[3] = {x, y, z};
    float t0[3] = {tx, ty, tz};
    for (int i=0; i<3; i++) 
    {
        this->tail[i] = p0[i];
        this->tr[i] = 0;
        //this->tr2[i] = 0;
        this->rings[0].t[i] = t0[i];
        this->rings[0].p[i] = p0[i] + SNAKE_RING_LENGTH * t0[i];
    }
    Snake::positionNextRing(this->rings, 0, w);
    Snake::computeVertices(this->rings);
    this->alpha = 0;

    for (int i=1; i<SNAKE_LENGTH; i++)
    {
        this->rings[i].radius = SNAKE_RADIUS;
        for (int j=0; j<3; j++) 
        {
            this->rings[i].t[j] = this->rings[i-1].t[j];
            this->rings[i].p[j] = this->rings[i-1].p[j] + SNAKE_RING_LENGTH * this->rings[i-1].t[j];
        }

        this->positionNextRing(this->rings + i, this->rings[i-1].face, w);

        Snake::computeVertices(this->rings + i);
        this->alpha = (this->alpha + 1) % SNAKE_PULSE;
        for (int j=0; j<3; j++)
        {
            this->rings[i].t[j] += SNAKE_AMPLITUDE * cosalpha[this->alpha] * this->rings[i].v[j];
        }
    }

    this->firstRing = SNAKE_LENGTH * SNAKE_SPEED - 1;
}

Snake::~Snake(void)
{
}

void Snake::precomputeTrigoValues(int N)
{
    Snake::trigo = new float[2*N];
    for (int i=0; i<N; i++)
    {
        Snake::trigo[2*i]     = cosf(i * 6.28318531f / N);
        Snake::trigo[2*i + 1] = sinf(i * 6.28318531f / N);
    }
    //for (int i=0; i<N; i++) cout << "cos: " << trigo[2*i] << "; sin: " << trigo[2*i+1] << endl;

    for (int i=0; i<SNAKE_PULSE; i++)
        Snake::cosalpha[i] = cosf(i * 6.28318531f / SNAKE_PULSE);
}

void Snake::precomputeTextureCoords(int N)
{
    Snake::textureCoordonates = new float[N];
    float dr = 1.0f / SNAKE_TESSELIZATION;
    for (int j=0; j<N; j++)
        Snake::textureCoordonates[j] = dr*j;
}

void Snake::positionNextRing(Ring * ring, int face, World * w)
{
    float c[3];
    //float pos[3] = {ring->p[0] + this->tr1[0], ring->p[1] + this->tr1[1], ring->p[2] + this->tr1[2]};

    ring->face = Planet::getCurrentFace(ring->p, face);

    Tunnel * tun = Planet::tunnels + ring->face;
    float xQ, yQ;

    yQ = 0; // yQ is pos .scal axis
    for (int i=0; i<3; i++) 
        yQ += ring->p[i] * tun->axis[i];

    // ATTENTION - this is where you switch planet (the JUMP)
    if (yQ <= tun->yC && Planet::oppositeTunnels[ring->face] != -1)
    {
        ring->insideHole = true;

        xQ = 0;
        for (int i=0; i<3; i++)
        {
            float dxQ = ring->p[i] - yQ * tun->axis[i];
            xQ += dxQ * dxQ;
        }
        xQ = sqrtf(xQ);

        float nn=0;
        for (int i=0; i<3; i++)
        {
            c[i] = tun->yC * tun->axis[i] + tun->xC * (ring->p[i] - yQ*tun->axis[i]) / xQ;
            ring->n[i] = ring->p[i] - c[i];
            nn += ring->n[i] * ring->n[i];
        }

        // normalize n and compute pos, the projection of itself on the surface
        nn = sqrtf(nn);
        for (int i=0; i<3; i++) 
        {
            ring->n[i] /= nn;
            ring->p[i] = c[i] + tun->radiusCurve*ring->n[i]; // - tr1[i];  // p=pos-tr1
        }

        // make t (tangeant) be normal to n (normal)
        xQ = ring->t[0]*ring->n[0]+ring->t[1]*ring->n[1]+ring->t[2]*ring->n[2];
        for (int i=0; i<3; i++) 
            ring->t[i] -= xQ*ring->n[i];

        // normalize t
        nn = sqrtf(ring->t[0]*ring->t[0] + ring->t[1]*ring->t[1] + ring->t[2]*ring->t[2]);
        for (int i=0; i<3; i++) 
            ring->t[i] /= nn;

        // compute v, the third axis
        ring->v[0] = ring->n[1]*ring->t[2]-ring->n[2]*ring->t[1];
        ring->v[1] = ring->n[2]*ring->t[0]-ring->n[0]*ring->t[2];
        ring->v[2] = ring->n[0]*ring->t[1]-ring->n[1]*ring->t[0];

        // update tr
        for (int i=0; i<3; i++) 
        {
            // tr2 is the translation you need to apply to the tail, that still lies ob the other planet
            //this->tr2[i] = this->tr1[i];
            this->tr[i] = 2 * tun->yC * tun->axis[i];
        }

        this->planet2 = this->planet1;
        this->planet1 = w->tunnelConnections[20 * this->planet1 + ring->face];
        ring->face = Planet::oppositeTunnels[ring->face];
        ring->planet = this->planet1;

        this->translateTail(this->tr);
        return;
    }

    xQ = 0;
    for (int i=0; i<3; i++)
    {
        float dxQ = ring->p[i] - yQ * tun->axis[i];
        xQ += dxQ * dxQ;
    }
    xQ = sqrtf(xQ);

    if (xQ <= tun->radiusExt && Planet::oppositeTunnels[ring->face] != -1)
    {
        ring->insideHole = true;

        float nn = 0;
        // compute yP of the projection P of Q onto the tunnel 
        // in the direction of QC (towards the curvature center)
  
        // local curvature center coordonates c and normal n
        for (int i=0; i<3; i++)
        {
            c[i] = tun->yC * tun->axis[i] + tun->xC * (ring->p[i] - yQ*tun->axis[i]) / xQ;
            ring->n[i] = ring->p[i] - c[i];
            nn += ring->n[i] * ring->n[i];
        }

        // normalize n and compute p, the projection of itself on the surface
        nn = sqrtf(nn);
        for (int i=0; i<3; i++) 
        {
            ring->n[i] /= nn;
            ring->p[i] = c[i] + tun->radiusCurve*ring->n[i]; // - tr1[i];  // p=pos-tr1
        }

        // make t (tangeant) be normal to n (normal)
        xQ = ring->t[0]*ring->n[0]+ring->t[1]*ring->n[1]+ring->t[2]*ring->n[2];
        for (int i=0; i<3; i++) 
            ring->t[i] -= xQ*ring->n[i];

        // normalize t
        nn = sqrtf(ring->t[0]*ring->t[0] + ring->t[1]*ring->t[1] + ring->t[2]*ring->t[2]);
        for (int i=0; i<3; i++) 
            ring->t[i] /= nn;

        ring->v[0] = ring->n[1]*ring->t[2]-ring->n[2]*ring->t[1];
        ring->v[1] = ring->n[2]*ring->t[0]-ring->n[0]*ring->t[2];
        ring->v[2] = ring->n[0]*ring->t[1]-ring->n[1]*ring->t[0];
    }
    else
    {
        ring->insideHole = false;

        // normalize pos and make n be pos
        yQ = sqrtf(ring->p[0]*ring->p[0]+ring->p[1]*ring->p[1]+ring->p[2]*ring->p[2]);
        for (int i=0; i<3; i++) 
        {
            ring->p[i] /= yQ;
            ring->n[i] = ring->p[i];
            //ring->p[i] = pos[i] - this->tr1[i];
        }

        // make t (tangeant) be normal to n (normal)
        xQ = ring->t[0]*ring->n[0]+ring->t[1]*ring->n[1]+ring->t[2]*ring->n[2];
        for (int i=0; i<3; i++) 
            ring->t[i] -= xQ*ring->n[i];

        // normalize t
        float nn = sqrtf(ring->t[0]*ring->t[0] + ring->t[1]*ring->t[1] + ring->t[2]*ring->t[2]);
        for (int i=0; i<3; i++) 
            ring->t[i] /= nn;

        ring->v[0] = ring->n[1]*ring->t[2]-ring->n[2]*ring->t[1];
        ring->v[1] = ring->n[2]*ring->t[0]-ring->n[0]*ring->t[2];
        ring->v[2] = ring->n[0]*ring->t[1]-ring->n[1]*ring->t[0];
    }

    ring->planet = this->planet1;
}

void Snake::computeVertices(Ring * ring)
{
    for (int i=0; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            ring->normalCoordonates[3*i + j] = - Snake::trigo[2*i] * ring->n[j]
                - Snake::trigo[2*i+1] * ring->v[j];
            ring->verticesCoordonates[3*i + j] = ring->p[j] //+Snake::trigo[2*i] * ring->v[j];
                + SNAKE_RADIUS * (0.7f*ring->n[j] + ring->normalCoordonates[3*i + j]);
        }
    }
}

void Snake::update(World * w)
{
    //if (this->detectImpact(w->snakes)) 
    //    return;

    int previousRing = this->firstRing;
    this->firstRing = (this->firstRing+1) % (SNAKE_LENGTH * SNAKE_SPEED);

    Ring * ring = this->rings + (this->firstRing/SNAKE_SPEED);
    Ring * pRing = this->rings + (previousRing/SNAKE_SPEED);

    if (! (this->firstRing % SNAKE_SPEED))
    {
        for (int i=0; i<3; i++)
            this->tail[i] = ring->p[i];
    }

    for (int i=0; i<3; i++) 
    {
        ring->t[i] = pRing->t[i];
        ring->p[i] = pRing->p[i] + (SNAKE_RING_LENGTH / SNAKE_SPEED) * pRing->t[i];
    }

    this->positionNextRing(ring, pRing->face, w);
    this->computeVertices(ring);

    if (! (this->firstRing % SNAKE_SPEED))
    {
        this->alpha = (this->alpha + 1) % SNAKE_PULSE;
    }

    for (int j=0; j<3; j++)
    {
        ring->t[j] += ((float) SNAKE_AMPLITUDE / SNAKE_SPEED) * cosalpha[this->alpha] * ring->v[j];
    }
}

void Snake::translateTail(float * tr)
{
    for (int i=0; i<SNAKE_LENGTH; i++)
    {
        Ring * ring = this->rings + i;
        for (int j=0; j<3; j++)
        {
            ring->p[j] -= tr[j];
            for (int k=0; k<SNAKE_TESSELIZATION; k++)
                ring->verticesCoordonates[3*k + j] -= tr[j];
        }
    }
}

int Snake::detectImpact(Star * star)
{
    // move star to the referentiel of the snake.
    float p1[3], p2[3];
    if (star->planet == this->planet1)
        for (int i=0; i<3; i++)
            p1[i] = star->position[i];
    else if (star->planet == this->planet2)
        for (int i=0; i<3; i++)
            p1[i] = star->position[i] - this->tr[i];
    else return NO_IMPACT;

    // if p1 is too far from the whole snake, return (no impact)
    Ring * ring = this->rings + ((this->firstRing/SNAKE_SPEED + SNAKE_LENGTH/2) % SNAKE_LENGTH);
    float d = 0;
    for (int i=0; i<3; i++)
    {
        float dd = p1[i] - ring->p[i];
        d += dd * dd;
    }
    float length = (SNAKE_LENGTH/2) * SNAKE_RING_LENGTH;
    if (d > length * length) return NO_IMPACT;

    // p1 is the current position and p2 the next position (actually a little further).
    float dp = SNAKE_RADIUS + star->dq;
    for (int i=0; i<3; i++) p2[i] = p1[i] + dp * star->t[i];
    dp *= dp;

    // for each ring of the snake, compare the position of p1 and p2 relative to the snake
    // if they are on both sides, then flag a collision
    int s = (this->firstRing/SNAKE_SPEED) % SNAKE_LENGTH; // one before last ring (tail)
    for (int i0=0; i0<SNAKE_LENGTH; i0++)
    {
        s = (s+1) % SNAKE_LENGTH;
        if (this->rings[s].planet != star->planet) continue;

        float * p0 = this->rings[s].p;
        float * v = this->rings[s].v;

        d = 0;
        float ps1 = 0;
        float ps2 = 0;

        for (int i=0; i<3; i++)
        {
            float dd = p1[i] - p0[i];
            d += dd * dd;
        }

        if (d > dp) continue; // to far from token to consider impact

        for (int i=0; i<3; i++)
        {
            ps1 += (p1[i] - p0[i]) * v[i];
            ps2 += (p2[i] - p0[i]) * v[i];
        }

        if (ps1 <= 0 && ps2 > 0) return s;
        if (ps1 > 0 && ps2 <= 0) return s;
    }

    return NO_IMPACT;
}

int Snake::closestRing(float * p0, int r0)
{
    // dichotomic search of the closest ring to p in the 2*k closest neighboors of r0
    int k = 8; // 2^N
    while (k>0)
    {
        float * p1 = this->rings[(r0 - k) % SNAKE_LENGTH].p;
        float * p2 = this->rings[(r0 + k) % SNAKE_LENGTH].p;

        float d1 = 0;
        float d2 = 0;
        for (int i=0; i<3; i++)
        {
            float dd = p1[i] - p0[i];
            d1 += dd * dd;
            dd = p2[i] - p0[i];
            d2 += dd * dd;
        }

        k /= 2;

        if (d1 < d2) r0 -= k;
        else r0 += k;
    }

    return r0;
}

bool Snake::detectImpact(Snake ** snakes)
{
    // p: the center of this snake's head
    Ring * ring = this->rings + ((this->firstRing/SNAKE_SPEED) % SNAKE_LENGTH);
    float p[3];
    for (int i=0; i<3; i++)
        p[i] = ring->p[i] + 3 * SNAKE_HEAD_LENGTH * ring->t[i];

    float r = 6 * SNAKE_HEAD_LENGTH + SNAKE_RADIUS; r *= r;

    int pl;
    for (int s=0; s<NUMBER_OF_SNAKES; s++)
    {
        Snake * that = snakes[s];

        if (that == this) 
            continue;

        if (that->planet1 == this->planet1)
        {
            pl = this->planet1;
        }
        else if (that->planet1 == this->planet2)
        {
            pl = this->planet2;
            for (int i=0; i<3; i++)
                p[i] -= this->tr[i];
        }
        else if (that->planet2 == this->planet1)
        {
            pl = this->planet1;
            for (int i=0; i<3; i++)
                p[i] += that->tr[i];
        }
        else if (that->planet2 == this->planet2)
        {
            pl = this->planet2;
            for (int i=0; i<3; i++)
                p[i] += that->tr[i] - this->tr[i];
        }
        else continue;

        if (that->detectImpact(p, r, pl)) 
            return true;
    }
    return false;
}

bool Snake::detectImpact(float * p, float r, int pl)
{
    // if p is too far from the whole snake, return false (no impact)
    Ring * ring = this->rings + ((this->firstRing/SNAKE_SPEED + SNAKE_LENGTH/2) % SNAKE_LENGTH);
    float d = 0;
    for (int i=0; i<3; i++)
    {
        float dd = p[i] - ring->p[i];
        d += dd * dd;
    }
    float length = (SNAKE_LENGTH/2) * SNAKE_RING_LENGTH;
    if (d > length * length) return false;

    int s = this->firstRing/SNAKE_SPEED % SNAKE_LENGTH; // one before last ring (tail)
    for (int i0=0; i0<SNAKE_LENGTH; i0++)
    {
        s = (s+1) % SNAKE_LENGTH;
        if (this->rings[s].planet != pl) continue;

        float * q = this->rings[s].p;
        float d = 0;
        for (int i=0; i<3; i++)
        {
            float dd = p[i]-q[i];
            d += dd * dd;
        }
        if (d<r) return true;
    }
    return false;
}

bool Snake::detectHeadImpact(float * p, float r)
{
    // h: the center of this snake's head
    Ring * ring = this->rings + ((this->firstRing/SNAKE_SPEED) % SNAKE_LENGTH);
    float h[3];
    for (int i=0; i<3; i++)
        h[i] = ring->p[i] + 3 * SNAKE_HEAD_LENGTH * ring->t[i];

    float d = 0;
    for (int i=0; i<3; i++)
    {
        float dd = p[i] - h[i];
        d += dd * dd;
    }

    if (d > r) return false;
    else return true;
}

bool Snake::isOnPlanet(int pl)
{
    // look for pl in rings; from tail to head for efficiency
    int i = (this->firstRing/SNAKE_SPEED) % SNAKE_LENGTH; // one before last ring (tail)
    for (int i0=0; i0<SNAKE_LENGTH - 2; i0++)
    {
        i = (i+1) % SNAKE_LENGTH;
        if (this->rings[i].planet == pl) return true;
    }

    if (pl == this->planet2 ) this->planet2 = this->planet1;
    return false;
}

bool Snake::isOnPlanet(int pl, int fc)
{
    // look for pl in rings; from tail to head for efficiency
    int i = (this->firstRing/SNAKE_SPEED + 1) % SNAKE_LENGTH; // last ring (tail)
    for (int i0=0; i0<SNAKE_LENGTH - 2; i0++)
    {
        i = (i+1) % SNAKE_LENGTH;
        if (this->rings[i].planet == pl && this->rings[i].face == fc) return true;
    }
    return false;
}

void Snake::draw(World * w, bool isDistant)
{
    glDisable(GL_CULL_FACE);

    if (w->wire)
    {
        glColor3f(1, 0.53f, 0.17f);
        glBindTexture(GL_TEXTURE_2D, w->whiteTextureId);
        glLineWidth(3);

        int i1, i2;
        float lCoord1, lCoord2;
        float r1, r2, dr;

        // draw the extreme end of the tail
        i2 = (this->firstRing/SNAKE_SPEED + 1) % SNAKE_LENGTH; // last ring (tail)
        lCoord2 = (SNAKE_SPEED-((float)(this->firstRing % SNAKE_SPEED))) * Snake::dl;
        r1 = ((float)(SNAKE_SPEED - (this->firstRing % SNAKE_SPEED)))/SNAKE_SPEED;
        r2 = 0.1f*r1;
        {
            glBegin(GL_LINE_STRIP);
            glVertex3f(r1*this->tail[0] + (1-r1)*this->rings[i2].p[0],
                    r1*this->tail[1] + (1-r1)*this->rings[i2].p[1],
                    r1*this->tail[2] + (1-r1)*this->rings[i2].p[2]);
            for (int j=1; j < SNAKE_TESSELIZATION-1; j++)
            {
                glNormal3fv(this->rings[i2].normalCoordonates + 3*j);
                glVertex3f(
                    (r2*this->rings[i2].verticesCoordonates[3*j] + (1-r2)*this->rings[i2].p[0]),
                    (r2*this->rings[i2].verticesCoordonates[3*j+1] + (1-r2)*this->rings[i2].p[1]),
                    (r2*this->rings[i2].verticesCoordonates[3*j+2] + (1-r2)*this->rings[i2].p[2]));

                glNormal3fv(this->rings[i2].normalCoordonates + 3*(j+1));
                glVertex3f(
                    (r2*this->rings[i2].verticesCoordonates[3*(j+1)] + (1-r2)*this->rings[i2].p[0]),
                    (r2*this->rings[i2].verticesCoordonates[3*(j+1)+1] + (1-r2)*this->rings[i2].p[1]),
                    (r2*this->rings[i2].verticesCoordonates[3*(j+1)+2] + (1-r2)*this->rings[i2].p[2]));

                glVertex3f(r1*this->tail[0] + (1-r1)*this->rings[i2].p[0],
                        r1*this->tail[1] + (1-r1)*this->rings[i2].p[1],
                        r1*this->tail[2] + (1-r1)*this->rings[i2].p[2]);
            }
            glEnd();
            glEnd();
        }

        // draw the tail (narrowing part of the body)
        dr = 0.9f/(SNAKE_TAIL_LENGTH-1);
        for (int i0=0; i0<SNAKE_TAIL_LENGTH; i0++)
        {
            i1 = i2;
            i2 = (i1 + 1) % SNAKE_LENGTH;
            lCoord1 = lCoord2;
            lCoord2 += SNAKE_SPEED * Snake::dl;
            r1 = r2;
            r2 += dr;
            if (r2>1) r2 = 1;

            glBegin(GL_LINE_STRIP);
            for (int j=1; j < SNAKE_TESSELIZATION-1; j++)
            {
                glNormal3fv(this->rings[i1].normalCoordonates + 3*j);
                glVertex3f(
                    (r1*this->rings[i1].verticesCoordonates[3*j] + (1-r1)*this->rings[i1].p[0]),
                    (r1*this->rings[i1].verticesCoordonates[3*j+1] + (1-r1)*this->rings[i1].p[1]),
                    (r1*this->rings[i1].verticesCoordonates[3*j+2] + (1-r1)*this->rings[i1].p[2]));

                glNormal3fv(this->rings[i2].normalCoordonates + 3*j);
                glVertex3f(
                    (r2*this->rings[i2].verticesCoordonates[3*j] + (1-r2)*this->rings[i2].p[0]),
                    (r2*this->rings[i2].verticesCoordonates[3*j+1] + (1-r2)*this->rings[i2].p[1]),
                    (r2*this->rings[i2].verticesCoordonates[3*j+2] + (1-r2)*this->rings[i2].p[2]));

                glNormal3fv(this->rings[i2].normalCoordonates + 3*(j+1));
                glVertex3f(
                    (r2*this->rings[i2].verticesCoordonates[3*(j+1)] + (1-r2)*this->rings[i2].p[0]),
                    (r2*this->rings[i2].verticesCoordonates[3*(j+1)+1] + (1-r2)*this->rings[i2].p[1]),
                    (r2*this->rings[i2].verticesCoordonates[3*(j+1)+2] + (1-r2)*this->rings[i2].p[2]));
            }
            glEnd();
        }

        // draw body from tail to head
        for (int i0=SNAKE_TAIL_LENGTH; i0<SNAKE_LENGTH - 2; i0++)
        {
            i1 = i2;
            i2 = (i1 + 1) % SNAKE_LENGTH;
            lCoord1 = lCoord2;
            lCoord2 += SNAKE_SPEED * Snake::dl;

            glBegin(GL_LINE_STRIP);
            for (int j=1; j < SNAKE_TESSELIZATION-1; j++)
            {
                glNormal3fv(this->rings[i1].normalCoordonates + 3*j);
                glVertex3fv(this->rings[i1].verticesCoordonates + 3*j);

                glNormal3fv(this->rings[i2].normalCoordonates + 3*j);
                glVertex3fv(this->rings[i2].verticesCoordonates + 3*j);

                glNormal3fv(this->rings[i2].normalCoordonates + 3*(j+1));
                glVertex3fv(this->rings[i2].verticesCoordonates + 3*(j+1));
            }
            glEnd();
        }

        // draw neck now (the texture coordonates are differente since the ring is shorter)
        {
            i1 = i2;
            i2 = (i1 + 1) % SNAKE_LENGTH;
            lCoord1 = lCoord2;
            lCoord2 += ((float)(this->firstRing % SNAKE_SPEED)) * Snake::dl;
            glBegin(GL_LINE_STRIP);
            for (int j=1; j < SNAKE_TESSELIZATION-1; j++)
            {
                glNormal3fv(this->rings[i1].normalCoordonates + 3*j);
                glVertex3fv(this->rings[i1].verticesCoordonates + 3*j);

                glNormal3fv(this->rings[i2].normalCoordonates + 3*j);
                glVertex3fv(this->rings[i2].verticesCoordonates + 3*j);

                glNormal3fv(this->rings[i2].normalCoordonates + 3*(j+1));
                glVertex3fv(this->rings[i2].verticesCoordonates + 3*(j+1));
            }
            glEnd();
        }

        // finally draw the head
        this->drawHeadW(this->rings + i2, lCoord2);
        this->drawEyesW(this->rings + i2, w);

        glLineWidth(1);
    }
    else
    {
        glColor3f(1, 1, 1);
        if (isDistant)
            glBindTexture(GL_TEXTURE_2D, w->snakeDistantTextureId);
        else
            glBindTexture(GL_TEXTURE_2D, w->snakeTextureId);

        int i1, i2;
        float lCoord1, lCoord2;
        float r1, r2, dr;

        // draw the extreme end of the tail
        i2 = (this->firstRing/SNAKE_SPEED + 1) % SNAKE_LENGTH; // last ring (tail)
        lCoord1 = 0;
        lCoord2 = (SNAKE_SPEED-((float)(this->firstRing % SNAKE_SPEED))) * Snake::dl;
        r1 = ((float)(SNAKE_SPEED - (this->firstRing % SNAKE_SPEED)))/SNAKE_SPEED;
        r2 = 0.1f*r1;
        {
            glBegin(GL_TRIANGLE_FAN);
            glTexCoord2f(lCoord1, this->textureCoordonates[0]);
            glVertex3f(r1*this->tail[0] + (1-r1)*this->rings[i2].p[0],
                    r1*this->tail[1] + (1-r1)*this->rings[i2].p[1],
                    r1*this->tail[2] + (1-r1)*this->rings[i2].p[2]);
            for (int j=1; j < SNAKE_TESSELIZATION; j++)
            {
                glTexCoord2f(lCoord2, this->textureCoordonates[j]);
                glNormal3fv(this->rings[i2].normalCoordonates + 3*j);
                glVertex3f(
                    (r2*this->rings[i2].verticesCoordonates[3*j] + (1-r2)*this->rings[i2].p[0]),
                    (r2*this->rings[i2].verticesCoordonates[3*j+1] + (1-r2)*this->rings[i2].p[1]),
                    (r2*this->rings[i2].verticesCoordonates[3*j+2] + (1-r2)*this->rings[i2].p[2]));
            }
            glEnd();
            glEnd();
        }

        // draw the tail (narrowing part of the body)
        dr = 0.9f/(SNAKE_TAIL_LENGTH-1);
        for (int i0=0; i0<SNAKE_TAIL_LENGTH; i0++)
        {
            i1 = i2;
            i2 = (i1 + 1) % SNAKE_LENGTH;
            lCoord1 = lCoord2;
            lCoord2 += SNAKE_SPEED * Snake::dl;
            r1 = r2;
            r2 += dr;
            if (r2>1) r2 = 1;

            glBegin(GL_QUAD_STRIP);
            for (int j=1; j < SNAKE_TESSELIZATION; j++)
            {
                glTexCoord2f(lCoord1, this->textureCoordonates[j]);
                glNormal3fv(this->rings[i1].normalCoordonates + 3*j);
                glVertex3f(
                    (r1*this->rings[i1].verticesCoordonates[3*j] + (1-r1)*this->rings[i1].p[0]),
                    (r1*this->rings[i1].verticesCoordonates[3*j+1] + (1-r1)*this->rings[i1].p[1]),
                    (r1*this->rings[i1].verticesCoordonates[3*j+2] + (1-r1)*this->rings[i1].p[2]));

                glTexCoord2f(lCoord2, this->textureCoordonates[j]);
                glNormal3fv(this->rings[i2].normalCoordonates + 3*j);
                glVertex3f(
                    (r2*this->rings[i2].verticesCoordonates[3*j] + (1-r2)*this->rings[i2].p[0]),
                    (r2*this->rings[i2].verticesCoordonates[3*j+1] + (1-r2)*this->rings[i2].p[1]),
                    (r2*this->rings[i2].verticesCoordonates[3*j+2] + (1-r2)*this->rings[i2].p[2]));
            }
            glEnd();
        }

        // draw body from tail to head
        for (int i0=SNAKE_TAIL_LENGTH; i0<SNAKE_LENGTH - 2; i0++)
        {
            i1 = i2;
            i2 = (i1 + 1) % SNAKE_LENGTH;
            lCoord1 = lCoord2;
            lCoord2 += SNAKE_SPEED * Snake::dl;
            //if (this->rings[i1].insideHole || this->rings[i2].insideHole)
                //(this->rings[i1].planet == pl || this->rings[i2].planet == pl)
            {
                glBegin(GL_QUAD_STRIP);
                for (int j=1; j < SNAKE_TESSELIZATION; j++)
                {
                    glTexCoord2f(lCoord1, this->textureCoordonates[j]);
                    glNormal3fv(this->rings[i1].normalCoordonates + 3*j);
                    glVertex3fv(this->rings[i1].verticesCoordonates + 3*j);

                    glTexCoord2f(lCoord2, this->textureCoordonates[j]);
                    glNormal3fv(this->rings[i2].normalCoordonates + 3*j);
                    glVertex3fv(this->rings[i2].verticesCoordonates + 3*j);
                }
            }
            glEnd();
        }

        // draw neck now (the texture coordonates are differente since the ring is shorter)
        {
            i1 = i2;
            i2 = (i1 + 1) % SNAKE_LENGTH;
            lCoord1 = lCoord2;
            lCoord2 += ((float)(this->firstRing % SNAKE_SPEED)) * Snake::dl;
            glBegin(GL_QUAD_STRIP);
            for (int j=1; j < SNAKE_TESSELIZATION; j++)
            {
                glTexCoord2f(lCoord1, this->textureCoordonates[j]);
                glNormal3fv(this->rings[i1].normalCoordonates + 3*j);
                glVertex3fv(this->rings[i1].verticesCoordonates + 3*j);

                glTexCoord2f(lCoord2, this->textureCoordonates[j]);
                glNormal3fv(this->rings[i2].normalCoordonates + 3*j);
                glVertex3fv(this->rings[i2].verticesCoordonates + 3*j);
            }
            glEnd();
        }

        // finally draw the head
        this->drawHead(this->rings + i2, lCoord2);
        this->drawEyes(this->rings + i2, w);
    }

    glEnable(GL_CULL_FACE);
}

void Snake::drawHead(Ring * ring, float lCoord2)
{
    float head[6*SNAKE_TESSELIZATION];
    for (int i=0; i<SNAKE_TESSELIZATION/2; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = ring->verticesCoordonates[3*i+j];
            head[6*i+3+j] = ring->verticesCoordonates[3*i+j] 
                + SNAKE_HEAD_LENGTH * (ring->t[j] - ring->v[j]);
        }
    }
    for (int i=SNAKE_TESSELIZATION/2; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = ring->verticesCoordonates[3*i+j];
            head[6*i+3+j] = ring->verticesCoordonates[3*i+j] 
                + SNAKE_HEAD_LENGTH * (ring->t[j] + ring->v[j]);
        }
    }
    float lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 1.1f * Snake::dl; //SNAKE_HEAD_LENGTH;
    glBegin(GL_QUAD_STRIP);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glTexCoord2f(lCoord2, this->textureCoordonates[i]);
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i);
        glTexCoord2f(lCoord1, this->textureCoordonates[i]);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();

    for (int i=0; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = head[6*i+j] + SNAKE_HEAD_LENGTH * ring->t[j] ;
        }
    }
    lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 0.66f * Snake::dl;
    glBegin(GL_QUAD_STRIP);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glTexCoord2f(lCoord1, this->textureCoordonates[i]);
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i);
        glTexCoord2f(lCoord2, this->textureCoordonates[i]);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();

    for (int i=0; i<SNAKE_TESSELIZATION/2; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.7f * (ring->verticesCoordonates[3*i+j] 
                - 1.2f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.3f * ring->p[j] + 3 * SNAKE_HEAD_LENGTH * ring->t[j] ;
        }
    }
    for (int i=SNAKE_TESSELIZATION/2; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.7f * (ring->verticesCoordonates[3*i+j] 
                + 1.2f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.3f * ring->p[j] + 3 * SNAKE_HEAD_LENGTH * ring->t[j] ;
        }
    }
    lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 1.2f * Snake::dl;
    glBegin(GL_QUAD_STRIP);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glTexCoord2f(lCoord1, this->textureCoordonates[i]);
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i);
        glTexCoord2f(lCoord2, this->textureCoordonates[i]);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();

    for (int i=0; i<SNAKE_TESSELIZATION/2; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.6f * (ring->verticesCoordonates[3*i+j] 
                - 0.9f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.4f * ring->p[j] + 4 * SNAKE_HEAD_LENGTH * ring->t[j] ;
        }
    }
    for (int i=SNAKE_TESSELIZATION/2; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.6f * (ring->verticesCoordonates[3*i+j] 
                + 0.9f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.4f * ring->p[j] + 4 * SNAKE_HEAD_LENGTH * ring->t[j] ;
        }
    }
    lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 0.5f * Snake::dl;
    glBegin(GL_QUAD_STRIP);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glTexCoord2f(lCoord2, this->textureCoordonates[i]);
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i);
        glTexCoord2f(lCoord1, this->textureCoordonates[i]);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();

    for (int i=0; i<SNAKE_TESSELIZATION/2; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.5f * (ring->verticesCoordonates[3*i+j] 
                - 0.9f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.5f * ring->p[j] + 5 * SNAKE_HEAD_LENGTH * ring->t[j] 
                + SNAKE_HEAD_HEIGHT * ring->n[j];
        }
    }
    for (int i=SNAKE_TESSELIZATION/2; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.5f * (ring->verticesCoordonates[3*i+j] 
                + 0.9f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.5f * ring->p[j] + 5 * SNAKE_HEAD_LENGTH * ring->t[j] 
                + SNAKE_HEAD_HEIGHT * ring->n[j];
        }
    }
    lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 0.5f * Snake::dl;
    glBegin(GL_QUAD_STRIP);
    for (int i=0; i < SNAKE_TESSELIZATION; i++)
    {
        glTexCoord2f(lCoord2, this->textureCoordonates[i]);
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i);
        glTexCoord2f(lCoord1, this->textureCoordonates[i]);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();

    for (int j=0; j<3; j++)
    {
        head[j] = ring->p[j] + 5.2f * SNAKE_HEAD_LENGTH * ring->t[j] 
            + 5*SNAKE_HEAD_HEIGHT * ring->n[j];
        //head[9+j] = 0.004f * (ring->p[j] + ring->v[j]) 
        //    + 0.996f * ring->p[j] + 6 * SNAKE_HEAD_LENGTH * ring->t[j] 
        //    + 6*SNAKE_HEAD_HEIGHT * ring->n[j];
    }
    lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 0.5f * Snake::dl;
    glBegin(GL_TRIANGLE_FAN);
    glVertex3fv(head);
    for (int i=0; i < SNAKE_TESSELIZATION; i++)
    {
        glTexCoord2f(lCoord2, this->textureCoordonates[i]);
        glNormal3fv(ring->normalCoordonates + 3*i);
        glTexCoord2f(lCoord1, this->textureCoordonates[i]);
        glVertex3fv(head + 6*i+3);
    }
    glVertex3fv(head + 3);
    glEnd();
}
void Snake::drawHeadW(Ring * ring, float lCoord2)
{
    float head[6*SNAKE_TESSELIZATION];
    for (int i=0; i<SNAKE_TESSELIZATION/2; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = ring->verticesCoordonates[3*i+j];
            head[6*i+3+j] = ring->verticesCoordonates[3*i+j] 
                + SNAKE_HEAD_LENGTH * (ring->t[j] - ring->v[j]);
        }
    }
    for (int i=SNAKE_TESSELIZATION/2; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = ring->verticesCoordonates[3*i+j];
            head[6*i+3+j] = ring->verticesCoordonates[3*i+j] 
                + SNAKE_HEAD_LENGTH * (ring->t[j] + ring->v[j]);
        }
    }
    float lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 1.1f * Snake::dl; //SNAKE_HEAD_LENGTH;
    glBegin(GL_LINES);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();
    glBegin(GL_LINE_LOOP);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();

    for (int i=0; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = head[6*i+j] + SNAKE_HEAD_LENGTH * ring->t[j] ;
        }
    }
    lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 0.66f * Snake::dl;
    glBegin(GL_LINES);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();
    glBegin(GL_LINE_LOOP);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();

    for (int i=0; i<SNAKE_TESSELIZATION/2; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.7f * (ring->verticesCoordonates[3*i+j] 
                - 1.2f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.3f * ring->p[j] + 3 * SNAKE_HEAD_LENGTH * ring->t[j] ;
        }
    }
    for (int i=SNAKE_TESSELIZATION/2; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.7f * (ring->verticesCoordonates[3*i+j] 
                + 1.2f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.3f * ring->p[j] + 3 * SNAKE_HEAD_LENGTH * ring->t[j] ;
        }
    }
    lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 1.2f * Snake::dl;
    glBegin(GL_LINES);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();
    glBegin(GL_LINE_LOOP);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();

    for (int i=0; i<SNAKE_TESSELIZATION/2; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.6f * (ring->verticesCoordonates[3*i+j] 
                - 0.9f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.4f * ring->p[j] + 4 * SNAKE_HEAD_LENGTH * ring->t[j] ;
        }
    }
    for (int i=SNAKE_TESSELIZATION/2; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.6f * (ring->verticesCoordonates[3*i+j] 
                + 0.9f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.4f * ring->p[j] + 4 * SNAKE_HEAD_LENGTH * ring->t[j] ;
        }
    }
    lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 0.5f * Snake::dl;
    glBegin(GL_LINES);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();
    glBegin(GL_LINE_LOOP);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();

    for (int i=0; i<SNAKE_TESSELIZATION/2; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.5f * (ring->verticesCoordonates[3*i+j] 
                - 0.9f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.5f * ring->p[j] + 5 * SNAKE_HEAD_LENGTH * ring->t[j] 
                + SNAKE_HEAD_HEIGHT * ring->n[j];
        }
    }
    for (int i=SNAKE_TESSELIZATION/2; i<SNAKE_TESSELIZATION; i++)
    {
        for (int j=0; j<3; j++)
        {
            head[6*i+j] = head[6*i+3+j];
            head[6*i+3+j] = 0.5f * (ring->verticesCoordonates[3*i+j] 
                + 0.9f * SNAKE_HEAD_LENGTH * ring->v[j]) 
                + 0.5f * ring->p[j] + 5 * SNAKE_HEAD_LENGTH * ring->t[j] 
                + SNAKE_HEAD_HEIGHT * ring->n[j];
        }
    }
    lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 0.5f * Snake::dl;
    glBegin(GL_LINES);
    for (int i=0; i < SNAKE_TESSELIZATION; i++)
    {
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();
    glBegin(GL_LINE_LOOP);
    for (int i=1; i < SNAKE_TESSELIZATION; i++)
    {
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i + 3);
    }
    glEnd();

    for (int j=0; j<3; j++)
    {
        head[j] = ring->p[j] + 5.2f * SNAKE_HEAD_LENGTH * ring->t[j] 
            + 5*SNAKE_HEAD_HEIGHT * ring->n[j];
        //head[9+j] = 0.004f * (ring->p[j] + ring->v[j]) 
        //    + 0.996f * ring->p[j] + 6 * SNAKE_HEAD_LENGTH * ring->t[j] 
        //    + 6*SNAKE_HEAD_HEIGHT * ring->n[j];
    }
    lCoord1 = lCoord2;
    lCoord2 += SNAKE_SPEED * 0.5f * Snake::dl;
    glBegin(GL_LINES);
    for (int i=0; i < SNAKE_TESSELIZATION; i++)
    {
        glVertex3fv(head);
        glNormal3fv(ring->normalCoordonates + 3*i);
        glVertex3fv(head + 6*i+3);
    }
    glVertex3fv(head + 3);
    glEnd();
}

void Snake::drawEyes(Ring * ring, World * w)
{
    glPushMatrix();
    float * p = ring->p;
    float * n = ring->n;
    float * v = ring->v;
    float * t = ring->t;
    glTranslatef(p[0]+SNAKE_EYES_HEIGHT*n[0] + 2*SNAKE_HEAD_LENGTH*t[0] - SNAKE_EYES_PARALAX*v[0], 
                 p[1]+SNAKE_EYES_HEIGHT*n[1] + 2*SNAKE_HEAD_LENGTH*t[1] - SNAKE_EYES_PARALAX*v[1], 
                 p[2]+SNAKE_EYES_HEIGHT*n[2] + 2*SNAKE_HEAD_LENGTH*t[2] - SNAKE_EYES_PARALAX*v[2]);
    Star::drawExplosion(w, 0.4f, 0.4f, 0.4f, SNAKE_EYES_RADIUS, 1, w->snakeDistantTextureId);
    glPopMatrix();

    glPushMatrix();
    glTranslatef(p[0]+SNAKE_EYES_HEIGHT*n[0] + 2*SNAKE_HEAD_LENGTH*t[0] + SNAKE_EYES_PARALAX*v[0], 
                 p[1]+SNAKE_EYES_HEIGHT*n[1] + 2*SNAKE_HEAD_LENGTH*t[1] + SNAKE_EYES_PARALAX*v[1], 
                 p[2]+SNAKE_EYES_HEIGHT*n[2] + 2*SNAKE_HEAD_LENGTH*t[2] + SNAKE_EYES_PARALAX*v[2]);
    Star::drawExplosion(w, 0.4f, 0.4f, 0.4f, SNAKE_EYES_RADIUS, 1, w->snakeDistantTextureId);
    glPopMatrix();
}

void Snake::drawEyesW(Ring * ring, World * w)
{
    glPushMatrix();
    float * p = ring->p;
    float * n = ring->n;
    float * v = ring->v;
    float * t = ring->t;
    glTranslatef(p[0]+SNAKE_EYES_HEIGHT*n[0] + 2*SNAKE_HEAD_LENGTH*t[0] - SNAKE_EYES_PARALAX*v[0], 
                 p[1]+SNAKE_EYES_HEIGHT*n[1] + 2*SNAKE_HEAD_LENGTH*t[1] - SNAKE_EYES_PARALAX*v[1], 
                 p[2]+SNAKE_EYES_HEIGHT*n[2] + 2*SNAKE_HEAD_LENGTH*t[2] - SNAKE_EYES_PARALAX*v[2]);
    Star::drawExplosionW(w, 0.4f, 0.53f * 0.4f, 0.17f * 0.4f, SNAKE_EYES_RADIUS);
    glPopMatrix();

    glPushMatrix();
    glTranslatef(p[0]+SNAKE_EYES_HEIGHT*n[0] + 2*SNAKE_HEAD_LENGTH*t[0] + SNAKE_EYES_PARALAX*v[0], 
                 p[1]+SNAKE_EYES_HEIGHT*n[1] + 2*SNAKE_HEAD_LENGTH*t[1] + SNAKE_EYES_PARALAX*v[1], 
                 p[2]+SNAKE_EYES_HEIGHT*n[2] + 2*SNAKE_HEAD_LENGTH*t[2] + SNAKE_EYES_PARALAX*v[2]);
    Star::drawExplosionW(w, 0.4f, 0.53f * 0.4f, 0.17f * 0.4f, SNAKE_EYES_RADIUS);
    glPopMatrix();
}