#include "Missile.h"
#include "World.h"


Missile::Missile(float radius, float x, float y, float z, float xt, float yt, float zt, int pl, World * w)
: Star(radius, x, y, z, xt, yt, zt, pl, w)
{
    this->isAlive = false;
    this->explosionFactor = 0;
    this->dq = INITIAL_MISSILE_SPEED;
    this->radiusInt = 1.0f;

    this->age = 0;
    this->tailOrigin = 0;
    this->tailLength = -1;

    for (int i=0; i<3; i++) this->tr[i] = 0;

    for (int i=0; i<3; i++)
    {
        this->c1[i] = 0.3f;
        this->c2[i] = 0.3f;
        this->c3[i] = 1.0f;
    }
    this->c3[3] = 0.2f;

    this->drivebySound = NUMBER_OF_SOUNDS_P - 1;

    this->update(w);
}

Missile::~Missile(void)
{
}

void Missile::reset(World * w)
{
    this->isAlive = true;
    this->explosionFactor = 0;
    this->planet = w->currentPlanet;
    this->face = w->currentFace;
    for (int j=0; j<3; j++)
    {
        this->q[j] = w->p[j] + 0.05f*w->t[j];
        this->t[j] = w->t[j];
        this->v[j] = w->v[j];
    }
    this->age = -30;
    this->tailLength = 0;
    if (w->dp >0)
        this->dq = INITIAL_MISSILE_SPEED + w->dp;
    else
        this->dq = INITIAL_MISSILE_SPEED;
}

void Missile::translateTail()
{
    for (int t=0; t<this->tailLength; t++)
    {
        struct TailItem * ti = this->tail + (this->tailOrigin + TAIL_LENGTH - t) % TAIL_LENGTH;
        for (int j=0; j<3; j++)
            ti->p[j] -= this->tr[j];
    }
}

bool Missile::isAtImpact(World * w)
{
    if (!this->isAlive) return false;
    if (this->planet != w->currentPlanet) return false;
    if (this->explosionFactor != 0) return false;

    float d = 0;
    for (int i=0; i<3; i++)
    {
        float dd = this->q[i] - w->p[i];
        d += dd*dd;
    }
    float R = this->radius + w->player->radius;
    if (d > R*R) 
    {
        if (this->age > 30)
        {
            int volume = MY_SDL_MIX_MAXVOLUME - (int)(200 * d);
            if (volume > 0) 
            {
                this->drivebySound = w->playSound(SOUND_DRIVEBY, volume);
                this->age = 0;
            }
        }
        return false;
    }

    w->player->health -= 8;
    if (w->player->health < 0) 
    {
        w->player->health = 0;
        w->gameIsOver = 10;
    }
    return true;
}

bool Missile::checkAllTargetsForImpact(World * w)
{
    if (!this->isAlive || this->explosionFactor != 0) return false; 

    for (int i=0; i<NUMBER_OF_TARGETS; i++)
    {
        Target * tr = w->targets[i];
        if (tr->isAlive
            && tr->planet == this->planet
            && tr->explosionFactor == 0)
        {
            float d = 0;
            for (int i=0; i<3; i++)
            {
                float dd = this->q[i] - tr->q[i];
                d += dd*dd;
            }
            float R = this->radius + tr->radius;
            if (d < R*R) 
            {
                if (tr->spikiness > SPIKINESS_AFTER_EXPLOSION + DELTA_SPIKINESS_WHEN_EXPLOSION)
                    tr->spikiness = SPIKINESS_AFTER_EXPLOSION;
                w->score += 3;
                return true;
            }
        }
    }

    for (int i=0; i<NUMBER_OF_MINES; i++)
    {
        Mine * tr = w->mines[i];
        if (tr->isAlive
            && tr->planet == this->planet
            && tr->explosionFactor == 0)
        {
            float d = 0;
            for (int i=0; i<3; i++)
            {
                float dd = this->q[i] - tr->q[i];
                d += dd*dd;
            }
            float R = this->radius + tr->radius;
            if (d < R*R) 
            {
                w->score ++;
                tr->isAlive = false;
                //tr->radiusInt = 1.0f;
                return true;
            }
        }
    }

    //for (int i=0; i<NUMBER_OF_SNAKES; i++)
    //{
    //    Snake * s = w->snakes[i];
    //    if (s->detectImpact(this) != NO_IMPACT)
    //        return true;
    //}

    return false;
}

void Missile::explode(World * w)
{
    if (!this->isAlive) return;

    //cout << "target " << this << " exploded" << endl;
    if (this->explosionFactor == 0)
    {
        float d=0;
        for (int i=0; i<3; i++)
        {
            float dd = -w->front[i] + w->planets[this->planet]->position[i]+ this->position[i];
            d += dd * dd;
        }
        int volume = MY_SDL_MIX_MAXVOLUME - (int)(60 * sqrtf(d));
        if (volume >0) 
        {
            if (this->age < 30) w->killSound(this->drivebySound);
            w->playSound(SOUND_EXPLOSION_MISSILE, volume);
        }
        this->explosionFactor = 21;
        this->dq = 0;
    }
}

void Missile::bounceIfCollision(World * w)
{
    for (int si = 0; si<NUMBER_OF_SNAKES; si++)
    {
        Snake * s = w->snakes[si];

        int r = s->detectImpact(this);
        if (r == NO_IMPACT) continue;

        Snake::Ring * ring = s->rings + s->closestRing(this->q, r);

        float ps = 0;
        for (int j=0; j<3; j++) ps += ring->v[j] * this->t[j];

        float d = 0;
        if (ring->planet == this->planet)
            for (int i=0; i<3; i++)
            {
                this->t[i] -= 2 * ps * ring->v[i];
                this->q[i] = ring->p[i] + this->dq * this->t[i];
                float dd = this->q[i] - w->p[i];
                d += dd*dd;
            }
        else
            for (int i=0; i<3; i++)
            {
                this->t[i] -= 2 * ps * ring->v[i];
                //this->q[i] += SNAKE_RADIUS * this->t[i];
                this->q[i] = ring->p[i] + this->tr[i] + this->dq * this->t[i];
                float dd = this->q[i] - w->p[i];
                d += dd*dd;
            }

        //int volume = MY_SDL_MIX_MAXVOLUME - (int)(100 * d);
        //if (volume >0) w->playSound(SOUND_BOING, volume);

        return;
    }

    // no bounce on snake bodies - look at sneak heads now
    float rImpact = this->radius + 1.5f * SNAKE_RADIUS; rImpact *= rImpact;
    for (int si = 0; si<NUMBER_OF_SNAKES; si++)
    {
        Snake * s = w->snakes[si];

        if (s->planet1 != this->planet)
            continue;

        if (! s->detectHeadImpact(this->q, rImpact))
            continue;

        //// h: this->q - the center of this snake's head
        //Snake::Ring * ring = s->rings + ((s->firstRing/SNAKE_SPEED) % SNAKE_LENGTH);
        //float h[3], n=0;
        //for (int i=0; i<3; i++)
        //{
        //    h[i] = this->q[i] - (ring->p[i] + 3 * SNAKE_HEAD_LENGTH * ring->t[i]);
        //    n += h[i] * h[i];
        //}
        //n = sqrtf(n);

        //float ps = 0;
        //for (int j=0; j<3; j++) ps += h[j] * this->t[j]; // < 0
        //if (ps > 0) continue;

        float d = 0;
        for (int i=0; i<3; i++)
        {
            this->t[i] = -this->t[i];
            //this->q[i] -= this->dq * h[i];
            float dd = this->q[i] - w->p[i];
            d += dd*dd;
        }

        //int volume = MY_SDL_MIX_MAXVOLUME - (int)(100 * d);
        //if (volume >0) w->playSound(SOUND_BOING, volume);

        return;
    }
}

void Missile::update(World * w)
{
    if (this->explosionFactor == 1)
    {
        this->isAlive = false;
        return;
    }

    if (this->isAlive) this->age++;

    // update tail
    this->tailOrigin = (this->tailOrigin+1) % TAIL_LENGTH;
    if (this->tailLength < TAIL_LENGTH) this->tailLength ++;
    for (int i=0; i<3; i++)
    {
        this->tail[this->tailOrigin].p[i] = this->position[i];
    }

    // update position and position of the tail
    if (this->move(w)) this->translateTail();

    // check if boom
    if (this->checkAllTargetsForImpact(w)) 
    {
        this->explode(w);
    }
    else
    {
        this->bounceIfCollision(w);

        // continue explosion
        if (this->explosionFactor != 0) this->explosionFactor--;
        
        // slow down the missile
        if (this->dq > INITIAL_MISSILE_SPEED) this->dq -= 0.0005f;
    }
}

void Missile::draw(World * w)
{
    glPushMatrix();
    //glColor3f(0.3f, 0.5f, 0.05f);

    glTranslatef(this->position[0], this->position[1], this->position[2]);
    glRotatef(this->angle, w->v[0], w->v[1], w->v[2]);
    if (this->planet == w->currentPlanet)
    {
        float mainRadius = this->radius * (1.5f - this->radiusInt);
        glScalef(mainRadius,mainRadius,mainRadius);
    }
    else
    {
        float mainRadius = 1.5f * (this->radius * (1.5f - this->radiusInt));
        glScalef(mainRadius,mainRadius,mainRadius);
    }

    if (w->wire)
    {
        if (this->explosionFactor == 0)
        {
            this->drawExplosionW(w, 0.01f, 0.01f, 0.01f, 1);
        }
        else
        {
            glScalef(1.5f,1.5f,1.5f);
            this->drawExplosionW(w, 0.8f, 1, 0, 5 - ((float)this->explosionFactor)/3);
        }
        glPopMatrix();

        glPushMatrix();

        for (int t=0; t<this->tailLength; t++)
        {
            struct TailItem * ti = this->tail + (this->tailOrigin + TAIL_LENGTH - t) % TAIL_LENGTH;

            glPushMatrix();
            glTranslatef(ti->p[0], ti->p[1], ti->p[2]);
            glRotatef(32*((float)(TAIL_LENGTH - t))/(TAIL_LENGTH), 1, 0, 0);

            this->drawExplosionW(w, 0.5f, 0.6f, 0.5f, 
                (4 - 3.5f*((float)(TAIL_LENGTH - t))/(TAIL_LENGTH)) 
                * (this->radius * (1.5f - this->radiusInt)));

            glPopMatrix();
        }
    }
    else
    {
        if (this->explosionFactor == 0)
        {
            this->drawExplosion(w, 0.01f, 0.01f, 0.01f, 1, 1, w->whiteTextureId);
        }
        else
        {
            glScalef(1.5f,1.5f,1.5f);
            this->drawExplosion(w, 1, 1, 1, 5 - ((float)this->explosionFactor)/3, 
                ((float)this->explosionFactor)/12, w->fireTextureId);
        }
        glPopMatrix();

        glPushMatrix();

        for (int t=0; t<this->tailLength; t++)
        {
            struct TailItem * ti = this->tail + (this->tailOrigin + TAIL_LENGTH - t) % TAIL_LENGTH;

            glPushMatrix();
            glTranslatef(ti->p[0], ti->p[1], ti->p[2]);
            glRotatef(32*((float)(TAIL_LENGTH - t))/(TAIL_LENGTH), 1, 0, 0);

            this->drawExplosion(w, 0.5f, 0.6f, 0.5f, 
                (4 - 3.5f*((float)(TAIL_LENGTH - t))/(TAIL_LENGTH)) 
                * (this->radius * (1.5f - this->radiusInt)), 
                0.1f + 0.2f*((float)(TAIL_LENGTH - t))/(TAIL_LENGTH), w->whiteTextureId);

            glPopMatrix();
        }
    }

    glPopMatrix();
}