/***************************************************************************
                          vectorutils.cpp  -  description
                             -------------------
    begin                : Tue Dec 17 2002
    copyright            : (C) 2002 by Chong Jiayi
    email                : jychong@stanford.edu
 ***************************************************************************/


#include "vectorutils.h"
#include <iostream.h>

void Vzero(Vector *v) {
    v->x = 0.0;
    v->y = 0.0;
    v->z = 0.0;
}

void Vset(Vector *v, float x, float y, float z) {
    v->x = x;
    v->y = y;
    v->z = z;
}

bool Vequal(const Vector *v1, const Vector *v2) {
	return( (v1->x == v2->x) && (v1->y == v2->y) && (v1->z == v2->z));
}

void Vsub(Vector *src1, Vector *src2, Vector *dst) {
    dst->x = src1->x - src2->x;
    dst->y = src1->y - src2->y;
    dst->z = src1->z - src2->z;
}

void Vcopy(const Vector *v1, Vector *v2)
{
	*v2 = *v1;
}

void Vcross(Vector *v1, Vector *v2, Vector *cross) {
    cross->x = (v1->y * v2->z) - (v1->z * v2->y);
    cross->y = (v1->z * v2->x) - (v1->x * v2->z);
    cross->z = (v1->x * v2->y) - (v1->y * v2->x);
}

float Vsqlength(const Vector *v) {
  return v->x * v->x + v->y * v->y + v->z * v->z;
}

float Vlength(const Vector *v) {
//    return (float)sqrt(Vsqlength(v));
    return (float)fast_sqrt(Vsqlength(v));
}

void Vscale(Vector *v, float div) {
    v->x *= div;
    v->y *= div;
    v->z *= div;
}

void Vnormal(Vector *v) {
	float mag = Vlength(v);
	if(mag > 0.0f) Vscale(v, 1.0f / mag );
}

float Vdot(const Vector *v1, const Vector *v2) {
    return v1->x*v2->x + v1->y*v2->y + v1->z*v2->z;
}

void Vadd(Vector *src1, Vector *src2, Vector *dst) {
    dst->x = src1->x + src2->x;
    dst->y = src1->y + src2->y;
    dst->z = src1->z + src2->z;
}

void Vinterpolate(Vector *from, Vector *to, Vector *result, int smoothness, int index) {
	float inv = 1.0f / (float)(smoothness);
	result->x = (to->x - from->x) * inv * index + from->x;
	result->y = (to->y - from->y) * inv * index + from->y;
	result->z = (to->z - from->z) * inv * index + from->z;
}

void PlaneInterpolate(Plane *from, Plane *to, Plane *result, int smoothness, int index) {
	float inv = 1.0f / (float)(smoothness);
	result->A = (to->A - from->A) * inv * index + from->A;
	result->B = (to->B - from->B) * inv * index + from->B;
	result->C = (to->C - from->C) * inv * index + from->D;
	result->D = (to->D - from->D) * inv * index + from->D;
}

GLfloat detMatrix(GLfloat mat[4][4]) {
	return
      mat[0][0] *
      (mat[1][1] * (mat[2][2] * mat[3][3] - mat[2][3] * mat[3][2]) +
       mat[1][2] * (mat[2][3] * mat[3][1] - mat[2][1] * mat[3][3]) +
       mat[1][3] * (mat[2][1] * mat[3][2] - mat[2][2] * mat[3][1]))
      - mat[0][1] *
      (mat[1][0] * (mat[2][2] * mat[3][3] - mat[2][3] * mat[3][2]) +
       mat[1][2] * (mat[2][3] * mat[3][0] - mat[2][0] * mat[3][3]) +
       mat[1][3] * (mat[2][0] * mat[3][2] - mat[2][2] * mat[3][0]))
      + mat[0][2] *
      (mat[1][0] * (mat[2][1] * mat[3][3] - mat[2][3] * mat[3][1]) +
       mat[1][1] * (mat[2][3] * mat[3][0] - mat[2][0] * mat[3][3]) +
       mat[1][3] * (mat[2][0] * mat[3][1] - mat[2][1] * mat[3][0]))
      - mat[0][3] *
      (mat[1][0] * (mat[2][1] * mat[3][2] - mat[2][2] * mat[3][1]) +
       mat[1][1] * (mat[2][2] * mat[3][0] - mat[2][0] * mat[3][2]) +
       mat[1][2] * (mat[2][0] * mat[3][1] - mat[2][1] * mat[3][0]));

}

void invert4x4Matrixf(GLfloat *out, const GLfloat *m) {
#define MAT(m,r,c) (m)[(c)*4+(r)]
	 GLfloat detMat[4][4];
	 for(int i = 0; i < 4; i++) {
		 for(int j = 0; j < 4; j++) {
       	detMat[i][j] = MAT(m, i, j);
		 }
  	 }
    GLdouble det = detMatrix(detMat);

    double
         a1 = MAT(m, 1, 2),
         a2 = MAT(m, 0, 3),
         a3 = MAT(m, 2, 1),
         a4 = MAT(m, 3, 0),
         a5 = MAT(m,1, 3),
         a6 = MAT(m,0, 2),
         a7 = MAT(m,2, 2),
         a8 = MAT(m,1, 1),
         a9 = MAT(m,2, 3),
        a10 = MAT(m,0, 1),
        a11 = MAT(m,3, 1),
        a12 = MAT(m,2, 0),
        a13 = MAT(m,3, 2),
        a14 = MAT(m,3, 3),
        a15 = MAT(m,1, 0),
        a16 = MAT(m,0, 0);

    MAT(out, 0, 0) = a8*a7*a14 - a8*a9*a13 - a3*a1*a14 + a3*a5*a13 + a11*a1*a9 - a11*
a5*a7;
    MAT(out, 0, 1) = -(a10*a7*a14 - a10*a9*a13 - a3*a6*a14 + a3*a2*a13 + a11*a6*a9 -
a11*a2*a7);
    MAT(out, 0, 2) = a10*a1*a14 - a10*a5*a13 - a8*a6*a14 + a8*a2*a13 + a11*a6*a5 - a11*a2*a1;
    MAT(out, 0, 3) = -(a10*a1*a9 - a10*a5*a7 - a8*a6*a9 + a8*a2*a7 + a3*a6*a5 - a3*a2
*a1);
    MAT(out, 1, 0) = -(a15*a7*a14 - a15*a9*a13 - a12*a1*a14 + a12*a5*a13 + a4*a1*a9 -
 a4*a5*a7);
    MAT(out, 1, 1) = a16*a7*a14 - a16*a9*a13 - a12*a6*a14 + a12*a2*a13 + a4*a6*a9 - a4*a2*a7;
    MAT(out, 1, 2) = -(a16*a1*a14 - a16*a5*a13 - a15*a6*a14 + a15*a2*a13 + a4*a6*a5 -
 a4*a2*a1);
    MAT(out, 1, 3) = a16*a1*a9 - a16*a5*a7 - a15*a6*a9 + a15*a2*a7 + a12*a6*a5 - a12*
a2*a1;
    MAT(out, 2, 0) = a15*a3*a14 - a15*a9*a11 - a12*a8*a14 + a12*a5*a11 + a4*a8*a9 - a4*a5*a3;
    MAT(out, 2, 1) = -(a16*a3*a14 - a16*a9*a11 - a12*a10*a14 + a12*a2*a11 + a4*a10*a9
 - a4*a2*a3);
    MAT(out, 2, 2) = a16*a8*a14 - a16*a5*a11 - a15*a10*a14 + a15*a2*a11 + a4*a10*a5 -
 a4*a2*a8;
    MAT(out, 2, 3) = -(a16*a8*a9 - a16*a5*a3 - a15*a10*a9 + a15*a2*a3 + a12*a10*a5 -
a12*a2*a8);
    MAT(out, 3, 0) = -(a15*a3*a13 - a15*a7*a11 - a12*a8*a13 + a12*a1*a11 + a4*a8*a7 -
 a4*a1*a3);
    MAT(out, 3, 1) = a16*a3*a13 - a16*a7*a11 - a12*a10*a13 + a12*a6*a11 + a4*a10*a7 -
 a4*a6*a3;
    MAT(out, 3, 2) = -(a16*a8*a13 - a16*a1*a11 - a15*a10*a13 + a15*a6*a11 + a4*a10*a1
 - a4*a6*a8);
    MAT(out, 3, 3) = a16*a8*a7 - a16*a1*a3 - a15*a10*a7 + a15*a6*a3 + a12*a10*a1 - a12*a6*a8;

    // divide each matrix element by the determinant using adjoint method
    det = 1.0f / det;                  
    for (int r=0; r<4; r++)
      for (int c=0; c<4; c++)
        MAT(out, r, c) *= det;

#undef MAT
}

/*
 * Compute inverse of 4x4 transformation SINGLE-PRECISION matrix.
 */

GLboolean
invertMatrixf(GLfloat *out, const GLfloat *m)
{
	invert4x4Matrixf(out, m);
	return true;
}
void buildFrustumMatrix(GLdouble m[16],
                   GLdouble l, GLdouble r, GLdouble b, GLdouble t,
                   GLdouble n, GLdouble f)
{
  m[0] = (2.0*n) / (r-l);
  m[1] = 0.0;
  m[2] = 0.0;
  m[3] = 0.0;

  m[4] = 0.0;
  m[5] = (2.0*n) / (t-b);
  m[6] = 0.0;
  m[7] = 0.0;

  m[8] = (r+l) / (r-l);
  m[9] = (t+b) / (t-b);
  m[10] = -(f+n) / (f-n);
  m[11] = -1.0;

  m[12] = 0.0;
  m[13] = 0.0;
  m[14] = -(2.0*f*n) / (f-n);
  m[15] = 0.0;
}

void buildPerspectiveMatrix(GLdouble m[16],
                       GLdouble fovy, GLdouble aspect,
                       GLdouble zNear, GLdouble zFar)
{
  GLdouble xmin, xmax, ymin, ymax;

  ymax = zNear * tan(fovy * MATH_PI / 360.0f);
  ymin = -ymax;

  xmin = ymin * aspect;
  xmax = ymax * aspect;

  buildFrustumMatrix(m, xmin, xmax, ymin, ymax, zNear, zFar);
}

/* Build a 4x4 matrix transform based on the parameters for gluLookAt.
 */
void buildLookAtMatrix(GLdouble m[16], Vector eye, Vector center, Vector up)
{
   GLdouble x[3], y[3], z[3];
   Vector vecX, vecY, vecZ;
   GLdouble mag;

   /* Make rotation matrix */

   /* Z vector */
   Vsub(&eye, &center, &vecZ);
   Vnormal(&vecZ);

   z[0] = vecZ.x;
   z[1] = vecZ.y;
   z[2] = vecZ.z;
 
   Vcross(&up, &vecZ, &vecX);
   Vcross(&vecZ, &vecX, &vecY);
   Vnormal(&vecX);
   Vnormal(&vecY);

   x[0] = vecX.x;
   x[1] = vecX.y;
   x[2] = vecZ.z;
  
   y[0] = vecY.x;
   y[1] = vecY.y;
   y[2] = vecY.z;

#define M(row,col)  m[col*4+row]
   M(0,0) = x[0];  M(0,1) = x[1];  M(0,2) = x[2];  M(0,3) = -x[0]*eye.x + -x[1]*eye.y + -x[2]*eye.z;
   M(1,0) = y[0];  M(1,1) = y[1];  M(1,2) = y[2];  M(1,3) = -y[0]*eye.x + -y[1]*eye.y + -y[2]*eye.z;
   M(2,0) = z[0];  M(2,1) = z[1];  M(2,2) = z[2];  M(2,3) = -z[0]*eye.x + -z[1]*eye.y + -z[2]*eye.z;
   M(3,0) = 0.0;   M(3,1) = 0.0;   M(3,2) = 0.0;   M(3,3) = 1.0;
#undef M
}


/* Applies matrix "m" onto the "in" vector and outputs it as "out" */
void transformVector(Vector *out, Vector *in, float m[16]) {
  float w;

 out->x = *(m) * in->x + *(m+4) * in->y + *(m+8) * in->z + *(m+12);
 out->y = *(m+1) * in->x + *(m+5) * in->y + *(m+9) * in->z + *(m+13);
 out->z = *(m+2) * in->x + *(m+6) * in->y + *(m+10) * in->z + *(m+14);
}

/* Creates a plane out of 3 points*/
void makePlane(Vector *in1, Vector *in2, Vector *in3, Plane * outPlane) {
	Vector workVec1, workVec2, workVec3;
   //find to vectors parallel to the plane
	Vsub(in1, in2, (&workVec1));
	Vsub(in3, in2, (&workVec2));
	Vcross((&workVec1), (&workVec2), (&workVec3) );
	Vnormal(&workVec3);
	 outPlane->A = workVec3.x;
	 outPlane->B = workVec3.y;
	 outPlane->C = workVec3.z;
	 outPlane->D = -(outPlane->A * in1->x + outPlane->B * in1->y + outPlane->C * in1->z);
}

void makeQuickPlane(Vector *in1, Vector *in2, Vector *in3, Plane * outPlane) {
	Vector workVec1, workVec2, workVec3;
   //find to vectors parallel to the plane
	Vsub(in1, in2, (&workVec1));
	Vsub(in3, in2, (&workVec2));
	Vcross((&workVec1), (&workVec2), (&workVec3) );
	 outPlane->A = workVec3.x;
	 outPlane->B = workVec3.y;
	 outPlane->C = workVec3.z;
	 outPlane->D = -(outPlane->A * in1->x + outPlane->B * in1->y + outPlane->C * in1->z);
}

void normalizePlane(Plane *inPlane) {

	float mag = inPlane->A * inPlane->A + inPlane->B * inPlane->B + inPlane->C * inPlane->C + inPlane->D * inPlane->D;
	float invMag = 1.0f / mag;

	inPlane->A = inPlane->A * invMag;
	inPlane->B = inPlane->B * invMag;
	inPlane->C = inPlane->C * invMag;
	inPlane->D = inPlane->D * invMag;
}

float vecPlaneOrientation(Vector *vec, Plane *plane) {
	
	return (vec->x * plane->A + vec->y * plane->B + vec->z * plane->C + plane->D);
}

void printVector(Vector *vec, char *title) {
	cout << title << " " << "X: " << vec->x << " Y: " << vec->y << " Z: " << vec->z << endl;
}

float fast_sqrt(float n) {
	float half_v = 0.5 * n;
	int i = *(int *)&n;
	i = 0x5f375a86 - (i >> 1);
	n = *(float *)&i;
	n = n * (1.5 - half_v * n * n);
	n = n * (1.5 - half_v * n * n);
	
	return 1.0 / n;
}
