/***************************************************************************
                          tparticle.cpp  -  description
                             -------------------
    begin                : Wed Jul 16 2003
    copyright            : (C) 2003 by Chong Jiayi
    email                : jychong@stanford.edu
 ***************************************************************************/

//using namespace std;

#include "tinfrastructure.h"
#include "tparticle.h"


/* Quick utility function for texture creation for converting a SDL surface
into an OpenGL texture, reference code in the SDL documentation: testgl.c */
static int power_of_two(int input)
{
    int value = 1;

    while ( value < input ) {
        value <<= 1;
    }
    return value;
}

GLuint SDL_GL_LoadTexture(SDL_Surface *surface)
{
    GLuint texture;
    int w, h;
    SDL_Surface *image;
    SDL_Rect area;
    Uint32 saved_flags;
    Uint8  saved_alpha;

    /* Use the surface width and height expanded to powers of 2 */
    w = power_of_two(surface->w);
    h = power_of_two(surface->h);

    image = SDL_CreateRGBSurface(
            SDL_SWSURFACE,
            w, h,
            32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
            0x000000FF,
            0x0000FF00,
            0x00FF0000,
            0xFF000000
#else
            0xFF000000,
            0x00FF0000,
            0x0000FF00,
            0x000000FF
#endif
               );
    if ( image == NULL ) {
        return 0;
    }

    /* Save the alpha blending attributes */
    saved_flags = surface->flags&(SDL_SRCALPHA|SDL_RLEACCELOK);
    saved_alpha = surface->format->alpha;
    if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
        SDL_SetAlpha(surface, 0, 0);
    }

    /* Copy the surface into the GL texture image */
    area.x = 0;
    area.y = 0;
    area.w = surface->w;
    area.h = surface->h;
    SDL_BlitSurface(surface, &area, image, &area);

    /* Restore the alpha blending attributes */
    if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
        SDL_SetAlpha(surface, saved_flags, saved_alpha);
    }

    /* Create an OpenGL texture for the image */
    glGenTextures(1, &texture);
    glBindTexture(GL_TEXTURE_2D, texture);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexImage2D(GL_TEXTURE_2D,
             0,
             GL_RGBA,
             w, h,
             0,
             GL_RGBA,
             GL_UNSIGNED_BYTE,
             image->pixels);
    SDL_FreeSurface(image); /* No longer needed */

    return texture;
}

//SDL draw pixel code from the SDL documentation: http://www.libsdl.org/index.php
void DrawPixel(SDL_Surface *screen, int x, int y,
                                    Uint8 R, Uint8 G, Uint8 B)
{
  Uint32 color = SDL_MapRGB(screen->format, R, G, B);
  switch (screen->format->BytesPerPixel)
  {
    case 1: // Assuming 8-bpp
      {
        Uint8 *bufp;
        bufp = (Uint8 *)screen->pixels + y*screen->pitch + x;
        *bufp = color;
      }
      break;
    case 2: // Probably 15-bpp or 16-bpp
      {
        Uint16 *bufp;
        bufp = (Uint16 *)screen->pixels + y*screen->pitch/2 + x;
        *bufp = color;
      }
      break;
    case 3: // Slow 24-bpp mode, usually not used
      {
        Uint8 *bufp;
        bufp = (Uint8 *)screen->pixels + y*screen->pitch + x * 3;
        if(SDL_BYTEORDER == SDL_LIL_ENDIAN)
        {
          bufp[0] = color;
          bufp[1] = color >> 8;
          bufp[2] = color >> 16;
        } else {
          bufp[2] = color;
          bufp[1] = color >> 8;
          bufp[0] = color >> 16;
        }
      }
      break;
    case 4: // Probably 32-bpp
      {
        Uint32 *bufp;
        bufp = (Uint32 *)screen->pixels + y*screen->pitch/4 + x;
        *bufp = color;
      }
      break;
  }
}

void Slock(SDL_Surface *screen)
{
  if ( SDL_MUSTLOCK(screen) )
  {
    if ( SDL_LockSurface(screen) < 0 )
    {
      return;
    }
  }
}

void Sulock(SDL_Surface *screen)
{
  if ( SDL_MUSTLOCK(screen) )
  {
    SDL_UnlockSurface(screen);
  }
}

SDL_Surface * CreateSDL_Surface(int width, int height) {
	SDL_Surface *image;

    image = SDL_CreateRGBSurface(
            SDL_SWSURFACE,
            width, height,
            32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
            0x000000FF,
            0x0000FF00,
            0x00FF0000,
            0xFF000000
#else
            0xFF000000,
            0x00FF0000,
            0x0000FF00,
            0x000000FF
#endif
               );
    if ( image == NULL ) {
        return 0;
    }
    
	 return image;

}

static void setColorPointer(GLfloat *start, GLfloat r, GLfloat g, GLfloat b, GLfloat alpha) {
	*(start)  = r;
	*(start + 1)  = g;
	*(start + 2)  = b;
	*(start + 3) = alpha;
}


TParticle::TParticle(){
	particleBuffer = 0;
	if(extgl_Extensions.ARB_vertex_buffer_object) { //see if we can use vertex buffer objects
		glGenBuffersARB(1, &particleBuffer);
		glGenBuffersARB(1, &particleColorBuffer);
		glGenBuffersARB(1, &particleTextureBuffer);
		
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleBuffer);
		glBufferDataARB(GL_ARRAY_BUFFER_ARB, MAX_PARTICLES * 4 * 3 * sizeof(GLfloat), NULL, GL_STREAM_DRAW_ARB);
		
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleColorBuffer);
		glBufferDataARB(GL_ARRAY_BUFFER_ARB, MAX_PARTICLES * 4 * 4 * sizeof(GLfloat), NULL, GL_STREAM_DRAW_ARB);
		
		GLfloat *texturePnt = new GLfloat[MAX_PARTICLES * 4 * 2]; //glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY_ARB); 
		GLfloat *workPnt = texturePnt;
		
		for(int i = 0; i < MAX_PARTICLES; i++) {
			*(workPnt) = 0.0;
			*(workPnt + 1) = 0.0;
			
			*(workPnt + 2) = 1.0;
			*(workPnt + 3) = 0.0;
			
			*(workPnt + 4) = 1.0;
			*(workPnt + 5) = 1.0;
		
			*(workPnt + 6) = 0.0;
			*(workPnt + 7) = 1.0;
			
			workPnt = workPnt + 8;
		}
		
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleTextureBuffer); 
		glBufferDataARB(GL_ARRAY_BUFFER_ARB, MAX_PARTICLES * 4 * 2 * sizeof(GLfloat), texturePnt, GL_STATIC_DRAW_ARB);
		
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
		delete [] texturePnt;
	}
	else cout << "Vertex Buffer Objects not supported!" << endl;
}
TParticle::~TParticle(){
	//release memory created for raster images
	if(rasterImages.size() > 0) {
		for(int i = 0; i < rasterImages.size(); i++) {
			glDeleteTextures(1, (unsigned int *)&rasterImages[i]);
		}
	}
	
	//release memory for vertex buffer object
	if(extgl_Extensions.ARB_vertex_program) {
		glDeleteBuffersARB(1, &particleBuffer);
		glDeleteBuffersARB(1, &particleColorBuffer);
		glDeleteBuffersARB(1, &particleTextureBuffer);
	}
}

//renders a glowing ball into an opengl texture
GLuint TParticle::createGlowBall(int r1, int g1, int b1, int r2, int g2, int b2) {
	SDL_Surface * image = CreateSDL_Surface(128, 128);
	
	//set surface to black first
	for(int x = 0; x < 128; x++) {
		for(int y = 0; y < 128; y++) {
			DrawPixel(image, x, y, 0, 0, 0);
		}
	}
	
	int midpntX = 64;
	int midpntY = 64;
	
	float maxDist = 30;
	float frameDist = 64 - maxDist;
	
	for(x = 0; x < 128; x++) {
		for(int y = 0; y < 128; y++) {
			int deltaX =  (x - midpntX);
			int deltaY =  (y - midpntY);
			float dist = sqrt((double)(deltaX * deltaX + deltaY * deltaY));
			
			if(dist <= maxDist) {
			DrawPixel(image, x, y, (float)(r2 - r1) / (float)maxDist * (float)dist + (float)r1,
						(float)(g2 - g1) / (float)maxDist * (float)dist + (float)g1,
						(float)(b2 - b1) / (float)maxDist * (float)dist + (float)b1); 
			}
			else {
				float curDist = (float)(dist - maxDist);
				if(curDist > frameDist) curDist = frameDist;
				DrawPixel(image, x, y, r2 - (float)(r2) / (float)frameDist * (float)(curDist),
						g2 - (float)(g2) / (float)frameDist * (float)(curDist),
						b2 - (float)(b2) / (float)frameDist * (float)(curDist));
			}
		}
	}
	
	GLuint retIndex = SDL_GL_LoadTexture(image);
	rasterImages.push_back(retIndex);
	
	SDL_FreeSurface(image);
	
	return retIndex;
}

void TParticle::prepareParticlePlainRender() {
	glActiveTextureARB( GL_TEXTURE1_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable( GL_TEXTURE_2D );

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );

	glDisable(GL_REGISTER_COMBINERS_NV);
	glEnable(GL_TEXTURE_2D);
	glDepthFunc(GL_LEQUAL);

	glDisable(GL_LIGHTING);
	
	glDisable(GL_STENCIL_TEST);
	glDepthMask(GL_FALSE);
	
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE);
	
}

void TParticle::endParticleRender() {
	glDisable(GL_BLEND);
	glDepthMask(GL_TRUE);
}


void TParticle::renderParticlePlain(GLfloat x, GLfloat y, GLfloat z, GLfloat width, GLfloat height, GLfloat angleX,
					GLfloat angleY, GLfloat angleZ, GLfloat *mat) {
	
	glColor4f(1.0, 1.0, 1.0, 1.0);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glLoadMatrixf(mat);
	glTranslatef(x, y, z);
	glRotatef(angleY, 0.0, 1.0, 0.0);
	glRotatef(angleX, 1.0, 0.0, 0.0);
	
		glBegin(GL_QUADS);
			glTexCoord2f(0.0, 0.0);
			glVertex3f(-width * 0.5, height * 0.5, 0.0);
		
			glTexCoord2f(1.0, 0.0);
			glVertex3f(width * 0.5, height * 0.5, 0.0);
			
			glTexCoord2f(1.0, 1.0);
			glVertex3f(width * 0.5, -height * 0.5, 0.0);
		
			glTexCoord2f(0.0, 1.0);
			glVertex3f(-width * 0.5, -height * 0.5, 0.0);
		glEnd();
	glPopMatrix();
}

void TParticle::renderParticleList(slist<GamerAmmo>::iterator begin, slist<GamerAmmo>::iterator end, 
					Vector gravity, float alphaDelta, 
					GLfloat r1, GLfloat g1, GLfloat b1, GLfloat r2, GLfloat g2, GLfloat b2,
					GLfloat r3, GLfloat g3, GLfloat b3, int maxFrame,
					GLfloat width, GLfloat height,
					GLfloat angleX, GLfloat angleY, GLfloat angleZ, GLfloat *mat) {
	slist<GamerAmmo>::iterator iter;
	Vector orgLoc;
	float maxDist = (float)maxFrame;
	float distHalf = maxDist * 0.5;
	float curDist = 0.0f;
	float curAlpha;
	
	Vector curVec;
	
	if(extgl_Extensions.ARB_vertex_buffer_object) { //see if we can use vertex buffer objects
		renderParticleListVBO(begin, end, gravity, alphaDelta, r1, g1, b1, r2, g2, b2, r3, g3, b3, maxFrame, width, height,
						angleX, angleY, angleZ, mat);
		return;
	}
	
	for(iter = begin; iter != end; iter++) {
		(*iter).frame++;
		
		if(iter == begin) {
			orgLoc = (*iter).loc;
			curAlpha = 1.0f - ((float)(*iter).frame * alphaDelta) ;
			if(curAlpha < 0.0) break;
			continue;
		}
		
		Vadd( &(*iter).vel, &gravity, &(*iter).vel);
		Vadd( &(*iter).vel, &(*iter).loc, &(*iter).loc);
		
		Vsub(&(*iter).loc, &orgLoc, &curVec);
		curDist = Vlength(&curVec);
		
		if(curDist < distHalf) {
			glColor4f( (r2 - r1) / distHalf * curDist + r1, (g2 - g1) / distHalf * curDist + g1, 
					(b2 - b1) / distHalf * curDist + b1, curAlpha);
		}
		else {
			float newDist = curDist - distHalf;
			glColor4f( (r3 - r2) / distHalf * newDist + r2, (g3 - g2) / distHalf * newDist + g2, 
					(b3 - b2) / distHalf * newDist + b2, curAlpha);
		}
		
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		glLoadMatrixf(mat);
		glTranslatef( (*iter).loc.x, (*iter).loc.y, (*iter).loc.z);
		glRotatef(angleY, 0.0, 1.0, 0.0);
		glRotatef(angleX, 1.0, 0.0, 0.0);
	
		glBegin(GL_QUADS);
			glTexCoord2f(0.0, 0.0);
			glVertex3f(-width * 0.5, height * 0.5, 0.0);
		
			glTexCoord2f(1.0, 0.0);
			glVertex3f(width * 0.5, height * 0.5, 0.0);
			
			glTexCoord2f(1.0, 1.0);
			glVertex3f(width * 0.5, -height * 0.5, 0.0);
		
			glTexCoord2f(0.0, 1.0);
			glVertex3f(-width * 0.5, -height * 0.5, 0.0);
		glEnd();
		
		glPopMatrix();
				
	}
	
	
	
}

void TParticle::renderParticleListVBO(slist<GamerAmmo>::iterator begin, slist<GamerAmmo>::iterator end, 
				Vector gravity, float alphaDelta, 
				GLfloat r1, GLfloat g1, GLfloat b1, GLfloat r2, GLfloat g2, GLfloat b2,
				GLfloat r3, GLfloat g3, GLfloat b3, int maxFrame,
				GLfloat width, GLfloat height,
				GLfloat angleX, GLfloat angleY, GLfloat angleZ, GLfloat *mat) {

	slist<GamerAmmo>::iterator iter;
	Vector orgLoc;
	GLfloat curRed = 0.0, curGreen = 0.0, curBlue = 0.0;
	float maxDist = (float)maxFrame;
	float distHalf = maxDist * 0.5;
	float curDist = 0.0f;
	float curAlpha;
	bool shouldStop = false;
	GLfloat curMatrix[16];
	Vector quadVert[4], quadVert2[4];
	Vector curVec;
	
	int cnt = MAX_PARTICLES;
	
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleBuffer);
	GLfloat *vertexPnt = (GLfloat *)glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY_ARB);
	
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleColorBuffer);
	GLfloat *colorPnt = (GLfloat *)glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY_ARB);
	
	
	glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		glRotatef(angleY, 0.0, 1.0, 0.0);
		glRotatef(angleX, 1.0, 0.0, 0.0);
		glGetFloatv(GL_MODELVIEW_MATRIX, &curMatrix[0]);
	glPopMatrix();
	
	Vset(&quadVert[0], -width * 0.5, height * 0.5, 0.0);
	Vset(&quadVert[1], width * 0.5, height * 0.5, 0.0);
	Vset(&quadVert[2], width * 0.5, -height * 0.5, 0.0);
	Vset(&quadVert[3], -width * 0.5, -height * 0.5, 0.0);
	
	transformVector(&quadVert2[0], &quadVert[0], curMatrix);
	transformVector(&quadVert2[1], &quadVert[1], curMatrix);
	transformVector(&quadVert2[2], &quadVert[2], curMatrix);
	transformVector(&quadVert2[3], &quadVert[3], curMatrix);
	
	cnt = 0;
	for(iter = begin; iter != end; iter++) {
		cnt++;
		(*iter).frame++;
		
		if(iter == begin) {
			orgLoc = (*iter).loc;
			curAlpha = 1.0f - ((float)(*iter).frame * alphaDelta) ;
			if(curAlpha < 0.0) {shouldStop = true; break; }
			continue;
		}
		
		Vadd( &(*iter).vel, &gravity, &(*iter).vel);
		Vadd( &(*iter).vel, &(*iter).loc, &(*iter).loc);
		
		Vsub(&(*iter).loc, &orgLoc, &curVec);
		curDist = Vlength(&curVec);
		
		if(curDist < distHalf) {
			curRed =  (r2 - r1) / distHalf * curDist + r1;
			curGreen = (g2 - g1) / distHalf * curDist + g1; 
			curBlue = (b2 - b1) / distHalf * curDist + b1;
		}
		else {
			float newDist = curDist - distHalf;
			curRed =  (r3 - r2) / distHalf * newDist + r2;
			curGreen = (g3 - g2) / distHalf * newDist + g2; 
			curBlue = (b3 - b2) / distHalf * newDist + b2;
			
		}
		
		setColorPointer(colorPnt, curRed, curGreen, curBlue, curAlpha);
		setColorPointer(colorPnt + 4, curRed, curGreen, curBlue, curAlpha);
		setColorPointer(colorPnt + 8, curRed, curGreen, curBlue, curAlpha); 
		setColorPointer(colorPnt + 12, curRed, curGreen, curBlue, curAlpha); 
		
		Vadd( &(*iter).loc, &quadVert2[0], (Vector *)vertexPnt);
		Vadd( &(*iter).loc, &quadVert2[1], (Vector *)(vertexPnt + 3)); 
		Vadd( &(*iter).loc, &quadVert2[2], (Vector *)(vertexPnt + 6)); 
		Vadd( &(*iter).loc, &quadVert2[3], (Vector *)(vertexPnt + 9)); 
		
		colorPnt = colorPnt + 16; 
		vertexPnt = vertexPnt + 12; 
	}
	
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleBuffer);
		glUnmapBufferARB(GL_ARRAY_BUFFER_ARB);
		
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleColorBuffer);
		glUnmapBufferARB(GL_ARRAY_BUFFER_ARB);
	
	if(!shouldStop) {
		//now render the vbo object
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleBuffer);
		glEnableClientState(GL_VERTEX_ARRAY);
		glVertexPointer(3, GL_FLOAT, 0, 0);
	
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleColorBuffer);
		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(4, GL_FLOAT, 0, 0);
	
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleTextureBuffer);
		glClientActiveTextureARB(GL_TEXTURE0_ARB);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2, GL_FLOAT, 0, 0);
	
		glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glLoadIdentity();
			glLoadMatrixf(mat);
	
		glDrawArrays(GL_QUADS, 0, cnt * 4);
		glPopMatrix();
		
		glDisableClientState(GL_VERTEX_ARRAY);
		glDisableClientState(GL_COLOR_ARRAY);
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	}
	
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
}


void TParticle::renderParticleList2(slist<GamerAmmo>::iterator begin, slist<GamerAmmo>::iterator end, 
				Vector gravity, float alphaDelta, 
				GLfloat r1, GLfloat g1, GLfloat b1, GLfloat r2, GLfloat g2, GLfloat b2,
				GLfloat r3, GLfloat g3, GLfloat b3, int maxFrame,
				GLfloat width, GLfloat height,
				GLfloat angleX, GLfloat angleY, GLfloat angleZ, GLfloat *mat) {
	
	slist<GamerAmmo>::iterator iter;
	Vector orgLoc;
	float maxDist = (float)maxFrame;
	float distHalf = maxDist * 0.5;
	float curDist = 0.0f;
	float curAlpha;
	
	Vector curVec;
	Vector curVel;
	
	Vset(&curVel, 0.0, 0.0, 0.0);

	if(extgl_Extensions.ARB_vertex_buffer_object) { //see if we can use vertex buffer objects
		renderParticleList2VBO(begin, end, gravity, alphaDelta, r1, g1, b1, r2, g2, b2, r3, g3, b3, maxFrame, width, height,
						angleX, angleY, angleZ, mat);
		return;
	}
		
	for(iter = begin; iter != end; iter++) {
		(*iter).frame++;
		
		if(iter == begin) {
			orgLoc = (*iter).loc;
			curAlpha = 1.0f - ((float)(*iter).frame * alphaDelta) ;
			if(curAlpha < 0.0) break;
			continue;
		}
		
		Vsub( &(*iter).loc, &gravity, &curVel);
		float dist = Vlength(&curVel);
		Vnormal(&curVel);
		Vscale(&curVel, dist / 20.0);
		Vadd( &curVel, &(*iter).vel, &(*iter).vel);

		Vadd( &(*iter).vel, &curVel, &(*iter).vel);
		Vadd( &(*iter).vel, &(*iter).loc, &(*iter).loc);
				
		Vsub(&(*iter).loc, &orgLoc, &curVec);
		curDist = Vlength(&curVec);
		
		if(curDist < distHalf) {
			glColor4f( (r2 - r1) / distHalf * curDist + r1, (g2 - g1) / distHalf * curDist + g1, 
					(b2 - b1) / distHalf * curDist + b1, curAlpha);
		}
		else {
			float newDist = curDist - distHalf;
			glColor4f( (r3 - r2) / distHalf * newDist + r2, (g3 - g2) / distHalf * newDist + g2, 
					(b3 - b2) / distHalf * newDist + b2, curAlpha);
		}
		
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		glLoadMatrixf(mat);
		glTranslatef( (*iter).loc.x, (*iter).loc.y, (*iter).loc.z);
		glRotatef(angleY, 0.0, 1.0, 0.0);
		glRotatef(angleX, 1.0, 0.0, 0.0);
	
		glBegin(GL_QUADS);
			glTexCoord2f(0.0, 0.0);
			glVertex3f(-width * 0.5, height * 0.5, 0.0);
		
			glTexCoord2f(1.0, 0.0);
			glVertex3f(width * 0.5, height * 0.5, 0.0);
			
			glTexCoord2f(1.0, 1.0);
			glVertex3f(width * 0.5, -height * 0.5, 0.0);
		
			glTexCoord2f(0.0, 1.0);
			glVertex3f(-width * 0.5, -height * 0.5, 0.0);
		glEnd();
		
		glPopMatrix();
				
	}
	
	
}

void TParticle::renderParticleList2VBO(slist<GamerAmmo>::iterator begin, slist<GamerAmmo>::iterator end, 
				Vector gravity, float alphaDelta, 
				GLfloat r1, GLfloat g1, GLfloat b1, GLfloat r2, GLfloat g2, GLfloat b2,
				GLfloat r3, GLfloat g3, GLfloat b3, int maxFrame,
				GLfloat width, GLfloat height,
				GLfloat angleX, GLfloat angleY, GLfloat angleZ, GLfloat *mat) {
	slist<GamerAmmo>::iterator iter;
	Vector orgLoc;
	GLfloat curRed = 0.0, curGreen = 0.0, curBlue = 0.0;
	float maxDist = (float)maxFrame;
	float distHalf = maxDist * 0.5;
	float curDist = 0.0f;
	float curAlpha;
	bool shouldStop = false;
	GLfloat curMatrix[16];
	Vector quadVert[4], quadVert2[4];
	Vector curVec;
	Vector curVel;
	
	int cnt = MAX_PARTICLES;
	Vset(&curVel, 0.0, 0.0, 0.0);
	
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleBuffer);
	GLfloat *vertexPnt = (GLfloat *)glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY_ARB);
	
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleColorBuffer);
	GLfloat *colorPnt = (GLfloat *)glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY_ARB);
	
	
	glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		glRotatef(angleY, 0.0, 1.0, 0.0);
		glRotatef(angleX, 1.0, 0.0, 0.0);
		glGetFloatv(GL_MODELVIEW_MATRIX, &curMatrix[0]);
	glPopMatrix();
	
	Vset(&quadVert[0], -width * 0.5, height * 0.5, 0.0);
	Vset(&quadVert[1], width * 0.5, height * 0.5, 0.0);
	Vset(&quadVert[2], width * 0.5, -height * 0.5, 0.0);
	Vset(&quadVert[3], -width * 0.5, -height * 0.5, 0.0);
	
	transformVector(&quadVert2[0], &quadVert[0], curMatrix);
	transformVector(&quadVert2[1], &quadVert[1], curMatrix);
	transformVector(&quadVert2[2], &quadVert[2], curMatrix);
	transformVector(&quadVert2[3], &quadVert[3], curMatrix);
	
	cnt = 0;
	for(iter = begin; iter != end; iter++) {
		cnt++;
		(*iter).frame++;
		
		if(iter == begin) {
			orgLoc = (*iter).loc;
			curAlpha = 1.0f - ((float)(*iter).frame * alphaDelta) ;
			if(curAlpha < 0.0) {shouldStop = true; break; }
			continue;
		}
		
		Vsub( &(*iter).loc, &gravity, &curVel);
		float dist = Vlength(&curVel);
		Vnormal(&curVel);
		Vscale(&curVel, dist / 20.0);
		Vadd( &curVel, &(*iter).vel, &(*iter).vel);

		Vadd( &(*iter).vel, &curVel, &(*iter).vel);
		Vadd( &(*iter).vel, &(*iter).loc, &(*iter).loc);
				
		Vsub(&(*iter).loc, &orgLoc, &curVec);
		curDist = Vlength(&curVec);
		
		if(curDist < distHalf) {
			curRed =  (r2 - r1) / distHalf * curDist + r1;
			curGreen = (g2 - g1) / distHalf * curDist + g1; 
			curBlue = (b2 - b1) / distHalf * curDist + b1;
		}
		else {
			float newDist = curDist - distHalf;
			curRed =  (r3 - r2) / distHalf * newDist + r2;
			curGreen = (g3 - g2) / distHalf * newDist + g2; 
			curBlue = (b3 - b2) / distHalf * newDist + b2;
			
		}
		
		setColorPointer(colorPnt, curRed, curGreen, curBlue, curAlpha);
		setColorPointer(colorPnt + 4, curRed, curGreen, curBlue, curAlpha);
		setColorPointer(colorPnt + 8, curRed, curGreen, curBlue, curAlpha); 
		setColorPointer(colorPnt + 12, curRed, curGreen, curBlue, curAlpha); 
		
		Vadd( &(*iter).loc, &quadVert2[0], (Vector *)vertexPnt);
		Vadd( &(*iter).loc, &quadVert2[1], (Vector *)(vertexPnt + 3)); 
		Vadd( &(*iter).loc, &quadVert2[2], (Vector *)(vertexPnt + 6)); 
		Vadd( &(*iter).loc, &quadVert2[3], (Vector *)(vertexPnt + 9)); 
		
		colorPnt = colorPnt + 16; 
		vertexPnt = vertexPnt + 12; 
	}
	
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleBuffer);
		glUnmapBufferARB(GL_ARRAY_BUFFER_ARB);
		
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleColorBuffer);
		glUnmapBufferARB(GL_ARRAY_BUFFER_ARB);
	
	if(!shouldStop) {
		//now render the vbo object
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleBuffer);
		glEnableClientState(GL_VERTEX_ARRAY);
		glVertexPointer(3, GL_FLOAT, 0, 0);
	
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleColorBuffer);
		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(4, GL_FLOAT, 0, 0);
	
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, particleTextureBuffer);
		glClientActiveTextureARB(GL_TEXTURE0_ARB);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2, GL_FLOAT, 0, 0);
	
		glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glLoadIdentity();
			glLoadMatrixf(mat);
	
		glDrawArrays(GL_QUADS, 0, cnt * 4);
		glPopMatrix();
		
		glDisableClientState(GL_VERTEX_ARRAY);
		glDisableClientState(GL_COLOR_ARRAY);
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	}
	
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
				
}





