/***************************************************************************
                          tobject.cpp  -  description
                             -------------------
    begin                : Tue Dec 17 2002
    copyright            : (C) 2002 by Chong Jiayi
    email                : jychong@stanford.edu
 ***************************************************************************/


#include "tobject.h"

#define BIG_NUMBER 100.0f
#define FRESNEL 2.0f
#define REFRACT_INDEX 1.2
#define REFRACT_INDEX_DELTA 0.03

void TObject::Initialize(TInfrastructure * curInfrastructure) {
	this->vertices = NULL;
	this->orthobasis = NULL;
	this->vertexConnect = NULL;
	this->infrastructure = curInfrastructure;
	this->texInfo = NULL;
	this->triangleNum = 0;
	this->curRenderMode = 0;
	this->lightBrightness = 100.0f;
	cubeMapSelect = -1;
	CubeMap32Ref = 0;
	CubeMap64Ref = 0;
	CubeMap128Ref = 0;
	CubeMap256Ref = 0;
	cubeRenderFrame = -1;
	
}

void TObject::Cleanup(){
	//object memory clean up
	if(this->vertices != NULL) delete [] this->vertices;
	if(this->orthobasis != NULL) delete [] this->orthobasis;
	if(this->vertexConnect != NULL) delete [] this->vertexConnect;

	if(CubeMap32Ref > 0) {
		glDeleteTextures(1, (unsigned int*)(&CubeMap32Ref));
		glDeleteTextures(1, (unsigned int*)(&CubeMap64Ref));
		glDeleteTextures(1, (unsigned int*)(&CubeMap128Ref));
		glDeleteTextures(1, (unsigned int*)(&CubeMap256Ref));
	}
} 

//renders the object into framebuffer
void TObject::Render() {
	if(GetType() == PLANE_TYPE) glDisable(GL_CULL_FACE );

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadMatrixf(this->transformMatrix);
	
	if(curRenderMode == REFRACT_MODE_1 || curRenderMode == REFRACT_MODE_2) {
		renderImmediateMode();
		glMatrixMode(GL_TEXTURE);
		glLoadIdentity();
		glMatrixMode(GL_MODELVIEW);
	}
	else glDrawArrays(GL_TRIANGLES, 0, 3 * triangleNum);
	glPopMatrix();

	if(GetType() == PLANE_TYPE) glEnable(GL_CULL_FACE );
}

void TObject::renderImmediateMode() {
	Vertex *workVertex = vertices;
	OrthoNormalBasis *workOrtho = orthobasis;
	Vector midpnt = GetBndOrgMidPnt();
	
	glBegin(GL_TRIANGLES);
	for(int i = 0; i < this->triangleNum; i++) {
		Vertex *curVertex1 = workVertex;
		Vertex *curVertex2 = workVertex + 1;
		Vertex *curVertex3 = workVertex + 2;
		
		Vector n1 = workOrtho->normal;
		Vector n2 = (workOrtho + 1)->normal;
		Vector n3 = (workOrtho + 2)->normal;
		
		glNormal3f(n1.x, n1.y, n1.z);
		glVertex3f(curVertex1->position.x, curVertex1->position.y, curVertex1->position.z);
		glNormal3f(n2.x, n2.y, n2.z);
		glVertex3f(curVertex2->position.x, curVertex2->position.y, curVertex2->position.z);
		glNormal3f(n3.x, n3.y, n3.z);
		glVertex3f(curVertex3->position.x, curVertex3->position.y, curVertex3->position.z);
		
		workVertex = workVertex + 3;
		workOrtho = workOrtho + 3;
	}
	
	glEnd();
}


void TObject::RenderWithMatrix(GLfloat *mat) {
	if(GetType() == PLANE_TYPE) glDisable(GL_CULL_FACE );

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glMultMatrixf(mat);
	glMultMatrixf(this->transformMatrix);
	glDrawArrays(GL_TRIANGLES, 0, 3 * triangleNum);
	glPopMatrix();

	if(GetType() == PLANE_TYPE) glEnable(GL_CULL_FACE );
}


//loads a texture and creates the corresponding bump map for the current object
void TObject::LoadTexture(char *decal_filename, char *bump_filename, int mipmaps, float scale) {
	texInfo = infrastructure->LoadTexture(decal_filename, mipmaps, bump_filename, scale);
	
}

//prepares the necessary vertex array stuff for rendering
void TObject::SetupVertexArraysForRendering(int mode) {
	switch(mode) {
		case PLAIN_MODE:
			glClientActiveTextureARB(GL_TEXTURE1_ARB);
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
		break;

		case DECAL_MODE:
			glClientActiveTextureARB(GL_TEXTURE1_ARB);
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			glTexCoordPointer(2, GL_FLOAT, ARRAY_STRIDE, &(vertices->s));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		break;

		case DIFFUSE_BUMP_MODE:
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			glTexCoordPointer(3, GL_FLOAT, ARRAY_STRIDE, &(vertices->tangentSpaceLightVector));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			glClientActiveTextureARB(GL_TEXTURE1_ARB);
			glTexCoordPointer(2, GL_FLOAT, ARRAY_STRIDE, &(vertices->s));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			break;

		case SPECULAR_BUMP_MODE:
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			glTexCoordPointer(3, GL_FLOAT, ARRAY_STRIDE, &(vertices->tangentSpaceHalfAngleVector));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			glClientActiveTextureARB(GL_TEXTURE1_ARB);
			glTexCoordPointer(2, GL_FLOAT, ARRAY_STRIDE, &(vertices->s));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		break;

		case ATTENUATE_MODE:
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			glTexCoordPointer(2, GL_FLOAT, ARRAY_STRIDE, &(vertices->primaryVec));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			glClientActiveTextureARB(GL_TEXTURE1_ARB);
			glTexCoordPointer(1, GL_FLOAT, ARRAY_STRIDE, &(vertices->primaryVec.z));
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		break;
		
		case REFRACT_MODE_1:
			glClientActiveTextureARB(GL_TEXTURE1_ARB);
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
//			glNormalPointer(GL_FLOAT, ARRAY_STRIDE2, &(orthobasis->normal));
//			glEnableClientState(GL_NORMAL_ARRAY);
		break;
	
		case REFRACT_MODE_2:
			glClientActiveTextureARB(GL_TEXTURE1_ARB);
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
//			glNormalPointer(GL_FLOAT, ARRAY_STRIDE2, &(orthobasis->normal));
//			glEnableClientState(GL_NORMAL_ARRAY);
		break;
	}

	glVertexPointer(3, GL_FLOAT, ARRAY_STRIDE, vertices);
	glEnableClientState(GL_VERTEX_ARRAY);
	curRenderMode = mode;
}

void TObject::setVertexPos(int index, GLfloat x, GLfloat y, GLfloat z) {
	(this->vertices + index)->position.x = x;
	(this->vertices + index)->position.y = y;
	(this->vertices + index)->position.z = z;
}

void TObject::setTextureCoords(int index, GLfloat s, GLfloat t) {
	(this->vertices + index)->s = s;
	(this->vertices + index)->t = t;
}

void TObject::setNormalCoords(int index, GLfloat x, GLfloat y, GLfloat z) {
	(this->orthobasis + index)->normal.x = x;
	(this->orthobasis + index)->normal.y = y;
	(this->orthobasis + index)->normal.z = z;
}

void TObject::setBinormalCoords(int index, GLfloat x, GLfloat y, GLfloat z) {
	(this->orthobasis + index)->binormal.x = x;
	(this->orthobasis + index)->binormal.y = y;
	(this->orthobasis + index)->binormal.z = z;
}

void TObject::setTangentCoords(int index, GLfloat x, GLfloat y, GLfloat z) {
	(this->orthobasis + index)->tangent.x = x;
	(this->orthobasis + index)->tangent.y = y;
	(this->orthobasis + index)->tangent.z = z;
}

/* Selects a single texture for decal rendering*/
void TObject::SelectSingleDecalTexture() {
	glColor3f(1.0f, 1.0f, 1.0f);
	glActiveTextureARB(GL_TEXTURE0_ARB);
	glDisable(GL_TEXTURE_CUBE_MAP_ARB);
	glEnable(GL_TEXTURE_2D);
	
	glBindTexture(GL_TEXTURE_2D, this->texInfo->decIndex);
	glDisable(GL_LIGHTING);
	glDisable(GL_BLEND);
}


/* Selects 2 textures: 1 cube map and 1 bump map for use in
    per-pixel bump-mapping, call this before doing any register
    combiners stuff
*/
void TObject::SelectCubeAndBumpTexture() {
	glColor3f(1.0f, 1.0f, 1.0f);
	glActiveTextureARB(GL_TEXTURE0_ARB);
	glDisable(GL_TEXTURE_2D);
	glEnable(GL_TEXTURE_CUBE_MAP_ARB);
	glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, this->infrastructure->GetCubeMapName());

	glActiveTextureARB(GL_TEXTURE1_ARB);
	glDisable(GL_TEXTURE_CUBE_MAP_ARB);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, this->texInfo->bumpIndex);
}

void TObject::CalculateOrthoBasis() {
	Vertex *curVertex = this->vertices;
	OrthoNormalBasis * curOrtho = this->orthobasis;
	for(int i = 0; i < this->triangleNum; i++) {
		Plane curTrianglePlane[3];
		Vertex *v1 = curVertex;
		Vertex *v2 = curVertex + 1;
		Vertex *v3 = curVertex + 2;

		Vector inV1[3], inV2[3], inV3[3];
      //make partial planes relative to x, y, and z
		inV1[0].x = v1->position.x;
		inV1[0].y = v1->s;
		inV1[0].z = v1->t;
		inV1[1].x = v2->position.x;
		inV1[1].y = v2->s;
		inV1[1].z = v2->t;
		inV1[2].x = v3->position.x;
		inV1[2].y = v3->s;
		inV1[2].z = v3->t;

		inV2[0].x = v1->position.y;
		inV2[0].y = v1->s;
		inV2[0].z = v1->t;
		inV2[1].x = v2->position.y;
		inV2[1].y = v2->s;
		inV2[1].z = v2->t;
		inV2[2].x = v3->position.y;
		inV2[2].y = v3->s;
		inV2[2].z = v3->t;

		inV3[0].x = v1->position.z;
		inV3[0].y = v1->s;
		inV3[0].z = v1->t;
		inV3[1].x = v2->position.z;
		inV3[1].y = v2->s;
		inV3[1].z = v2->t;
		inV3[2].x = v3->position.z;
		inV3[2].y = v3->s;
		inV3[2].z = v3->t;


		
		makePlane(inV1, &inV1[1], &inV1[2], &curTrianglePlane[0]);
		makePlane(inV2, &inV2[1], &inV2[2], &curTrianglePlane[1]);
		makePlane(inV3, &inV3[1], &inV3[2], &curTrianglePlane[2]);	
		//now calculate the partial derivatives
		curOrtho->tangent.x = -(curTrianglePlane[0].B / curTrianglePlane[0].A);
		curOrtho->tangent.y = -(curTrianglePlane[1].B / curTrianglePlane[1].A);
		curOrtho->tangent.z = -(curTrianglePlane[2].B / curTrianglePlane[1].A);
		Vnormal((&(curOrtho->tangent)));
		(curOrtho + 1)->tangent = curOrtho->tangent;
		(curOrtho + 2)->tangent = curOrtho->tangent;
      
		curOrtho->binormal.x = -(curTrianglePlane[0].C / curTrianglePlane[0].A);
		curOrtho->binormal.y = -(curTrianglePlane[1].C / curTrianglePlane[1].A);
		curOrtho->binormal.z = -(curTrianglePlane[2].C / curTrianglePlane[1].A);
		Vnormal((&(curOrtho->tangent)));
		(curOrtho + 1)->binormal = curOrtho->binormal;
		(curOrtho + 2)->binormal = curOrtho->binormal;
		
		curVertex = curVertex + 3;
		curOrtho = curOrtho + 3;

	}
}

void TObject::CalculateSphericalOrthoBasis() {
	OrthoNormalBasis * curOrtho = this->orthobasis;
	GLfloat curMat[16];
	for(int i = 0; i < (this->triangleNum * 3); i++) {
		Vector *curNormal = &(curOrtho->normal);
		float horizAngle = atan2(curNormal->z, curNormal->x) / MATH_PI * 180, vertAngle = atan2(curNormal->x, curNormal->y) / MATH_PI * 180;
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		glRotatef(horizAngle, 0.0, 1.0, 0.0);
		glRotatef(vertAngle, 0.0, 0.0, 1.0);
		glGetFloatv(GL_MODELVIEW_MATRIX, curMat);
		glPopMatrix();

		Vector vec1, vec2, dstVec1, dstVec2;
		Vset(&vec1, 1.0f, 0.0f, 0.0f);
		Vset(&vec2, 0.0f, 1.0f, 0.0f);
		transformVector(&dstVec1, &vec1, curMat);
		transformVector(&dstVec2, &vec2, curMat);
		curOrtho->tangent = dstVec1;
		curOrtho->binormal = dstVec2;
		curOrtho++;
	}
}

/* Sets up the neighbouring connectives for shadow volume projections*/
void TObject::SetConnectives() {

	Connective *curConnective = this->vertexConnect;
	Vertex *curVertex = this->vertices;
	vector <TEdgeTrack *> EdgeList;
	for(int m = 0; m < (this->triangleNum * 3); m++) (curConnective + m)->neighbourIndex = UNDEFINED;

	for(int i = 0; i < (this->triangleNum); i++) {
		for(int j = 0; j < 3; j++) {
			Vertex v1_1 = *(curVertex + (j % 3));
			Vertex v1_2 = *(curVertex + ((j + 1) % 3));
			bool flag = false;
			if(!EdgeList.empty()) { // go through list to see if a similar edge already exists
				for(int k = 0; k < EdgeList.size(); k++) {
					Vertex v2_1 = EdgeList[k]->v1;
					Vertex v2_2 = EdgeList[k]->v2;
					if(ShareEdge(&v1_1, &v1_2, &v2_1, &v2_2)) { flag = true; break; }
				}
				
			}

			if(flag == false) {
				TEdgeTrack *newTrack = new TEdgeTrack;
				newTrack->index = i * 3 + j;
				newTrack->v1 = v1_1;
				newTrack->v2 = v1_2;
				newTrack->done = false;
				EdgeList.push_back(newTrack);
			}
		}
		curVertex = curVertex + 3;
	}

	curConnective = this->vertexConnect;
	curVertex = this->vertices;

	for(i = 0; i < (this->triangleNum); i++) {
		 for(int j = 0; j < 3; j++) {
			Vertex v1_1 = *(curVertex + (j % 3));
			Vertex v1_2 = *(curVertex + ((j + 1) % 3));
			for(int k = 0; k < EdgeList.size(); k++) {
				if((EdgeList[k]->index != i * 3 + j) && !EdgeList[k]->done) {
						Vertex v2_1 = EdgeList[k]->v1;
						Vertex v2_2 = EdgeList[k]->v2;
						if(ShareEdge(&v1_1, &v1_2, &v2_1, &v2_2)) {
							Connective *orgConnect = this->vertexConnect + EdgeList[k]->index;
							Connective *dstConnect = this->vertexConnect + i * 3 + j;
							orgConnect->neighbourIndex = i * 3 + j;
							dstConnect->neighbourIndex = EdgeList[k]->index;
							EdgeList[k]->done = true;
						}
      		}
			}
		}
		curVertex = curVertex + 3;
	}

	curConnective = this->vertexConnect;
	curVertex = this->vertices;
	
	//clean up
	int size = EdgeList.size();
	for(i = size - 1; i >= 0; i--) {
		TEdgeTrack *curTrack = EdgeList[i];
		delete curTrack;
	}

}

bool TObject::ShareEdge(Vertex *edge11, Vertex * edge12, Vertex *edge21, Vertex *edge22) {

	Vector v1Pos1 = edge11->position;
	Vector v1Pos2 = edge12->position;
	Vector v2Pos1 = edge21->position;
	Vector v2Pos2 = edge22->position;

	if(Vequal(&v1Pos1, &v2Pos1) && Vequal(&v1Pos2, &v2Pos2)) return true;
	if(Vequal(&v1Pos1, &v2Pos2) && Vequal(&v1Pos2, &v2Pos1)) return true;

	return false;
}

float TObject::DiffEdge(Vertex *edge11, Vertex * edge12, Vertex *edge21, Vertex *edge22) {
	Vector v1Pos1 = edge11->position;
	Vector v1Pos2 = edge12->position;
	Vector v2Pos1 = edge21->position;
	Vector v2Pos2 = edge22->position;
	
	Vector vDist1;
	Vsub(&v2Pos1, &v1Pos1, &vDist1);
	Vector vDist2;
	Vsub(&v2Pos2, &v1Pos2, &vDist2);
	
	return Vlength(&vDist1) + Vlength(&vDist2);
}


bool TObject::SameTriangle(Vector *v11, Vector *v12, Vector *v13, Vector *v21, Vector *v22, Vector *v23) {
	if(Vequal(v11, v21) && Vequal(v12, v22) && Vequal(v13, v23)) return true;
	if(Vequal(v11, v21) && Vequal(v12, v23) && Vequal(v13, v22)) return true;
	if(Vequal(v11, v22) && Vequal(v12, v21) && Vequal(v13, v23)) return true;
	if(Vequal(v11, v22) && Vequal(v12, v23) && Vequal(v13, v21)) return true;
	if(Vequal(v11, v23) && Vequal(v12, v21) && Vequal(v13, v22)) return true;
	if(Vequal(v11, v23) && Vequal(v12, v22) && Vequal(v13, v21)) return true;

	return false;
}

void TObject::CalculatePlanes() {
	Connective *curConnective = this->vertexConnect;
	Vertex *curVertex = this->vertices;
	OrthoNormalBasis *curBasis = this->orthobasis;
	
	for(int i = 0; i < this->triangleNum; i++) {
		Vector *v1 = &(curVertex->position);
		Vector *v2 = &((curVertex + 1)->position);
		Vector *v3 = &((curVertex + 2)->position);

		makePlane(v1, v2, v3, &(curConnective->plane));

		(curConnective + 1)->plane = curConnective->plane;
		(curConnective + 2)->plane = curConnective->plane;


		curConnective = curConnective + 3;
		curVertex = curVertex + 3;
		curBasis = curBasis + 3;
	}
}


void TObject::PrepareTransform() {
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
}

void TObject::EndTransform() {
	glMatrixMode(GL_MODELVIEW);
	glGetFloatv(GL_MODELVIEW_MATRIX, this->transformMatrix);
	invertMatrixf(this->inverseTransformMatrix, this->transformMatrix);
	glPopMatrix();

	UpdateBounds();
}

void TObject::UpdateBounds() {
	transformVector(&DstBound1, &(this->Bound1), (this->transformMatrix) );
	transformVector(&DstBound2, &(this->Bound2), (this->transformMatrix) );
	transformVector(&DstBound3, &(this->Bound3), (this->transformMatrix) );
	transformVector(&DstBound4, &(this->Bound4), (this->transformMatrix) );
	transformVector(&DstBound5, &(this->Bound5), (this->transformMatrix) );
	transformVector(&DstBound6, &(this->Bound6), (this->transformMatrix) );
	transformVector(&DstBound7, &(this->Bound7), (this->transformMatrix) );
	transformVector(&DstBound8, &(this->Bound8), (this->transformMatrix) );
}

void TObject::MatrixMult(GLfloat *tMatrix) {
	PrepareTransform();
		glLoadMatrixf(tMatrix);
		glMultMatrixf(this->transformMatrix);
	EndTransform();
}

void TObject::SaveMatrix() {
	memcpy(transformMatrix2, transformMatrix, sizeof(GLfloat) * 16);
	memcpy(inverseTransformMatrix2, inverseTransformMatrix, sizeof(GLfloat) * 16);
}

void TObject::RestoreMatrix() {
	memcpy(transformMatrix, transformMatrix2, sizeof(GLfloat) * 16);
	memcpy(inverseTransformMatrix, inverseTransformMatrix, sizeof(GLfloat) * 16);
	UpdateBounds();
}

void TObject::ApplyTransformPermanent() {
	Vertex *curVertex = this->vertices;
	for(int i = 0; i < this->triangleNum * 3; i++) {
		Vector curPos = curVertex->position;
		transformVector(&curPos, &curVertex->position, this->transformMatrix);
		curVertex->position = curPos;
		curVertex++;
	}

	CalcBounds();
	CalculatePlanes();
	CalculateSphericalOrthoBasis();
}



/* Prepares register combiners for bump-mapping, based on Mark Kilgard's Paper "A Practical and Robust Bump-mapping Technique
    for Today's GPUs", http://developer.nvidia.com */

void TObject::EnableNVRegCombinersDiffuse(GLfloat aRed, GLfloat aGreen, GLfloat aBlue, GLfloat aAlpha, GLfloat dRed, GLfloat dGreen, GLfloat dBlue, GLfloat dAlpha) {
	GLfloat curAmbient[4];
	GLfloat curDiffuse[4];

	curAmbient[0] = aRed;
	curAmbient[1] = aGreen;
	curAmbient[2] = aBlue;
	curAmbient[3] = aAlpha;

	curDiffuse[0] = dRed;
	curDiffuse[1] = dGreen;
	curDiffuse[2] = dBlue;
	curDiffuse[3] = dAlpha;

	glCombinerParameteriNV(GL_NUM_GENERAL_COMBINERS_NV, 2);
	glCombinerParameterfvNV(GL_CONSTANT_COLOR0_NV, curAmbient);
	glCombinerParameterfvNV(GL_CONSTANT_COLOR1_NV, curDiffuse);
 /*** GENERAL Combiner ZERO, RGB portion. ***/
 /* Argb = 3x3 matrix column1 = expand(texture0rgb) = N' */
 glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_A_NV,
 GL_TEXTURE0_ARB, GL_EXPAND_NORMAL_NV, GL_RGB);
 /* Brgb = expand(texture1rgb) = L */
 glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_B_NV,
 GL_TEXTURE1_ARB, GL_EXPAND_NORMAL_NV, GL_RGB);
 /* spare0rgb = Argb dot Brgb
 =expand(texture0rgb) dot expand(texture1rgb) = L dot N' */
 glCombinerOutputNV(GL_COMBINER0_NV, GL_RGB,
 GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV,
 GL_NONE, GL_NONE, GL_TRUE, GL_FALSE, GL_FALSE);
 /*** GENERAL Combiner ZERO, Alpha portion. ***/
 /* Aa = 1 */
 glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_A_NV,
 GL_ZERO, GL_UNSIGNED_INVERT_NV, GL_ALPHA);
 /* Ba = expand(texture1b) = Lz */
 glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_B_NV,
 GL_TEXTURE1_ARB, GL_EXPAND_NORMAL_NV, GL_BLUE);
 /* Ca = 1 */
 glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_C_NV,
 GL_ZERO, GL_UNSIGNED_INVERT_NV, GL_ALPHA);
 /* Da = expand(texture1b) = Lz */
 glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_D_NV,
 GL_TEXTURE1_ARB, GL_EXPAND_NORMAL_NV, GL_BLUE);
 /* spare0a = 4*(1*Lz + 1*Lz) = 8*expand(texture1b) */
 glCombinerOutputNV(GL_COMBINER0_NV, GL_ALPHA,
 GL_DISCARD_NV, GL_DISCARD_NV, GL_SPARE0_NV,
 GL_SCALE_BY_FOUR_NV, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);
 /*** GENERAL Combiner ONE, RGB portion. ***/
 /* Argb = spare0rgb = L dot N' */
 glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_A_NV,
 GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
 /* Brgb = expand(texture0a) = normal map denormalization factor */
 glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_B_NV,
 GL_TEXTURE0_ARB, GL_UNSIGNED_IDENTITY_NV, GL_ALPHA);
 /* spare0rgb = Argb Brgb = L dot N' scaled by the normal map denormalization factor */
 glCombinerOutputNV(GL_COMBINER1_NV, GL_RGB,
 GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV,
 GL_NONE, GL_NONE, GL_TRUE, GL_FALSE, GL_FALSE);
 /*** GENERAL Combiner ONE, Alpha portion. ***/
 /* Discard all outputs. */
 glCombinerOutputNV(GL_COMBINER1_NV, GL_ALPHA,
 GL_DISCARD_NV, GL_DISCARD_NV, GL_DISCARD_NV,
 GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);
 /*** FINAL Combiner. ***/
 /* A = spare0a = per-pixel self-shadowing term */
 glFinalCombinerInputNV(GL_VARIABLE_A_NV,
 GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_ALPHA);
 /* B = EF */
 glFinalCombinerInputNV(GL_VARIABLE_B_NV,
 GL_E_TIMES_F_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
 /* C = zero */
 glFinalCombinerInputNV(GL_VARIABLE_C_NV,
 GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
 /* D = C0 = ambient illumination contribution */
 glFinalCombinerInputNV(GL_VARIABLE_D_NV,
 GL_CONSTANT_COLOR0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
 /* E = C1 = diffuse material characteristic */
 glFinalCombinerInputNV(GL_VARIABLE_E_NV,
 GL_CONSTANT_COLOR1_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
 /* F = spare0rgb = diffuse illumination contribution = L dot N' */
 glFinalCombinerInputNV(GL_VARIABLE_F_NV,
 GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
 /* diffuse RGB color = A*E*F + D = diffuse modulated by self-shadowing term and the
 diffuse material characteristic + ambient */
 /* G = spare0a = self-shadowing term = 8*expand(texture1b) */
 glFinalCombinerInputNV(GL_VARIABLE_G_NV,
 GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_ALPHA);

 glEnable(GL_REGISTER_COMBINERS_NV);

}

void TObject::EnableNVRegCombinersSpecular() {
    glCombinerParameteriNV(GL_NUM_GENERAL_COMBINERS_NV, 2);

    /*** GENERAL Combiner ZERO, RGB portion. ***/
    /* Argb = 3x3 matrix column1 = expand(texture0rgb) = N' */
    glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_A_NV,
      GL_TEXTURE0_ARB, GL_EXPAND_NORMAL_NV, GL_RGB);
    /* Brgb = expand(texture1rgb) = L (or H if specular) */
    glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_B_NV,
      GL_TEXTURE1_ARB, GL_EXPAND_NORMAL_NV, GL_RGB);

    /* spare0rgb = Argb dot Brgb
                 = expand(texture0rgb) dot expand(texture1rgb)
                 = H dot N' */
    glCombinerOutputNV(GL_COMBINER0_NV, GL_RGB,
      GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV,
      GL_NONE, GL_NONE, GL_TRUE, GL_FALSE, GL_FALSE);

    /* Aa = 1 */
    glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_A_NV,
      GL_ZERO, GL_UNSIGNED_INVERT_NV, GL_ALPHA);
    /* Ba = texture1b = unexpanded Lz (or Hz if specular) */
    glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_B_NV,
      GL_TEXTURE1_ARB, GL_UNSIGNED_IDENTITY_NV, GL_BLUE);

    /* spare0a = 1 * texture1b = unexpanded Lz (or Hz if specular) */
    glCombinerOutputNV(GL_COMBINER0_NV, GL_ALPHA,
      GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV,
      GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);

    /*** GENERAL Combiner ONE, RGB portion. ***/
    /* Argb = 0 */
    glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_A_NV,
      GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    /* Brgb = 0 */
    glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_B_NV,
      GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    /* Crgb = spare0rgb = H dot N' */
    glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_C_NV,
      GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    /* Drgb = spare0rgb = H dot N' */
    glCombinerInputNV(GL_COMBINER1_NV, GL_RGB, GL_VARIABLE_D_NV,
      GL_SPARE0_NV, GL_SIGNED_IDENTITY_NV, GL_RGB);

    /* spare0rgb = ((spare0a >= 0.5) ? spare0rgb^2 : 0)
                 = ((H dot N > 0) ? (H dot N')^2 : 0) */
    glCombinerOutputNV(GL_COMBINER1_NV, GL_RGB,
      GL_DISCARD_NV, GL_DISCARD_NV, GL_SPARE0_NV,
      GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_TRUE);

    /*** GENERAL Combiner ONE, Alpha portion. ***/
    /* Aa = 1 */
    glCombinerInputNV(GL_COMBINER1_NV, GL_ALPHA, GL_VARIABLE_A_NV,
      GL_ZERO, GL_UNSIGNED_INVERT_NV, GL_ALPHA);
    /* Ba = expand(texture1b) = Lz */
    glCombinerInputNV(GL_COMBINER1_NV, GL_ALPHA, GL_VARIABLE_B_NV,
      GL_TEXTURE1_ARB, GL_EXPAND_NORMAL_NV, GL_BLUE);

    /* spare0a = 1 * expand(texture1b) = Lz (or Hz if specular) */
    glCombinerOutputNV(GL_COMBINER1_NV, GL_ALPHA,
      GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV,
      GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);

    /*** FINAL Combiner. ***/
    /* A = EF */
    glFinalCombinerInputNV(GL_VARIABLE_A_NV,
      GL_E_TIMES_F_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    /* B = EF */
    glFinalCombinerInputNV(GL_VARIABLE_B_NV,
      GL_E_TIMES_F_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    /* C = zero */
    glFinalCombinerInputNV(GL_VARIABLE_C_NV,
      GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    /* D = zero = no extra specular illumination contribution */
    glFinalCombinerInputNV(GL_VARIABLE_D_NV,
      GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    /* E = spare0rgb = diffuse illumination contribution = H dot N' */
    glFinalCombinerInputNV(GL_VARIABLE_E_NV,
      GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    /* F = spare0rgb = diffuse illumination contribution = H dot N' */
    glFinalCombinerInputNV(GL_VARIABLE_F_NV,
      GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

    /* specular RGB color = A*E*F
                          = specular modulated by self-shadowing term */

    /* G = spare0a = diffuse illumination contribution = H dot N' */
    glFinalCombinerInputNV(GL_VARIABLE_G_NV,
      GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_ALPHA);

	glEnable(GL_REGISTER_COMBINERS_NV);
}

void TObject::EnableNVRefract1() {
	glCombinerParameteriNV(GL_NUM_GENERAL_COMBINERS_NV, 1);

	GLfloat constant0[]={0.0f, 1.0f, 0.0f, 0.0f};
	glCombinerParameterfvNV(GL_CONSTANT_COLOR0_NV, constant0);
	GLfloat constant1[]={0.0f, 0.0f, 1.0f, 0.0f};
	glCombinerParameterfvNV(GL_CONSTANT_COLOR1_NV, constant1);

	//Combiner 0: green(tex1)+blue(tex0)->spare0.rgb
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_A_NV, GL_TEXTURE1_ARB,
					GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_B_NV, GL_CONSTANT_COLOR0_NV,
					GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_C_NV, GL_TEXTURE0_ARB,
					GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_D_NV, GL_CONSTANT_COLOR1_NV,
					GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glCombinerOutputNV(GL_COMBINER0_NV, GL_RGB, GL_DISCARD_NV, GL_DISCARD_NV, GL_SPARE0_NV,
					GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);

	glFinalCombinerInputNV(GL_VARIABLE_A_NV, GL_PRIMARY_COLOR_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glFinalCombinerInputNV(GL_VARIABLE_B_NV, GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glFinalCombinerInputNV(GL_VARIABLE_C_NV, GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glFinalCombinerInputNV(GL_VARIABLE_D_NV, GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

	glEnable(GL_REGISTER_COMBINERS_NV);
}

void TObject::EnableNVRefract2() {
	glCombinerParameteriNV(GL_NUM_GENERAL_COMBINERS_NV, 1);
	GLfloat constant0[]={1.0f, 0.0f, 0.0f, 0.0f};
	glCombinerParameterfvNV(GL_CONSTANT_COLOR0_NV, constant0);
	
	//Combiner 0 : red(tex0)->spare0.rgb
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_A_NV, GL_TEXTURE0_ARB,
					GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_B_NV, GL_CONSTANT_COLOR0_NV,
					GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glCombinerOutputNV(GL_COMBINER0_NV, GL_RGB, GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV,
					GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);

	glFinalCombinerInputNV(GL_VARIABLE_A_NV, GL_PRIMARY_COLOR_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glFinalCombinerInputNV(GL_VARIABLE_B_NV, GL_TEXTURE1_ARB, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glFinalCombinerInputNV(GL_VARIABLE_C_NV, GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glFinalCombinerInputNV(GL_VARIABLE_D_NV, GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

	glEnable(GL_REGISTER_COMBINERS_NV);
}


void TObject::EnableNVAttenuate(GLfloat r, GLfloat g, GLfloat b) {
	GLfloat color[4];
	color[0] = r;
	color[1] = g;
	color[2] = b;
	color[3] = 1.0f;
	glCombinerParameteriNV(GL_NUM_GENERAL_COMBINERS_NV, 1);
	glCombinerParameterfvNV(GL_CONSTANT_COLOR0_NV, (float*)&color);

	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_A_NV, GL_TEXTURE0_ARB, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_B_NV, GL_ZERO, GL_UNSIGNED_INVERT_NV, GL_RGB);
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_C_NV, GL_TEXTURE1_ARB, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_D_NV, GL_ZERO, GL_UNSIGNED_INVERT_NV, GL_RGB);
	glCombinerOutputNV(GL_COMBINER0_NV, GL_RGB, GL_DISCARD_NV, GL_DISCARD_NV, GL_SPARE0_NV, GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);

	glFinalCombinerInputNV(GL_VARIABLE_A_NV, GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glFinalCombinerInputNV(GL_VARIABLE_B_NV, GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glFinalCombinerInputNV(GL_VARIABLE_C_NV, GL_CONSTANT_COLOR0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glFinalCombinerInputNV(GL_VARIABLE_D_NV, GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
	glFinalCombinerInputNV(GL_VARIABLE_G_NV, GL_ZERO, GL_UNSIGNED_INVERT_NV, GL_ALPHA);

	glEnable(GL_REGISTER_COMBINERS_NV);
}

void TObject::SetLightPos(GLfloat x, GLfloat y, GLfloat z) {
	this->lightPos.x = x;
	this->lightPos.y = y;
	this->lightPos.z = z;
}

void TObject::SetEyePos(GLfloat x, GLfloat y, GLfloat z) {
	this->eyePos.x = x;
	this->eyePos.y = y;
	this->eyePos.z = z;
}

void TObject::SetupLightVectorsForRendering() {
	Vertex *curVertex = this->vertices;
	OrthoNormalBasis *curBasis = this->orthobasis;
	Connective *curConnective = this->vertexConnect;
	Vector addVec;
	addVec.x = addVec.y = addVec.z =0.5f;

	Vector attenLightPos, attenLightPos2;
	Vector lightVec;
	Vector eyeVec;
	transformVector(&lightVec, &(this->lightPos), (this->inverseTransformMatrix) );
	transformVector(&eyeVec, &(this->eyePos), (this->inverseTransformMatrix) );
		
	for(int i = 0; i < this->triangleNum * 3; i++) {
		//do light vector calculations
		Vsub((&lightVec), (&(curVertex->position)), (&(curVertex->lightPos)) );
		curVertex->tangentSpaceLightVector.x = Vdot((&(curVertex->lightPos)), (&(curBasis->tangent)) );
		curVertex->tangentSpaceLightVector.y = Vdot((&(curVertex->lightPos)), (&(curBasis->binormal)) );
		curVertex->tangentSpaceLightVector.z = Vdot((&(curVertex->lightPos)), (&(curBasis->normal)) );
		Vnormal((&(curVertex->tangentSpaceLightVector)) );
		//now do specular half-angle calculations
		Vsub((&eyeVec), (&(curVertex->position)), (&(curVertex->eyePos)) );
		curVertex->tangentSpaceHalfAngleVector.x = Vdot((&(curVertex->eyePos)), (&(curBasis->tangent)) );
		curVertex->tangentSpaceHalfAngleVector.y = Vdot((&(curVertex->eyePos)), (&(curBasis->binormal)) );
		curVertex->tangentSpaceHalfAngleVector.z = Vdot((&(curVertex->eyePos)), (&(curBasis->normal)) );  
		Vnormal((&(curVertex->tangentSpaceHalfAngleVector)) );
		Vadd((&(curVertex->tangentSpaceHalfAngleVector)), &(curVertex->tangentSpaceLightVector), &(curVertex->tangentSpaceHalfAngleVector) );
		Vnormal((&(curVertex->tangentSpaceHalfAngleVector)) );

		//do calculations for distance attenuation
		attenLightPos.x =  -curVertex->lightPos.x;
		attenLightPos.y =  -curVertex->lightPos.y;
		attenLightPos.z =  -curVertex->lightPos.z;

		Vscale(&(attenLightPos), 1.0f / (lightBrightness * 2.0f) );
		Vadd(&(attenLightPos), &addVec, &(curVertex->primaryVec));

		curVertex = curVertex + 1;
		curBasis = curBasis + 1;
	}

	curBasis = this->orthobasis;
	//now check if the plane is visible to the light for shadow volume stuff
	for(int j = 0; j < this->triangleNum; j++) {
		if(vecPlaneOrientation(&lightVec, &curConnective->plane) > 0) {
			curConnective->visible = true;
		}
		else curConnective->visible = false;     

		(curConnective + 1)->visible = curConnective->visible;
		(curConnective + 2)->visible = curConnective->visible;
		curConnective =  curConnective + 3;
		curBasis = curBasis + 3;
	}

}

void TObject::EnablePlainDecal()
{
	glActiveTextureARB( GL_TEXTURE1_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable( GL_TEXTURE_2D );

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable( GL_TEXTURE_2D );

	glDisable(GL_REGISTER_COMBINERS_NV);
	glDisable(GL_BLEND);
	glDepthFunc(GL_LEQUAL);

	glDisable(GL_LIGHTING);

	curRenderMode = DECAL_MODE;
}

void TObject:: EnableShadowedDecal() {
	glActiveTextureARB( GL_TEXTURE1_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable( GL_TEXTURE_2D );

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable( GL_TEXTURE_2D );

	glDisable(GL_REGISTER_COMBINERS_NV);
	glDisable(GL_BLEND);
	glDisable(GL_LIGHTING);

	glDepthFunc(GL_EQUAL);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_GEQUAL, 0, 0xffffffff);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

	curRenderMode = DECAL_MODE;
}

void TObject:: EnableStencilDecal(GLfloat r, GLfloat g, GLfloat b) {
	glActiveTextureARB( GL_TEXTURE1_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable( GL_TEXTURE_2D );

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable( GL_TEXTURE_2D );

	glDisable(GL_REGISTER_COMBINERS_NV);
	glDisable(GL_BLEND);
	glDisable(GL_LIGHTING);

	glDepthFunc(GL_EQUAL);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_GEQUAL, 0, 0xffffffff);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

	curRenderMode = DECAL_MODE;
}

void TObject::EnableDiffusePass(GLfloat aRed, GLfloat aGreen, GLfloat aBlue, GLfloat aAlpha, GLfloat dRed, GLfloat dGreen, GLfloat dBlue, GLfloat dAlpha) {
	EnableNVRegCombinersDiffuse(aRed, aGreen, aBlue, aAlpha, dRed, dGreen, dBlue, dAlpha);
	glDepthFunc(GL_EQUAL);

	glEnable(GL_BLEND);
	glBlendFunc(GL_DST_COLOR, GL_ZERO);

	curRenderMode = DIFFUSE_BUMP_MODE;

}

void TObject::EnableSpecularPass() {
//	glDisable(GL_STENCIL_TEST);
	EnableNVRegCombinersSpecular();
	glDepthFunc(GL_EQUAL);

	glEnable(GL_BLEND);
	glBlendFunc(GL_DST_ALPHA, GL_ONE);

	curRenderMode = SPECULAR_BUMP_MODE;
}

void TObject::EnableRefractPass1(GLfloat *mat) {
	glDisable(GL_BLEND);
	EnableNVRefract1();
	glBindProgramNV(GL_VERTEX_PROGRAM_NV, infrastructure->GetRefract_NV1_VP() );
	//Set track matrices
	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 0, GL_MODELVIEW_PROJECTION_NV, GL_IDENTITY_NV);
	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 4, GL_MODELVIEW, GL_IDENTITY_NV);
	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 8, GL_MODELVIEW, GL_INVERSE_TRANSPOSE_NV);
	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 12, GL_TEXTURE, GL_IDENTITY_NV);
	//Fresnel multipication factor in register c[28]
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 28, FRESNEL, FRESNEL, FRESNEL, 1.0f);
	//Refractive indices in registers: c[32], c[33] and c[34]
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 32, REFRACT_INDEX, REFRACT_INDEX*REFRACT_INDEX, 0.0f, 1.0f);
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 33, REFRACT_INDEX+REFRACT_INDEX_DELTA,
						(REFRACT_INDEX+REFRACT_INDEX_DELTA)*(REFRACT_INDEX+REFRACT_INDEX_DELTA), 0.0f, 1.0f);
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 34, REFRACT_INDEX+2*REFRACT_INDEX_DELTA,
						(REFRACT_INDEX+2*REFRACT_INDEX_DELTA)*(REFRACT_INDEX+2*REFRACT_INDEX_DELTA), 0.0f, 1.0f);
	//Constants in register c[64]
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 64, 0.0f, 1.0f, 2.0f, 0.5f);

	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glActiveTextureARB(GL_TEXTURE0_ARB);
	glDisable(GL_TEXTURE_2D);
	glEnable(GL_TEXTURE_CUBE_MAP_ARB);
	glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, GetCurrentCubeMap());

	glActiveTextureARB(GL_TEXTURE1_ARB);
	glDisable(GL_TEXTURE_2D);
	glEnable(GL_TEXTURE_CUBE_MAP_ARB);
	glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, GetCurrentCubeMap());

	glMatrixMode(GL_TEXTURE);
	glLoadIdentity();
	glMultMatrixf(mat);
	glMatrixMode(GL_MODELVIEW);
	
	curRenderMode = REFRACT_MODE_1;
}

void TObject::EnableRefractPass2(GLfloat *mat) {
	EnableNVRefract2();
	glBindProgramNV(GL_VERTEX_PROGRAM_NV, infrastructure->GetRefract_NV2_VP() );
	//Set track matrices
/*	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 0, GL_MODELVIEW_PROJECTION_NV, GL_IDENTITY_NV);
	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 4, GL_MODELVIEW, GL_IDENTITY_NV);
	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 8, GL_MODELVIEW, GL_INVERSE_TRANSPOSE_NV);
	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 12, GL_TEXTURE, GL_IDENTITY_NV);
	//Fresnel multipication factor in register c[28]
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 28, FRESNEL, FRESNEL, FRESNEL, 1.0f);
	//Refractive indices in registers: c[32], c[33] and c[34]
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 32, REFRACT_INDEX, REFRACT_INDEX*REFRACT_INDEX, 0.0f, 1.0f);
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 33, REFRACT_INDEX+REFRACT_INDEX_DELTA,
						(REFRACT_INDEX+REFRACT_INDEX_DELTA)*(REFRACT_INDEX+REFRACT_INDEX_DELTA), 0.0f, 1.0f);
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 34, REFRACT_INDEX+2*REFRACT_INDEX_DELTA,
						(REFRACT_INDEX+2*REFRACT_INDEX_DELTA)*(REFRACT_INDEX+2*REFRACT_INDEX_DELTA), 0.0f, 1.0f); 
	//Constants in register c[64]
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 64, 0.0f, 1.0f, 2.0f, 0.5f);  */

	
	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glActiveTextureARB(GL_TEXTURE0_ARB);
	glDisable(GL_TEXTURE_2D);
	glEnable(GL_TEXTURE_CUBE_MAP_ARB);
	glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, GetCurrentCubeMap());

	glActiveTextureARB(GL_TEXTURE1_ARB);
	glDisable(GL_TEXTURE_2D);
	glEnable(GL_TEXTURE_CUBE_MAP_ARB);
	glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, GetCurrentCubeMap());
		
	glBlendFunc(GL_ONE, GL_ONE);
	glEnable(GL_BLEND);

	glMatrixMode(GL_TEXTURE);
	glLoadIdentity();
	glMultMatrixf(mat);
	glMatrixMode(GL_MODELVIEW);
	
	curRenderMode = REFRACT_MODE_2;
}

void TObject::EnableAttenuatePass(GLfloat r, GLfloat g, GLfloat b) {
	//active texture 1 (the 1D radial ramp texture)
	glActiveTextureARB(GL_TEXTURE1_ARB);
	glEnable(GL_TEXTURE_1D);
	glBindTexture(GL_TEXTURE_1D, this->infrastructure->GetAtten1D());

	//active texture 0 (the 2D radial map texture)
	glActiveTextureARB(GL_TEXTURE0_ARB);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, this->infrastructure->GetAtten2D());
	EnableNVAttenuate(r, g, b);

	glEnable(GL_BLEND);
	glBlendFunc(GL_DST_COLOR, GL_ZERO);

	curRenderMode = ATTENUATE_MODE;

}

void TObject::EnableAttenuatePass2(GLfloat r, GLfloat g, GLfloat b) {
	//active texture 1 (the 1D radial ramp texture)
	glActiveTextureARB(GL_TEXTURE1_ARB);
	glEnable(GL_TEXTURE_1D);
	glBindTexture(GL_TEXTURE_1D, this->infrastructure->GetAtten1D());

	//active texture 0 (the 2D radial map texture)
	glActiveTextureARB(GL_TEXTURE0_ARB);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, this->infrastructure->GetAtten2D());
	EnableNVAttenuate(r, g, b);

	glEnable(GL_BLEND);
//	glBlendFunc(GL_SRC_ALPHA,GL_ONE);
	glBlendFunc(GL_ONE, GL_ONE);

	curRenderMode = ATTENUATE_MODE;

}

/*shadow volume algorithm using ZFail or ZPass, based on "Practical and Robust Stenciled Shadow Volumes for
	Hardware-Accelerated Rendering" by Case Everitt and Mark J.Kilgard   */
void TObject::CastShadowDepthPass(bool flag) {
	Vertex *curVertex = this->vertices;
	Connective *curConnective = this->vertexConnect;
	Connective *theConnectives = this->vertexConnect;

	Vector lightVec;
	transformVector(&lightVec, &(this->lightPos), (this->inverseTransformMatrix) );

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadMatrixf(this->transformMatrix);

	if(flag == true) {
		glCullFace(GL_BACK);
		glStencilOp(GL_KEEP, GL_KEEP, GL_INCR);
		ShadowPass(curConnective, curVertex, theConnectives, &lightVec, true);

		glCullFace(GL_FRONT);
		glStencilOp(GL_KEEP, GL_KEEP, GL_DECR);
		ShadowPass(curConnective, curVertex, theConnectives, &lightVec, false);
	}
	else {
		glCullFace(GL_FRONT);
		glStencilOp(GL_KEEP, GL_INCR, GL_KEEP);
		ShadowPass(curConnective, curVertex, theConnectives, &lightVec, true);
		CapShadowVolume(theConnectives, curVertex, &lightVec);

		glCullFace(GL_BACK);
		glStencilOp(GL_KEEP, GL_DECR, GL_KEEP);
		ShadowPass(curConnective, curVertex, theConnectives, &lightVec, false);
		CapShadowVolume(theConnectives, curVertex, &lightVec);
	}

	


	glCullFace(GL_BACK);
	glPopMatrix();
}

void TObject::ShadowPass(Connective *curConnective, Vertex *curVertex, Connective *theConnectives, Vector *lightVec, bool flag) {
	glBegin(GL_QUADS);
	for(int i = 0; i < this->triangleNum; i++) {
			
		if(curConnective->visible == true) {
			for(int j = 0; j < 3; j++) {
				if((curConnective + j)->neighbourIndex == UNDEFINED || (theConnectives + ((curConnective + j)->neighbourIndex))->visible == false) {
					Vector v1 = (curVertex + j)->position;
					Vector v2 = (curVertex + ((j + 1) % 3) )->position;
					Vector v3, v4;
					Vsub(&v1, lightVec, &v3);
					Vsub(&v2, lightVec, &v4);

						glVertex3f(v2.x, v2.y, v2.z);
						glVertex3f(v1.x, v1.y, v1.z);
						glVertex4f(v3.x, v3.y, v3.z, 0.0f);
						glVertex4f(v4.x, v4.y, v4.z, 0.0f);
				}
			}
		}
		curConnective = curConnective + 3;
		curVertex = curVertex + 3;
	}

	glEnd();
}

void TObject::CapShadowVolume(Connective *curConnective, Vertex *curVertex, Vector *lightVec) {
	Vector v1, v2, v3;
	Vector *vpnt1, *vpnt2, *vpnt3;
	glBegin(GL_TRIANGLES);
	for(int i = 0; i < this->triangleNum; i++) {
		vpnt1 = &((curVertex)->position);
		vpnt2 = &((curVertex + 1)->position);
		vpnt3 = &((curVertex + 2)->position);

		if(curConnective->visible == false) {
			Vsub(vpnt1, lightVec, &v1);
			Vsub(vpnt2, lightVec, &v2);
			Vsub(vpnt3, lightVec, &v3);

			glVertex4f(v1.x, v1.y, v1.z, 0);
			glVertex4f(v2.x, v2.y, v2.z, 0);
			glVertex4f(v3.x, v3.y, v3.z, 0);
		}
		else {
			glVertex3f(vpnt1->x, vpnt1->y, vpnt1->z);
			glVertex3f(vpnt2->x, vpnt2->y, vpnt2->z);
			glVertex3f(vpnt3->x, vpnt3->y, vpnt3->z);
		}
		
		
		curConnective = curConnective + 3;
		curVertex = curVertex + 3;
	}

	glEnd();
}


bool TObject::IsDegenerate(Vertex *curTriangle) {
	Vector *v1 = &(curTriangle->position);
	Vector *v2 = &((curTriangle + 1)->position);
	Vector *v3 = &((curTriangle + 2)->position);

	if(Vequal(v1, v2) || Vequal(v1, v3) || Vequal(v2, v3)) return true;

	return false;
}

//calculates bounding box for object
void TObject::CalcBounds() {
	Bound1.x = Bound1.y = Bound1.z = 999999.0f;
	Bound2.x = Bound2.y = Bound2.z = -999999.0f;
	Vertex *curvertex = this->vertices;
	
	for(int i = 0; i < this->triangleNum * 3; i++) {
		if(curvertex->position.x < Bound1.x) Bound1.x = curvertex->position.x;
		if(curvertex->position.y < Bound1.y) Bound1.y = curvertex->position.y;
		if(curvertex->position.z < Bound1.z) Bound1.z = curvertex->position.z;
		
		if(curvertex->position.x > Bound2.x) Bound2.x = curvertex->position.x;
		if(curvertex->position.y > Bound2.y) Bound2.y = curvertex->position.y;
		if(curvertex->position.z > Bound2.z) Bound2.z = curvertex->position.z;

		curvertex = curvertex + 1;
	}
	
	
	Bound3.x = Bound2.x;
	Bound3.y = Bound1.y;
	Bound3.z = Bound1.z;

	Bound4.x = Bound2.x;
	Bound4.y = Bound1.y;
	Bound4.z = Bound2.z;

	Bound5.x = Bound1.x;
	Bound5.y = Bound1.y;
	Bound5.z = Bound2.z;

	Bound6.x = Bound2.x;
	Bound6.y = Bound2.y;
	Bound6.z = Bound1.z;

	Bound7.x = Bound1.x;
	Bound7.y = Bound2.y;
	Bound7.z = Bound2.z;

	Bound8.x = Bound1.x;
	Bound8.y = Bound2.y;
	Bound8.z = Bound1.z;
}

Vector TObject::GetMaxBounds() {
	Vector MaxBound = DstBound1;

	if(DstBound2.x > MaxBound.x) MaxBound.x = DstBound2.x;
	if(DstBound2.y > MaxBound.y) MaxBound.y = DstBound2.y;
	if(DstBound2.z > MaxBound.z) MaxBound.z = DstBound2.z;

	if(DstBound3.x > MaxBound.x) MaxBound.x = DstBound3.x;
	if(DstBound3.y > MaxBound.y) MaxBound.y = DstBound3.y;
	if(DstBound3.z > MaxBound.z) MaxBound.z = DstBound3.z;

	if(DstBound4.x > MaxBound.x) MaxBound.x = DstBound4.x;
	if(DstBound4.y > MaxBound.y) MaxBound.y = DstBound4.y;
	if(DstBound4.z > MaxBound.z) MaxBound.z = DstBound4.z;

	if(DstBound5.x > MaxBound.x) MaxBound.x = DstBound5.x;
	if(DstBound5.y > MaxBound.y) MaxBound.y = DstBound5.y;
	if(DstBound5.z > MaxBound.z) MaxBound.z = DstBound5.z;

	if(DstBound6.x > MaxBound.x) MaxBound.x = DstBound6.x;
	if(DstBound6.y > MaxBound.y) MaxBound.y = DstBound6.y;
	if(DstBound6.z > MaxBound.z) MaxBound.z = DstBound6.z;

	if(DstBound7.x > MaxBound.x) MaxBound.x = DstBound7.x;
	if(DstBound7.y > MaxBound.y) MaxBound.y = DstBound7.y;
	if(DstBound7.z > MaxBound.z) MaxBound.z = DstBound7.z;

	if(DstBound8.x > MaxBound.x) MaxBound.x = DstBound8.x;
	if(DstBound8.y > MaxBound.y) MaxBound.y = DstBound8.y;
	if(DstBound8.z > MaxBound.z) MaxBound.z = DstBound8.z;

	return MaxBound;
}

Vector TObject::GetMinBounds() {
	Vector MinBound = DstBound1;

	if(DstBound2.x < MinBound.x) MinBound.x = DstBound2.x;
	if(DstBound2.y < MinBound.y) MinBound.y = DstBound2.y;
	if(DstBound2.z < MinBound.z) MinBound.z = DstBound2.z;

	if(DstBound3.x < MinBound.x) MinBound.x = DstBound3.x;
	if(DstBound3.y < MinBound.y) MinBound.y = DstBound3.y;
	if(DstBound3.z < MinBound.z) MinBound.z = DstBound3.z;

	if(DstBound4.x < MinBound.x) MinBound.x = DstBound4.x;
	if(DstBound4.y < MinBound.y) MinBound.y = DstBound4.y;
	if(DstBound4.z < MinBound.z) MinBound.z = DstBound4.z;

	if(DstBound5.x < MinBound.x) MinBound.x = DstBound5.x;
	if(DstBound5.y < MinBound.y) MinBound.y = DstBound5.y;
	if(DstBound5.z < MinBound.z) MinBound.z = DstBound5.z;

	if(DstBound6.x < MinBound.x) MinBound.x = DstBound6.x;
	if(DstBound6.y < MinBound.y) MinBound.y = DstBound6.y;
	if(DstBound6.z < MinBound.z) MinBound.z = DstBound6.z;

	if(DstBound7.x < MinBound.x) MinBound.x = DstBound7.x;
	if(DstBound7.y < MinBound.y) MinBound.y = DstBound7.y;
	if(DstBound7.z < MinBound.z) MinBound.z = DstBound7.z;

	if(DstBound8.x < MinBound.x) MinBound.x = DstBound8.x;
	if(DstBound8.y < MinBound.y) MinBound.y = DstBound8.y;
	if(DstBound8.z < MinBound.z) MinBound.z = DstBound8.z;

	return MinBound;
}

Vector TObject::GetBndMidPnt() {
	Vector midPnt;
	
	Vadd(&GetMinBounds(), &GetMaxBounds(), &midPnt);
	Vscale(&midPnt, 0.5f);
	
	return midPnt;
}

Vector TObject::GetBndOrgMidPnt() {
	Vector midPnt;
	
	Vadd(&GetMinOrgBounds(), &GetMaxOrgBounds(), &midPnt);
	Vscale(&midPnt, 0.5f);
	
	return midPnt;
}


Vector TObject::GetMaxOrgBounds() {
	Vector MaxBound = Bound1;

	if(Bound2.x > MaxBound.x) MaxBound.x = Bound2.x;
	if(Bound2.y > MaxBound.y) MaxBound.y = Bound2.y;
	if(Bound2.z > MaxBound.z) MaxBound.z = Bound2.z;

	if(Bound3.x > MaxBound.x) MaxBound.x = Bound3.x;
	if(Bound3.y > MaxBound.y) MaxBound.y = Bound3.y;
	if(Bound3.z > MaxBound.z) MaxBound.z = Bound3.z;

	if(Bound4.x > MaxBound.x) MaxBound.x = Bound4.x;
	if(Bound4.y > MaxBound.y) MaxBound.y = Bound4.y;
	if(Bound4.z > MaxBound.z) MaxBound.z = Bound4.z;

	if(Bound5.x > MaxBound.x) MaxBound.x = Bound5.x;
	if(Bound5.y > MaxBound.y) MaxBound.y = Bound5.y;
	if(Bound5.z > MaxBound.z) MaxBound.z = Bound5.z;

	if(Bound6.x > MaxBound.x) MaxBound.x = Bound6.x;
	if(Bound6.y > MaxBound.y) MaxBound.y = Bound6.y;
	if(Bound6.z > MaxBound.z) MaxBound.z = Bound6.z;

	if(Bound7.x > MaxBound.x) MaxBound.x = Bound7.x;
	if(Bound7.y > MaxBound.y) MaxBound.y = Bound7.y;
	if(Bound7.z > MaxBound.z) MaxBound.z = Bound7.z;

	if(Bound8.x > MaxBound.x) MaxBound.x = Bound8.x;
	if(Bound8.y > MaxBound.y) MaxBound.y = Bound8.y;
	if(Bound8.z > MaxBound.z) MaxBound.z = Bound8.z;

	return MaxBound;
}

Vector TObject::GetMinOrgBounds() {
	Vector MinBound = Bound1;

	if(Bound2.x < MinBound.x) MinBound.x = Bound2.x;
	if(Bound2.y < MinBound.y) MinBound.y = Bound2.y;
	if(Bound2.z < MinBound.z) MinBound.z = Bound2.z;

	if(Bound3.x < MinBound.x) MinBound.x = Bound3.x;
	if(Bound3.y < MinBound.y) MinBound.y = Bound3.y;
	if(Bound3.z < MinBound.z) MinBound.z = Bound3.z;

	if(Bound4.x < MinBound.x) MinBound.x = Bound4.x;
	if(Bound4.y < MinBound.y) MinBound.y = Bound4.y;
	if(Bound4.z < MinBound.z) MinBound.z = Bound4.z;

	if(Bound5.x < MinBound.x) MinBound.x = Bound5.x;
	if(Bound5.y < MinBound.y) MinBound.y = Bound5.y;
	if(Bound5.z < MinBound.z) MinBound.z = Bound5.z;

	if(Bound6.x < MinBound.x) MinBound.x = Bound6.x;
	if(Bound6.y < MinBound.y) MinBound.y = Bound6.y;
	if(Bound6.z < MinBound.z) MinBound.z = Bound6.z;

	if(Bound7.x < MinBound.x) MinBound.x = Bound7.x;
	if(Bound7.y < MinBound.y) MinBound.y = Bound7.y;
	if(Bound7.z < MinBound.z) MinBound.z = Bound7.z;

	if(Bound8.x < MinBound.x) MinBound.x = Bound8.x;
	if(Bound8.y < MinBound.y) MinBound.y = Bound8.y;
	if(Bound8.z < MinBound.z) MinBound.z = Bound8.z;

	return MinBound;
}


void TObject::createCubeMapTextures() {
	cout << "Creating Refraction Map Textures for Object: " << this << endl;
	//create 4 sets of Cube Maps of sizes: 32x32, 64x64,128x128 and 256x256
	glGenTextures(1, &CubeMap32Ref);
	createCubeMapTextureSet(32, CubeMap32Ref);

	glGenTextures(1, &CubeMap64Ref);
	createCubeMapTextureSet(64, CubeMap64Ref);

	glGenTextures(1, &CubeMap128Ref);
	createCubeMapTextureSet(128, CubeMap128Ref);

	glGenTextures(1, &CubeMap256Ref);
	createCubeMapTextureSet(256, CubeMap256Ref);
}

void TObject::createCubeMapTextureSet(int size, int ID) {
	unsigned int *pTexture = new unsigned int [size * size * 4];
	memset(pTexture, 0, size * size * 4 * sizeof(unsigned int));
	
	glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, ID);
	glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X_ARB,
					0, 4, size, size, 0,
					GL_RGBA, GL_UNSIGNED_INT, pTexture);
	
	glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_X_ARB,
					0, 4, size, size, 0,
					GL_RGBA, GL_UNSIGNED_INT, pTexture);

	glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_Y_ARB,
					0, 4, size, size, 0,
					GL_RGBA, GL_UNSIGNED_INT, pTexture);
	
	glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_ARB,
					0, 4, size, size, 0,
					GL_RGBA, GL_UNSIGNED_INT, pTexture);

	glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_Z_ARB,
					0, 4, size, size, 0,
					GL_RGBA, GL_UNSIGNED_INT, pTexture);
	
	glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_ARB,
					0, 4, size, size, 0,
					GL_RGBA, GL_UNSIGNED_INT, pTexture);

	glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	
	delete [] pTexture;
}

void TObject::copyScreenToCubeMap(GLenum target) {
	if(CubeMap32Ref == 0 || CubeMap64Ref == 0 || CubeMap128Ref == 0) {
//		createCubeMapTextures();
	}
	
	glActiveTextureARB(GL_TEXTURE1_ARB );
	glDisable(GL_TEXTURE_CUBE_MAP_ARB );
	glDisable(GL_TEXTURE_2D);

	glActiveTextureARB( GL_TEXTURE0_ARB );
	glDisable( GL_TEXTURE_CUBE_MAP_ARB );
	glDisable(GL_REGISTER_COMBINERS_NV);
	glDisable(GL_TEXTURE_2D);
	
	glEnable( GL_TEXTURE_CUBE_MAP_ARB);
	int size = 0;
	switch(cubeMapSelect) {
		case 0:
			glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, CubeMap32Ref);
			size = 32;
		break;
		
		case 1:
			glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, CubeMap64Ref);
			size = 64;
		break;
		
		case 2:
			glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, CubeMap128Ref);
			size = 128;
		break;
		
		case 3:
			glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, CubeMap256Ref);
			size = 256;
		break;
	}
	
	
	
//	glCopyTexImage2D(target, 0, GL_RGB, 0, 0, size, size, 0);
	glCopyTexSubImage2D(target, 0, 0, 0, 0, 0, size, size);
	glDisable( GL_TEXTURE_CUBE_MAP_ARB);
}

int TObject::GetCurrentCubeMap() {
	int retval = 0;
	switch(cubeMapSelect) {
		case 0:
			retval = CubeMap32Ref;
		break;
		
		case 1:
			retval = CubeMap64Ref;
		break;
		
		case 2:
			retval = CubeMap128Ref;
		break;
		
		case 3:
			retval = CubeMap256Ref;
		break;
	}
	
	return retval;
}




