/***************************************************************************
                          tmd2.h  -  loads an MD2 file. Also does animation
                          with reference to :
                           Ben Humphrey (DigiBen), Game Programmer
                           DigiBen@GameTutorials.com
                           Co-Web Host of www.GameTutorials.com

                             -------------------
    begin                : Thu Dec 26 2002
    copyright            : (C) 2002 by Chong Jiayi
    email                : jychong@stanford.edu
 ***************************************************************************/


#ifndef TMD2_H
#define TMD2_H

#include <tobject.h>

#define MD2_MAX_TRIANGLES       4096
#define MD2_MAX_VERTICES        2048
#define MD2_MAX_TEXCOORDS       2048
#define MD2_MAX_FRAMES          512
#define MD2_MAX_SKINS           32
#define MD2_MAX_FRAMESIZE       (MD2_MAX_VERTICES * 4 + 128)

// This stores the speed of the animation between each key frame
#define kAnimationSpeed         5.0f

typedef unsigned char byte;

// This holds the header information that is read in at the beginning of the file
struct tMd2Header {
   int magic;                   // This is used to identify the file
   int version;                 // The version number of the file (Must be 8)
   int skinWidth;               // The skin width in pixels
   int skinHeight;              // The skin height in pixels
   int frameSize;               // The size in bytes the frames are
   int numSkins;                // The number of skins associated with the model
   int numVertices;             // The number of vertices (constant for each frame)
   int numTexCoords;            // The number of texture coordinates
   int numTriangles;            // The number of faces (polygons)
   int numGlCommands;           // The number of gl commands
   int numFrames;               // The number of animation frames
   int offsetSkins;             // The offset in the file for the skin data
   int offsetTexCoords;         // The offset in the file for the texture data
   int offsetTriangles;         // The offset in the file for the face data
   int offsetFrames;            // The offset in the file for the frames data
   int offsetGlCommands;        // The offset in the file for the gl commands data
   int offsetEnd;               // The end of the file offset
};


// This is used to store the vertices that are read in for the current frame
struct tMd2AliasTriangle {
   byte vertex[3];
   byte lightNormalIndex;
};

// This stores the normals and vertices for the frames
struct tMd2Triangle {
   float vertex[3];
   float normal[3];
};

// This stores the indices into the vertex and texture coordinate arrays
struct tMd2Face {
   short vertexIndices[3];
   short textureIndices[3];
};

// This stores UV coordinates
struct tMd2TexCoord {
   short u, v;
};

// This stores the animation scale, translation and name information for a frame, plus verts
struct tMd2AliasFrame {
   float scale[3];
   float translate[3];
   char name[16];
   tMd2AliasTriangle aliasVertices[1];
};

// This stores the frames vertices after they have been transformed
struct tMd2Frame {
   char strName[16];
   tMd2Triangle *pVertices;
};

// This stores a skin name
typedef char tMd2Skin[64];

//info for the MD2 animation
struct tMD2AnimationInfo {
	char Name[255];          // This stores the name of the animation (Jump, Pain, etc..)
	int startFrame;             // This stores the first frame number for this animation
	int endFrame;               // This stores the last frame number for this animation
};



/**
  *@author Chong Jiayi
  */

class TMd2 : public TObject  {

protected:
	tMd2Header m_Header;           // The header data
	tMd2Skin *m_pSkins;          // The skin data
	tMd2TexCoord *m_pTexCoords;      // The texture coordinates
	tMd2Face *m_pTriangles;      // Face index information
	tMd2Frame *m_pFrames;         // The frames of animation (vertices)

	vector<Vertex *> verticesList;
	vector<OrthoNormalBasis *> orthoList;
	vector<tMD2AnimationInfo> animList;
	vector<Connective *> connectList;
	int numFrames;
	int startKeyFrame;
	int curKeyFrame;
	int curTransitionFrame;
	int endKeyFrame;
	int interpolateNum;
	bool flipNormals;
	Vertex *animateVertices;
	OrthoNormalBasis *animateBasis;
	Connective *animateConnect;
	
	void RetrieveMd2(FILE *filePtr);
	void ParseToInternalModel();
	void CalculateNormals(Vertex *vertices, OrthoNormalBasis *basis, int verticeNum);
	void ExtractAnimations();
	void EndFileIO(FILE *filePtr);

public: 
	TMd2(TInfrastructure *curInfrastructure);
	~TMd2();
	bool Loadable() { return true; }
	void LoadGeom(char *filename);
	int GetNumKeyFrames() const { return numFrames; }
	int GetNumAnimation() const { return animList.size(); }
	void SetCurrentAnimation(int animationIndex);
	void SetLoadFlippedNormals(bool flag) { flipNormals = flag; }
	void SetStaticKeyFrame(int keyFrame);
	void SetCurrentAnimation(int animationIndex, int smoothness);
	void PlayAnimation();
};

#endif
