/* File: wwindow.h
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 *	CS248 Fall 2001
 *	HW3 - Video Game
 *
 *  Declares the CWWindow class, which is the basic container control in the
 *	Widget UI. (WWindow = Widget Window.) A CWWindow can hold items and other
 *	controls such as check boxes and text boxes. A CWWindow can *not* hold other
 *	CWWindows or controls that are not derived from CWItem. CWWindow has an array
 *	of CWItems and a count, and allows adding items and sorting them.
 *
 *	There is a static member of CWWindow that is a linked list of all the
 *	windows currently open in the system. This is useful for event handling,
 *	etc.
 *
 *	Here is the class hierarchy:
 *
 *
 *							CWidget
 *						  /	  |		-
 *						/	  |		  -
 *				CWWindow	CWItem		CWScrollBar
 *				  /			 /	-
 *				/			/	  -
 *	CWScrolledWindow  CWTextBox	CWCheckBox
 *
 *
 *	NOTE: CWTextbox has been suspended since the game doesn't need it.
 *
 */


#ifndef _WWINDOW_H
#define _WWINDOW_H

#include "widget.h"


// -----------------------------------------------------------------------------
// FORWARD DECLARATIONS
// -----------------------------------------------------------------------------
class CWScrolledWindow;
class CWItem;
class CWCheckbox;


// -----------------------------------------------------------------------------
// CWWINDOW CLASS
// -----------------------------------------------------------------------------
class CWWindow : public CWidget
{
public:
	/* Ctor
	 * ----
	 * Creates a new window with the given caption at the point (x, y) in
	 * screen coordinates.
	 */
	CWWindow(const string &caption, int x, int y);

	// dtor
	virtual ~CWWindow();

	/* Add*
	 * ----
	 * Add an item, text box, or checkbox to the window. Returns a pointer to
	 * the item that was added. You must pass a callback which will be called
	 * when the item is selected. When the callback is called, a pointer to the
	 * selected item is passed as the argument.
	 *
	 * You can also pass a pointer to user-defined data. (This is optional...
	 * if you don't need it, just pass NULL for the userData parameter.)
	 */
	virtual CWItem *AddItem(
		const string &caption,
		void (*callback)(CWItem *item),
		void *userData = NULL);

	virtual CWCheckbox *AddCheckbox(
		const string &caption,
		void (*callback)(CWItem *item),
		bool checked,
		void *userData = NULL);

//	void AddTextBox(const string &caption, int width,
//		const string &text = string::c_empty_str);

	/* OpenChild, CloseChild
	 * ---------
	 * Accessors for this window's child window. OpenChild creates a child
	 * window and returns a pointer to it, CloseChild closes the current child
	 * window (if there is one). A window can only have one child window at a
	 * time.
	 * OpenChild takes a pointer to a CWItem - this is the item that the child
	 * window is logically connected to. If possible, the child window will be
	 * drawn next to this item.
	 */
	virtual CWWindow			*OpenChild(CWItem *parent);
	virtual CWScrolledWindow	*OpenChild(CWItem *parent, int width, int height);
	virtual void				CloseChild();

	/* Draw
	 * ----
	 * Overridden from CWidget. Draws the window and the items it contains.
	 */
	virtual void Draw();

	/* Select
	 * ------
	 * Overridden from CWidget.
	 */
	virtual void Select();

	/* Sort
	 * ----
	 * Sorts the items in the window alphabetically.
	 */
	void Sort(bool ascending = true);


	// static member functions...

	/* IsAWindowOpen
	 * -------------
	 * Returns true if any window is open, false otherwise.
	 */
	static bool IsAWindowOpen()	{ return (sList != 0); }

	/* CloseAll
	 * --------
	 * Closes all open windows.
	 */
	static void CloseAll();


protected:
	friend class CWidget;	// allows CWidget::DrawAll to access linked list
	friend class CWItem;	// allows items to get their parent's dimensions

	// ctor (used if this is a child window)
	CWWindow(CWItem *parent);

	// helper methods
	void Alloc(int size);
	void PreAdd();
	virtual void PostAdd();
	void FindBorderCoords();
	virtual void Reposition();
	virtual CWidget *WidgetUnderMouse();

	void AddToList();
	void RemoveFromList();


	// members
	CWItem			**mItems;
	int				mNumItems, mAlloc;
	CWWindow		*mChild;			// non-NULL if this window has a child
	CWItem			*mParent;			// non-NULL if this is a child window
	int				mLeftx, mRightx;	// used for drawing border with caption


	// the linked list of all existing CWWindows. this should NOT be allowed
	// to get very large. (on the order of tens at most.)
	static CWWindow	*sList;

	// points to the next window in the linked list
	CWWindow		*next;


//	Inherited from CWidget:
//
//	int			mx, my, mWidth, mHeight;
//	string		mCaption;



private:
	CWWindow();	// disallow empty constructor
};

#endif	// _WWINDOW_H
