/* File: wwindow.cpp
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 *	CS248 Fall 2001
 *	HW3 - Video Game
 *
 *  Defines the CWWindow class, which is the window class for the Widget UI. For
 *	more information, see wwindow.h.
 *
 */


#include <GL/glut.h>
#include <assert.h>
#include "witem.h"
#include "wcheckbox.h"
#include "wscrolledwindow.h"
#include "wwindow.h"


// ----------------------------------------------------------------------------
// initialize static members
// ----------------------------------------------------------------------------
CWWindow	*CWWindow::sList = 0;


// ----------------------------------------------------------------------------
// public methods
// ----------------------------------------------------------------------------

// ctor
CWWindow::CWWindow(const string &caption, int x, int y)
	: CWidget(caption, x, y), mItems(NULL), mNumItems(0), mAlloc(0)
{
	assert(CWidget::sScreenWidth > 0 && CWidget::sScreenHeight > 0);

	mWidth = StringWidth(caption) + kWndCaptionOffset + kWndMargin;
	mHeight = kWndMargin;

	mParent = NULL;
	mChild = NULL;
	mLeftx = mRightx = 0;

	Reposition();

	AddToList();	// add to linked list that stores all CWindows
}


// dtor
CWWindow::~CWWindow()
{
	for (int i = 0; i < mNumItems; i++)
		DEL(mItems[i]);

	ARRAYDEL(mItems);

	if (mChild)
		CloseChild();

	RemoveFromList();	// add to linked list that stores all CWindows
}


// AddItem
CWItem *CWWindow::AddItem(const string & caption, CWItem::ItemFn callback,
						  void *userData)	// = NULL
{
	CWItem *add;

	PreAdd();	// expands the mItems array if necessary

	add = new CWItem(caption, callback, this, userData);
	assert(add);
	mItems[mNumItems++] = add;

	PostAdd();	// resizes the window if necessary

	return add;
}

// AddCheckBox
CWCheckbox *CWWindow::AddCheckbox(const string &caption, CWItem::ItemFn callback,
								  bool checked, void *userData)	// = NULL
{
	CWCheckbox *add;

	PreAdd();	// expands the mItems array if necessary

	add = new CWCheckbox(caption, callback, checked, this, userData);
	assert(add);
	mItems[mNumItems++] = add;

	PostAdd();	// resizes the window if necessary

	return add;
}

/*
// AddTextBox
void CWWindow::AddTextBox(const string &caption, int width, const string &text)
{
	// sanity check
	assert(mNumItems <= mAlloc);

	if (mNumItems == mAlloc)
		Resize(mAlloc + 1);

	mItems[mNumItems++] = new CTextBox(caption, width, text);
}
*/


// OpenChild, CloseChild
CWWindow *CWWindow::OpenChild(CWItem *parent)
{
	CloseChild();
	mChild = new CWWindow(parent);
	return mChild;
}

CWScrolledWindow *CWWindow::OpenChild(CWItem *parent, int width, int height)
{
	CloseChild();
	mChild = new CWScrolledWindow(parent, width, height);
	return (CWScrolledWindow *)mChild;
}

void CWWindow::CloseChild()
{
	// took out this check for convenience - can call CloseChild just to make
	// sure a window doesn't have a child open, whether or not it did in the
	// first place
//	assert(mChild);
	DEL(mChild);
}


// CloseAll
void CWWindow::CloseAll()
{
	while (sList)
		DEL(sList);
}



// Select
void CWWindow::Select()
{
//	CConsole::Print("CWWindow %s was selected!", mCaption.as_cstr());
}


// Sort
void CWWindow::Sort(bool ascending)
{
	qsort(mItems, (uint)mNumItems, sizeof(CWItem *), CWItem::CompareItems);

	Reposition();
}



// ----------------------------------------------------------------------------
// protected helper methods
// ----------------------------------------------------------------------------

// ctor (used if this is a child window)
CWWindow::CWWindow(CWItem *parent) : CWidget(parent->mCaption, 0, 0),
	mItems(NULL), mNumItems(0), mAlloc(0), mParent(parent)
{
	mWidth = StringWidth(mCaption) + kWndCaptionOffset + kWndMargin;
	mHeight = kWndMargin;

	mChild = NULL;
	mLeftx = mRightx = 0;

	Reposition();

	AddToList();	// add to linked list that stores all CWindows
}


/* Draw
 * ----
 * Draws the window and the items it contains.
 */
void CWWindow::Draw()
{
	int i;

	// sanity check
	// commented out for convenience - all Widget UI code can deal with a
	// window with no items, but if it happens, the code that created the
	// CWWindow is probably broken
	//assert(mNumItems > 0);

	// draw background
	SetColor(kBackColor);
	DrawRect(mx, my, mWidth, mHeight, true);

	if (mLeftx == 0 || mRightx == 0)
		FindBorderCoords();

	assert(!glIsEnabled(GL_SCISSOR_TEST));
	glEnable(GL_SCISSOR_TEST);
	glScissor(mx, sScreenHeight - (my + mHeight),
			  mWidth, mHeight - kFontHeight / 2 - kFontHeightMargin);

	// draw items
	for (i = 0; i < mNumItems; i++)
		mItems[i]->Draw();

	glDisable(GL_SCISSOR_TEST);

	if (mChild)	// if we're not the active window, obscure item captions
	{
		SetColor(kObscureColor);
		DrawRect(mx, my, mWidth, mHeight, true);
	}

	// draw caption
	DrawString(mx + kWndCaptionOffset,
		my + kFontHeightMargin + kFontHeight / 2 - 2,
		mCaption);

	// draw border
	glVec2 points[6] = {
		{ mLeftx, my },
		{ mx, my },
		{ mx, my + mHeight },
		{ mx + mWidth, my + mHeight },
		{ mx + mWidth, my },
		{ mRightx, my }
	};

	SetColor(kBorderColor);
	glBegin(GL_LINE_STRIP);
	for (i = 0; i < 6; i++)
	  glVertex2i(points[i][0], points[i][1]);
	glEnd();

	// draw child
	if (mChild)
		mChild->Draw();
}


/* Alloc
 * -----
 * Allocates the window's internal storage (the mItems array) to the given
 * size. If size is smaller than the current allocated size, Resize throws an
 * assertion.
 */
void CWWindow::Alloc(int size)
{
	CWItem **arr;

	// sanity check
	assert(size > mAlloc);

	arr = new CWItem *[size];							// create new array
	assert(arr);
	memcpy(arr, mItems, mNumItems * sizeof(CWItem *));	// copy items over
	ARRAYDEL(mItems);									// delete old array

	mItems = arr;
	mAlloc = size;
}


/* PreAdd
 * ------
 * Called before an item is added. If the window is full, Alloc is called to
 * allocate more space.
 */
void CWWindow::PreAdd()
{
	// sanity check
	assert(mNumItems <= mAlloc);

	if (mNumItems == mAlloc)
		Alloc(mAlloc + 1);
}

/* PostAdd
 * -------
 * Called after an item is added. The window height (and, if necessary, the
 * window width) is increased so that the item fits.
 */
void CWWindow::PostAdd()
{
	CWItem *add = mItems[mNumItems - 1];

	if (add->mWidth > mWidth)
	{
		mWidth = add->mWidth;	// item is wider than window; increase width
		for (int i = 0; i < mNumItems; i++)	// increase all items' widths too
			mItems[i]->mWidth = mWidth;
	} else
		add->mWidth = mWidth;

	mHeight += add->mHeight;

	Reposition();
}


/* FindBorderCoords
 * ----------------
 * Finds the pixels where the window's caption text starts and stops, so that
 * the border lines can start at the right places.
 */
void CWWindow::FindBorderCoords()
{
	typedef float pfVec3[3];
	pfVec3		*before, *after;

	// set up before and after pixel buffers
	before = new pfVec3[mWidth];
	assert(before);
	after = new pfVec3[mWidth];
	assert(after);

	glReadPixels(mx, sScreenHeight - my, mWidth, 1, GL_RGB, GL_FLOAT, before);

	// draw caption
	DrawString(mx + kWndCaptionOffset,
		my + kFontHeightMargin + kFontHeight / 2 - 2,
		mCaption);

	glReadPixels(mx, sScreenHeight - my, mWidth, 1, GL_RGB, GL_FLOAT, after);

	// find right-side pixel to start border at (first pixel that is different)
	for (mRightx = mx + mWidth - 1; mRightx > mx; mRightx--)
	{
		if (memcmp(before[mRightx - mx], after[mRightx - mx], sizeof(pfVec3)) != 0)
			break;
	}
	assert(mRightx >= mx);

	// find left-side pixel to start border at (first pixel that is different)
	for (mLeftx = mx; mLeftx < mRightx; mLeftx++)
	{
		if (memcmp(before[mLeftx - mx], after[mLeftx - mx], sizeof(pfVec3)) != 0)
			break;
	}
	assert(mLeftx < mx + mWidth);
	if (mLeftx > mx)
		mLeftx--;


	ARRAYDEL(before);
	ARRAYDEL(after);
}


/* Reposition
 * ----------
 * Moves the window so that it is visible and not off the screen (if possible).
 */
void CWWindow::Reposition()
{
	// get x and y, reposition x
	if (mParent)	// child window, position relative to parent item
	{
		mx = mParent->mx - mWidth - kChildMargin;
		my = mParent->my;
		if (mx < 0)
			mx = mParent->mx + mParent->mWidth + kChildMargin;
		if (mx + mWidth > sScreenWidth)
			mx = sScreenWidth - mWidth;
	}
	else	// top-level window, position relative to mouse
	{
		if (mx + mWidth > sScreenWidth)
			mx -= mWidth;
		if (mx < 0)
			mx = 0;
	}

	// reposition y
	if (my + mHeight > sScreenHeight)
		my -= mHeight;
	if (my < kFontHeight)
		my = kFontHeight;	// for window caption

	// reposition items
	for (int i = 0; i < mNumItems; i++)
	{
		mItems[i]->mx = mx;
		mItems[i]->my = my + i * mItems[i]->mHeight + kWndMargin;
	}
}


/* AddToList
 * ---------
 * Adds the CWWindow to the linked list that stores all open CWindows. The list
 * is pointed to by the static member sList.
 */
void CWWindow::AddToList()
{
	// add to head of list
	this->next = sList;
	sList = this;
}


/* RemoveFromList
 * --------------
 * Removes the CWWindow from the linked list that stores all open CWindows.
 */
void CWWindow::RemoveFromList()
{
	CWWindow *cur;

	if (sList == this)	// found at head of list
	{
		sList = sList->next;
	}
	else
	{
		// find this console variable in the list
		for (cur = sList; cur && cur->next != this; cur = cur->next)
		{}
		//assert(cur, "could not find variable in list of console variables!");
		assert(cur);

		// relink
		cur->next = cur->next->next;
	}
}


/* WidgetUnderMouse (non-static, executes on a CWWindow!)
 * ----------------
 * This is a NON-STATIC method! It runs on a single CWWindow and returns the
 * widget under the mouse, or NULL if the mouse is not over a widget IN THIS
 * WINDOW.
 */
CWidget *CWWindow::WidgetUnderMouse()
{
	for (int i = 0; i < mNumItems; i++)
		if (mItems[i]->IsMouseOver())
			return mItems[i];

	return this;
}



