/* File: widget.cpp
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 *	CS248 Fall 2001
 *	HW3 - Video Game
 *
 *  Defines the CWScrolledWindow class, which is the scrollable window class
 *	for the Widget UI. For more information, see wscrolledwindow.h.
 *
 */


#include <GL/glut.h>
#include <assert.h>
#include "wscrolledwindow.h"
#include "witem.h"


// ----------------------------------------------------------------------------
// public methods
// ----------------------------------------------------------------------------

// ctor
CWScrolledWindow::CWScrolledWindow(const string &caption,
								   int x, int y,
								   int width, int height)
	: CWWindow(caption, x, y),
	  mHoriz(CWScrollbar::eHorizontal),
	  mVert(CWScrollbar::eVertical)
{
	mHoriz.mParent = this;
	mVert.mParent = this;

	if (mWidth < width)
	  mWidth = width;
	if (mHeight < height)
	  mHeight = height;
	mTotalWidth = mWidth;
	mTotalHeight = mHeight;

	Reposition();
	PositionScrollbars();
}

// dtor
CWScrolledWindow::~CWScrolledWindow()
{
	// mHoriz and mVert scrollbars are destroyed automatically, since they're
	// not dynamically allocated members
}


// AddItem
CWItem *CWScrolledWindow::AddItem(const string & caption,
	CWItem::ItemFn callback, void *userData)	// = NULL
{
	int oldWidth, oldHeight;
	CWItem *add;

	// preserve dimensions
	oldWidth = mWidth;
	oldHeight = mHeight;

	add = CWWindow::AddItem(caption, callback, userData);

	mWidth = oldWidth;
	mHeight = oldHeight;
	Reposition();

	return add;
}


// AddCheckbox
CWCheckbox *CWScrolledWindow::AddCheckbox(const string & caption,
	CWItem::ItemFn callback, bool checked, void *userData)	// = NULL
{
	int oldWidth, oldHeight;
	CWCheckbox *add;

	// preserve dimensions
	oldWidth = mWidth;
	oldHeight = mHeight;

	add = CWWindow::AddCheckbox(caption, callback, checked, userData);

	mWidth = oldWidth;
	mHeight = oldHeight;
	Reposition();

	return add;
}


// Draw
//
// NOTE: assumes that the OpenGL scissor box is disabled when this method is
// called. If it is enabled, Draw throws an assertion.
void CWScrolledWindow::Draw()
{
	RepositionItems();

	CWWindow::Draw();	// draw window and items

	// draw scrollbars
	mHoriz.Draw();
	mVert.Draw();
}


// IsMouseOver
bool CWScrolledWindow::IsMouseOver() const
{
	int x, y;

	assert(sMouseCallback);
	sMouseCallback(x, y);

	if (mHoriz.IsMouseOver() || mVert.IsMouseOver())
		return true;

	return (x >= mx && x <= mx + mWidth && y >= my && y < my + mHeight);
}


// ----------------------------------------------------------------------------
// protected methods
// ----------------------------------------------------------------------------

// ctor (used if this is a child window)
CWScrolledWindow::CWScrolledWindow(CWItem *parent, int width, int height)
	: CWWindow(parent),
	  mHoriz(CWScrollbar::eHorizontal),
	  mVert(CWScrollbar::eVertical)
{
	mHoriz.mParent = this;
	mVert.mParent = this;

	if (mWidth < width)
	  mWidth = width;
	if (mHeight < height)
	  mHeight = height;
	mTotalWidth = mWidth;
	mTotalHeight = mHeight;

	Reposition();
	PositionScrollbars();
}


/* PostAdd
 * -------
 * Overridden from CWWindow. This is called after an item is added. It is
 * overridden because unlike CWWindow, items' widths and heights should not
 * be modified when they are added to a CWScrolledWindow.
 */
void CWScrolledWindow::PostAdd()
{
	CWItem *add = mItems[mNumItems - 1];
	int sum;

	// position new item
	add->mx = mx;
	add->my = my + (mNumItems - 1) * add->mHeight + kWndMargin;

	// increase total width if necessary
	if (add->mWidth > mTotalWidth)
	  mTotalWidth = add->mWidth;

	// increase total height if necessary
	sum = kFontHeight / 2 + kFontHeightMargin;
	for (int i = 0; i < mNumItems; i++)
		sum += mItems[i]->mHeight;
	if (sum > mTotalHeight)
	  mTotalHeight =  sum;
}


/* RepositionItems
 * ---------------
 * Repositions the window's items based on the values of the horizontal and
 * vertical scrollbars.
 */
void CWScrolledWindow::RepositionItems()
{
	for (int i = 0; i < mNumItems; i++)
	{
		mItems[i]->mx = mx - static_cast<int>(mHoriz.mVal * (mTotalWidth - mWidth));
		mItems[i]->my = my + i * mItems[i]->mHeight + kWndMargin;
  		mItems[i]->my -= static_cast<int>(mVert.mVal * (mTotalHeight - mHeight));
	}
}


/* PositionScrollbars
 * ------------------
 * Sets the scrollbars' positions and dimensions based on the position and
 * dimension of the window.
 */
void CWScrolledWindow::PositionScrollbars()
{
	mHoriz.mx = mx;
	mHoriz.my = my + mHeight + kChildMargin;
	mHoriz.mWidth = mWidth;
	mHoriz.mHeight = CWScrollbar::kScrollbarWidth;

	mVert.mx = mx + mWidth + kChildMargin;
	mVert.my = my;
	mVert.mWidth = CWScrollbar::kScrollbarWidth;
	mVert.mHeight = mHeight;
}


/* WidgetUnderMouse
 * ----------------
 * Overridden from CWWindow. Returns the widget in this window under the mouse
 * cursor, or NULL if the mouse is not over a widget in the window. This is
 * overridden so that it can return the scrollbars if the mouse is over one of
 * them.
 */
CWidget *CWScrolledWindow::WidgetUnderMouse()
{
	// check scrollbars
	if (mHoriz.IsMouseOver())
		return &mHoriz;
	if (mVert.IsMouseOver())
		return &mVert;

	// mouse was not over the scrollbars, fall back to CWWindow to check items
	return CWWindow::WidgetUnderMouse();
}


