/* File: wscrollbar.cpp
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 *	CS248 Fall 2001
 *	HW3 - Video Game
 *
 *  Defines the CWScrollbar class, which is the scrollbar class for the Widget
 *	UI. For more information, see wscrollbar.h.
 *
 */

#include <GL/glut.h>
#include <assert.h>
#include "wscrollbar.h"



// ----------------------------------------------------------------------------
// initialize static members
// ----------------------------------------------------------------------------
const int	CWScrollbar::kScrollbarWidth	= 10;
const int	CWScrollbar::kThumbLen			= 40;
const float	CWScrollbar::kThumbJump			= 0.2f;
CWScrollbar	*CWScrollbar::sDragging			= NULL;


// ----------------------------------------------------------------------------
// public methods
// ----------------------------------------------------------------------------

// dtor
CWScrollbar::~CWScrollbar()
{
	// no dynamic allocation, nothing to delete
}


// ----------------------------------------------------------------------------
// protected methods
// ----------------------------------------------------------------------------

CWScrollbar::CWScrollbar(EDirection dir) : CWidget("", 0, 0)
{
	// sanity checks:
	assert(dir == eHorizontal || dir == eVertical);

	mDir = dir;
	mVal = 0;

}

/* Draw
 * ----
 * Overridden from CWidget. This is very different from a generic widget's draw
 * function. The scrollbar is drawn, and then the thumb is drawn on top of it
 * since the thumb is stored as a texture.
 */
void CWScrollbar::Draw()
{
	bool horiz = (mDir == eHorizontal);
	int tx, ty;

	if (sDragging == this)
		DragThumb();

	tx = ThumbX();
	ty = ThumbY();


	// draw thumb
	int len = kThumbLen / 2;
	glColor col, white = { 255, 255, 255, 196 };
	glVec2 pt1 = { tx, ty },
			pt2 = { tx + kScrollbarWidth - 1, ty + kScrollbarWidth - 1 },
			pt3 = { tx, ty },
			pt4 = { tx + kScrollbarWidth - 1, ty + kScrollbarWidth - 1 };

	for (int i = 0; i < len; i++)
	{
		float f = static_cast<float>(i) / len;

		// get color (lerp between white and background color)
		for (int j = 0; j < 4; j++)
			col[j] = static_cast<unsigned char>
			  ((1 - f) * white[j] + f * kDarkBackColor[j]);
		SetColor(col);

		// draw lines
		glBegin(GL_LINES);

		if (horiz)
		{
			pt1[0] = pt2[0] = tx + len - i - 1;
			pt3[0] = pt4[0] = tx + len + i;
			glVertex2i(pt1[0], pt1[1]);
			glVertex2i(pt2[0], pt2[1]);
			glVertex2i(pt3[0], pt3[1]);
			glVertex2i(pt4[0], pt4[1]);
		}
		else
		{
			pt1[1] = pt2[1] = ty + len - i - 1;
			pt3[1] = pt4[1] = ty + len + i;
			glVertex2i(pt1[0], pt1[1]);
			glVertex2i(pt2[0], pt2[1]);
			glVertex2i(pt3[0], pt3[1]);
			glVertex2i(pt4[0], pt4[1]);
		}

		glEnd();
	}


	// draw background (except where thumb is)

	SetColor(kDarkBackColor);

	// area before thumb
	int x2 = static_cast<int>(pt2[0]), y2 = static_cast<int>(pt2[1]),
		x3 = static_cast<int>(pt3[0]), y3 = static_cast<int>(pt3[1]);

	DrawRect(mx, my, x2 - mx, y2 - my, true);

	// area after thumb
	DrawRect(x3, y3, mx + mWidth - x3, my + mHeight - y3, true);


	// draw border
	SetColor(kBorderColor);
	DrawRect(mx, my, mWidth, mHeight, false);
}


/* Select
 * ------
 * Needed so that the user can jump-move the thumb.
 */
void CWScrollbar::Select()
{
	int x, y, tx, ty;

	sMouseCallback(x, y);
	tx = ThumbX();
	ty = ThumbY();

	// do nothing if they clicked directly on the thumb
	if (x >= tx && x <= tx + kThumbLen && y >= ty && y <= ty + kThumbLen)
		return;

	// jump
	if ((mDir == eHorizontal && x > tx) || (mDir == eVertical && y > ty))
		mVal += kThumbJump;
	else
		mVal -= kThumbJump;

	mVal = CLAMP(mVal, 0, 1);	// clamp to [0, 1]
}

/* MouseDown, MouseUp
 * ---------
 * These are needed so that the user can drag the thumb.
 *
 * NOTE: These are overriden from CWidget. Don't do this at home, for trained
 * professionals only. :P
 */
void CWScrollbar::MouseDown()
{
	int mousex, mousey, tx, ty;

	CWidget::MouseDown();

	tx = ThumbX();
	ty = ThumbY();
	sMouseCallback(mousex, mousey);

	// check for dragging
	if (mousex < tx || mousey < ty)
		return;
	if ((mDir == eHorizontal && mousex <= tx + kThumbLen && mousey <= ty + kScrollbarWidth) ||
		(mDir == eVertical && mousey <= ty + kThumbLen && mousex <= tx + kScrollbarWidth))
	{
		assert(!sDragging);
		sDragging = this;
	}
}


/* DragThumb
 * ---------
 * Moves the thumb with the mouse pointer if it is being dragged.
 */
void CWScrollbar::DragThumb()
{
	int mousex, mousey;
	int tx, ty;

	assert(sDragging == this);

	// get mouse and thumb coords
	sMouseCallback(mousex, mousey);
	mousex = CLAMP(mousex, mx, mx + mWidth);
	mousey = CLAMP(mousey, my, my + mHeight);
	tx = ThumbX();
	ty = ThumbY();

	// should the thumb move?
	if (mDir == eHorizontal)
	{
		if (mousex >= tx && mousex <= tx + kThumbLen)
			return;

		// move the thumb
		if (mousex > tx + kThumbLen)
			mousex -= kThumbLen;
		mVal = static_cast<float>(mousex - mx) / (mWidth - kThumbLen);
	}
	else	// mDir == eVertical
	{
		if (mousey >= ty && mousey <= ty + kThumbLen)
			return;

		// move the thumb
		if (mousey > ty + kThumbLen)
			mousey -= kThumbLen;
		mVal = static_cast<float>(mousey - my) / (mHeight - kThumbLen);
	}

}


/* ThumbX, ThumbY
 * -------
 * Return the pixel x and/or y coordinate of the thumb (upper left corner),
 * calculated from mVal.
 */
inline int CWScrollbar::ThumbX()
{
	int offset = 0;

	if (mDir == eHorizontal)
	  offset = static_cast<int>(mVal * (mWidth - kThumbLen - 1));

	return mx + offset + 1;
}
inline int CWScrollbar::ThumbY()
{
	int offset = 0;

	if (mDir == eVertical)
	  offset = static_cast<int>(mVal * (mHeight - kThumbLen)) + 1;

	return my + offset + 1;
}


