/* File: widget.h
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 *	CS248 Fall 2001
 *	HW3 - Video Game
 *
 *	Base class for the Widget UI. CWidget holds basic information about a UI
 *	control such as position, size, and name. It also provides basic hooks
 *	for drawing and mouse events. These are usually overridden by classes that
 *	derive from CWidget.
 *
 *
 *	Here is the class hierarchy:
 *
 *
 *							CWidget
 *						  /	  |		-
 *						/	  |		  -
 *				CWWindow	CWItem		CWScrollBar
 *				  /			 /	-
 *				/			/	  -
 *	CWScrolledWindow  CWTextBox	CWCheckBox
 *
 *
 *	NOTE: CWTextbox has been suspended since the game doesn't need it.
 *
 *
 *	AVAILABLE WIDGETS:
 *
 *		CWWindow
 *	A window that can contain items and checkboxes. A CWWindow can also
 *	have sub-windows, or children, which work like nested menus in Windows
 *	applications. A CWWindow automatically resizes to accomodate every item
 *	it contains.
 *
 *		CWScrolledWindow
 *	Does everything that a CWWindow does and it is scrollable. Unlike
 *	CWWindows, these have a fixed width and height that is set at the time
 *	the object is created.
 *
 *		CWItem
 *	A selectable item in a window. Similar to a standard menu item in
 *	Windows. When an item is selected, it calls a user-defined callback
 *	function and passes a pointer to itself as a parameter. An item can
 *	also contain a pointer to user data.
 *
 *		CWCheckbox
 *	A checkbox does everything an item does and also can be selected or
 *	not selected. At runtime, this can be determined by an accessor.
 *
 *
 *	QUICK-START:
 *
 *	To create and use a window in your code, follow these steps:
 *
 *	1) Write a callback function that will be called when an item in your
 *	window is selected. (If you don't want your item to do anything when it
 *	is selected, you can skip this step.) Ex:
 *
 *	void *ItemCallback(CWItem *selected) {
 *		printf("Item %s was selected!", selected->GetCaption());
 *	}
 *
 *
 *	2) Create an instance of CWWindow, add items to it, and keep it around
 *	as long as you want to use it. (If you didn't write a callback function in
 *	step 1, you can pass NULL for the callback.) Ex:
 *
 *		CWWindow *wnd = new CWWindow("hello world", 100, 100);
 *		wnd->AddItem("item1", ItemCallback, NULL);
 *
 *
 *	3) Call CWidget::DrawAll in your main draw routine. Ex:
 *
 *	void DrawAll() {
 *		...
 *		CWidget::DrawAll();
 *	}
 *
 *
 *	4) When you want to close the window, simply delete it. Ex:
 *
 *		if (doneWithWnd)
 *			delete wnd;
 *
 *	Note that the widget system does not maintain any information about windows
 *	after they are deleted, so to show the window again, it must be created
 *	from scratch and the items must be added again.
 *
 */


#ifndef _WIDGET_H
#define _WIDGET_H


#include <string>
using namespace std;


// -----------------------------------------------------------------------------
// MACROS
// -----------------------------------------------------------------------------
// max and min
#ifndef MAX
#	define MAX(x, y)		(((x) >= (y)) ? (x) : (y))
#endif
#ifndef MIN
#	define MIN(x, y)		(((x) <= (y)) ? (x) : (y))
#endif

// clamp
#define CLAMP(x, min, max) (MIN( MAX(x, min), max))

// smarter delete
#define DEL(ptr)		{ if (ptr) delete (ptr); (ptr) = NULL; }
// smarter delete []
#define ARRAYDEL(ptr)	{ if (ptr) delete [] (ptr); (ptr) = NULL; }

// -----------------------------------------------------------------------------
// TYPES
// -----------------------------------------------------------------------------
typedef unsigned int	uint;
typedef unsigned char	uchar;
typedef int				glVec2[2];
typedef uchar			glColor[4];	// glColor - OpenGL color type (like pfVec4)


// -----------------------------------------------------------------------------
// CWIDGET CLASS
// -----------------------------------------------------------------------------
class CWidget
{
public:
	// exposed methods
	virtual ~CWidget();		// dtor

	/* Draw
	 * ----
	 * Draws the widget at its current location.
	 *
	 * Pure virtual, so any derived classes *must* implement this if they are
	 * instantiated. 
	 */
	virtual void Draw() = 0;

	/* Select
	 * ------
	 * Called when the widget is clicked on or otherwise selected.
	 *
	 * Pure virtual, so any derived classes *must* implement this if they are
	 * instantiated. 
	 */
	virtual void Select() = 0;

	/* IsMouseOver
	 * -----------
	 * Returns true if the mouse is over the widget, false otherwise.
	 *
	 * NOTE: This shouldn't be overridden unless you have a *really* good
	 * reason. Trust me, you'll be a lot happier if you let it do its job.
	 */
	virtual bool IsMouseOver() const;

	/* MouseDown, MouseUp
	 * ---------
	 * Called when a mouse button is pressed or released and the mouse is over
	 * the widget.
	 *
	 * NOTE: These shouldn't be overridden unless you have a *really* good
	 * reason. Same caveat as above.
	 */
	virtual void MouseDown();
	virtual void MouseUp();


	/* DrawAll
	 * -------
	 * Draws all open windows and the mouse. Should probably be called in your
	 * main drawing routine.
	 */
	static void DrawAll();

	/* SendMouseDown, SendMouseUp
	 * -------------
	 * The external mouse code should call these when a mouse button is pressed
	 * or released. These are static so that the MouseDown and MouseUp messages
	 * can be routed to the widget under the mouse cursor.
	 *
	 * They return true if a widget received the message (the mouse was over a
	 * widget), false otherwise.
	 */
	static bool SendMouseDown();
	static bool SendMouseUp();

	// static members
	static int		sScreenWidth;
	static int		sScreenHeight;

	// mouse callback - a function the debug UI can call to get the current
	// mouse coordinates
	typedef void	(*MouseFn)(int &x, int &y);
	static MouseFn	sMouseCallback;

	// mouse alpha value (used with mouse fade)
	static unsigned char sMouseAlpha;

protected:
	// members
	string			mCaption;
	int				mx, my, mWidth, mHeight;

	// ctors
	CWidget(const string &caption) : mCaption(caption) {}
	CWidget(const string &caption, int x, int y) :
		mCaption(caption), mx(x), my(y) {}



	// static members
	// true if OpenGL is ready to draw 2D stuff, false otherwise
	static bool		m2dgl;	

	// if the mouse button is down, this is the widget that the mouse was over
	// when the button was pressed. otherwise, this is NULL.
	static CWidget	*pressed;
	static uint		sMouseTimer;

	// fill and line colors
	static glColor	sFillColor;
	static glColor	sLineColor;


	// static helper methods
	static void		Setup2DGL();
	static void		Release2DGL();
	static CWidget	*FindWidgetUnderMouse();
	static void		DrawMouse();
	static void		SetFillColor(const glColor col);
	static void		SetLineColor(const glColor col);
	static void		SetColor(const glColor col);
	static void		DrawRect(int x, int y, int width, int height, bool filled);
	static void		DrawString(int x, int y, const string &str);
	static int		CharWidth(char ch);
	static int		StringWidth(const string &str);

	// static constants
	static const int		kFontHeightMargin;
	static const int		kFontHeight;
	static const int		kAvgFontWidth;
	static const int		kWndCaptionOffset;
	static const int		kItemMargin;
	static const int		kWndMargin;
	static const int		kChildMargin;

	static const glColor	kSelectColor;
	static const glColor	kBorderColor;
	static const glColor	kBackColor;
	static const glColor	kLightBackColor;
	static const glColor	kMidBackColor;
	static const glColor	kDarkBackColor;
	static const glColor	kObscureColor;

	static const uint		kMouseHideTime;	// time until mouse hides
	static const uint		kMouseFadeTime;	// time that mouse fade takes
	static const glColor	kMouseFillColor;
	static const glColor	kMouseLineColor;
	static const int		kNumMouseCoords;
	static const glVec2		kMouseCoords[];

	static const int		fontWidths[];

private:
	// disallow the empty constructor
	CWidget();
};


#endif	// _WIDGET_H
