/* File: widget.cpp
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 *	CS248 Fall 2001
 *	HW3 - Video Game
 *
 *	Defines the CWidget class, which is the base class for the Widget UI. For
 *	more information, see widget.h.
 *
 *	TODO:
 *	- change CWScrolledWindow so that it composites CWWindow instead of
 *	derives from it
 *
 */

#include <GL/glut.h>
#include <assert.h>
#include <stdio.h>		// for memcpy
#include "widget.h"
#include "witem.h"
#include "wwindow.h"	// for drawing windows
#include "wscrollbar.h"	// for turning off scrollbar dragging
//#include "timer.h"		// for pointer hiding


// ----------------------------------------------------------------------------
// INITIALIZE STATIC MEMBERS
// ----------------------------------------------------------------------------
// constants
int					CWidget::sScreenWidth		= 0;
int					CWidget::sScreenHeight		= 0;
CWidget::MouseFn	CWidget::sMouseCallback		= NULL;
const int			CWidget::kFontHeightMargin	= 4;
const int			CWidget::kFontHeight		= 14;
const int			CWidget::kAvgFontWidth		= 10;
const int			CWidget::kWndCaptionOffset	= 20;
const int			CWidget::kItemMargin		= 4;
const int			CWidget::kWndMargin			= 10;
const int			CWidget::kChildMargin		= 4;
const glColor		CWidget::kSelectColor		= { 0, 255, 255,  64 };
const glColor		CWidget::kBorderColor		= { 0, 255, 255, 255 };
const glColor		CWidget::kBackColor			= { 0, 128, 128,  64 };
const glColor		CWidget::kLightBackColor	= { 0, 128, 128, 100 };
const glColor		CWidget::kMidBackColor		= { 0, 128, 128,  50 };
const glColor		CWidget::kDarkBackColor		= { 0, 128, 128, 196 };
const glColor		CWidget::kObscureColor		= { 0,  50,  50, 128 };

const uint			CWidget::kMouseHideTime		= 10 * 1000;	// 10 seconds
const uint			CWidget::kMouseFadeTime		= 1 * 1000;		// 1 second
const glColor		CWidget::kMouseFillColor	= {	0, 128, 128, 128 };
const glColor		CWidget::kMouseLineColor	= { 0, 255, 255, 255 };
const int			CWidget::kNumMouseCoords	= 3;
const glVec2		CWidget::kMouseCoords[kNumMouseCoords] =
{	{  0,  0 },	// mouse pointer
	{  0, 20 },
	{ 10, 15 }
};

// static members
CWidget				*CWidget::pressed		= NULL;
uint				CWidget::sMouseTimer	= 0;
unsigned char		CWidget::sMouseAlpha	= 255;
glColor				CWidget::sFillColor		= { 200, 200, 200, 255 };
glColor				CWidget::sLineColor		= { 255, 255, 255, 255 };
bool				CWidget::m2dgl			= false;




// ----------------------------------------------------------------------------
// PUBLIC METHODS
// ----------------------------------------------------------------------------

// dtor
CWidget::~CWidget()
{}



// IsMouseOver
bool CWidget::IsMouseOver() const
{
	int x, y;

	assert(sMouseCallback);
	sMouseCallback(x, y);

	return (x >= mx && x <= mx + mWidth && y >= my && y < my + mHeight);
}


// MouseDown, MouseUp
void CWidget::MouseDown()
{
	assert(!pressed);

	pressed = this;
}

void CWidget::MouseUp()
{
	if (pressed == this)	// only select if the mouse button was both pressed
		Select();			// and released on this widget

	pressed = NULL;
}


// DrawAll (static)
void CWidget::DrawAll()
{
  bool shouldRelease = !m2dgl;

  if (!m2dgl)
	Setup2DGL();
	
  // draw windows
  for (CWWindow *cur = CWWindow::sList; cur != NULL; cur = cur->next)
	cur->Draw();

  // draw mouse
  DrawMouse();

  if (shouldRelease)
	Release2DGL();
}


// SendMouseDown, SendMouseUp (static!)
bool CWidget::SendMouseDown()
{
	CWidget *widget;

	widget = FindWidgetUnderMouse();
	if (widget)
	{
		widget->MouseDown();
		return true;
	} else
		return false;
}

bool CWidget::SendMouseUp()
{
	CWidget *widget;

	// turn off scrollbar dragging
	CWScrollbar::sDragging = NULL;

	widget = FindWidgetUnderMouse();
	if (widget)
	{
		widget->MouseUp();
		return true;
	}
	else
	{
		CWidget::pressed = NULL;
		return false;
	}
}




// ----------------------------------------------------------------------------
// PROTECTED HELPER METHODS
// ----------------------------------------------------------------------------

/* SetFillColor, SetLineColor
 * --------------------------
 * Set the current fill or line color. For use with DrawRect.
 */
void CWidget::SetFillColor(const glColor col)
{
	memcpy(sFillColor, col, sizeof(glColor));
}

void CWidget::SetLineColor(const glColor col)
{
	memcpy(sLineColor, col, sizeof(glColor));
}

/* SetColor
 * --------
 * Sets the immediate color to be used.
 */
void CWidget::SetColor(const glColor col)
{
	SetFillColor(col);
	SetLineColor(col);
	glColor4ubv(col);
}


/* Setup2DGL
 * ---------
 * Sets up OpenGL to do 2D graphics (orthographic projection, no world or view
 * transformations, etc.).
 */
void CWidget::Setup2DGL()
{
	// sanity check
	assert(!m2dgl);

	m2dgl = true;

	// set up matrix stacks
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glOrtho(0, (float)sScreenWidth, 0, (float)sScreenHeight, 0, 1000);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	// flip it so that we're in upper-left=0,0 coords
	glScalef(1.0f, -1.0f, 1.0f);
	glTranslatef(0.0f, -(float)sScreenHeight, 0.0f);

	// set up rendering settings
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE);
	glDisable(GL_LIGHTING);
	glDisable(GL_TEXTURE_2D);
	glEnable(GL_BLEND);

	glLineWidth(1.0);
}


/* Release2DGL
 * -----------
 * Pops the changes we made to the matrix stack and other stuff.
 */
void CWidget::Release2DGL()
{
	// sanity check
	assert(m2dgl);

	m2dgl = false;

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
}


/* FindWidgetUnderMouse (static!)
 * --------------------
 * This is a STATIC method! It iterates over ALL of the windows currently open
 * and returns the widget currently under the mouse pointer, or NULL if the
 * mouse is not over a widget.
 *
 * If a window has children, its children are checked for a widget under the
 * mouse before its own items are.
 */
CWidget *CWidget::FindWidgetUnderMouse()
{
	CWWindow *cur;

	for (cur = CWWindow::sList; cur != NULL; cur = cur->next)
	{
		// find last child
		for ( ; cur->mChild != NULL; cur = cur->mChild)
		{}

		// iterate over all windows in the family, from bottom to top
		while (true)
		{
			if (cur->IsMouseOver())
				return cur->WidgetUnderMouse();
			if (!cur->mParent)
				break;
			cur = cur->mParent->mParent;
		}
	}

	return NULL;	// no window under mouse
}


/* DrawMouse
 * ---------
 * Draws the mouse pointer and manages the mouse hide/fade timer.
 */
void CWidget::DrawMouse()
{
	int i, mousex, mousey;
	static int lastmousex = 0, lastmousey = 0;
	glVec2 pointer[kNumMouseCoords];
	//static uint lastTime = GetMillisecondTimer();
	//static uint lastTime = 0;
	uint deltaTime = 0;

	sMouseCallback(mousex, mousey);

	// get time since last Manage call
	//deltaTime = GetMillisecondTimer() - lastTime;
	//lastTime += deltaTime;

	// manage mouse timer
	sMouseTimer += deltaTime;
	if (lastmousex != mousex || lastmousey != mousey)
		sMouseTimer = 0;
	lastmousex = mousex;
	lastmousey = mousey;

	if (sMouseTimer > kMouseHideTime + kMouseFadeTime)
		sMouseAlpha = 0;
	else if (sMouseTimer > kMouseHideTime)
		sMouseAlpha = static_cast<unsigned char>(
			255 * (kMouseHideTime + kMouseFadeTime - sMouseTimer) / kMouseFadeTime);
	else
		sMouseAlpha = 255;

	// set mouse colors
	glColor mousefill, mouseline;
	memcpy(mousefill, kMouseFillColor, sizeof(glColor));
	memcpy(mouseline, kMouseLineColor, sizeof(glColor));
	mousefill[3] = mouseline[3] = sMouseAlpha;
	SetFillColor(mousefill);
	SetLineColor(mouseline);

	// setup mouse pointer coordinates
	for (i = 0; i < kNumMouseCoords; i++)
	{
		pointer[i][0] = kMouseCoords[i][0] + mousex;
		pointer[i][1] = kMouseCoords[i][1] + mousey;
	}

	// draw mouse pointer
	glEnable(GL_BLEND);

	// draw inside
	glColor4ubv(sFillColor);
	glBegin(GL_POLYGON);
	for (i = 0; i < kNumMouseCoords; i++)
	  glVertex2i(pointer[i][0], pointer[i][1]);
	glEnd();

	// draw inside
	glColor4ubv(sLineColor);
	glBegin(GL_LINE_STRIP);
	for (i = 0; i < kNumMouseCoords; i++)
	  glVertex2i(pointer[i][0], pointer[i][1]);
	glVertex2i(pointer[0][0], pointer[0][1]);
	glEnd();
}


/* Rect
 * ----
 * Draws a rectangle in screen coordinates. If filled is true, the rectangle
 * is filled; otherwise it is outlined.
 *
 * NOTE: turns on alpha blending!
 */
void CWidget::DrawRect(int x, int y, int width, int height, bool filled)
{
	assert(m2dgl);

	glEnable(GL_BLEND);

	if (filled) {
	  glColor4ubv(sFillColor);

	  glBegin(GL_QUADS);
	  glVertex2i(x, y);
	  glVertex2i(x + width, y);
	  glVertex2i(x + width, y + height);
	  glVertex2i(x, y + height);
	  glEnd();
	}
	else {		// outline
	  glColor4ubv(sLineColor);

	  glBegin(GL_LINE_STRIP);
	  glVertex2i(x, y);
	  glVertex2i(x + width, y);
	  glVertex2i(x + width, y + height);
	  glVertex2i(x, y + height);
	  glVertex2i(x, y);
	  glEnd();
	}
}


void CWidget::DrawString(int x, int y, const string &str)
{
  assert(m2dgl);

  glColor4ub(255, 255, 255, 255);

  // move to (x, y)
  glRasterPos2i(x, y - 4);

  // draw string
  for (unsigned int i = 0; i < str.length(); i++)
	glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18, str[i]);
	//glutStrokeCharacter(GLUT_STROKE_ROMAN, str[i]);
}


int CWidget::StringWidth(const string &str)
{
	uint len;
	int width = 0;

	len = str.length();
	for (uint i = 0; i < len; i++)
		width += CharWidth(str[i]);

	return width;
}


inline int CWidget::CharWidth(char ch)
{
  return glutBitmapWidth(GLUT_BITMAP_HELVETICA_18, ch);
  //return glutStrokeWidth(GLUT_STROKE_ROMAN, ch);
}



