/* File: water.h
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 * CS248 Fall 2001
 * HW3 - Video Game
 *
 * Declares the CWater class. CWater is the class that implements the water's
 * surface as a height field. It does lots of the meat of this program, including
 * the wave/ripple generation and the refraction calculations (as well as other
 * cool rendering effects).
 *
 * Its methods are:
 *  void		Draw();
 *  void		Update(int msecs);
 *  void		Touch(CVector3& pos);
 */

#ifndef _WATER_H
#define _WATER_H

#include "vector3.h"


// -----------------------------------------------------------------------------
// FORWARD DECLARATIONS
// -----------------------------------------------------------------------------


// -----------------------------------------------------------------------------
// CWATER CLASS
// -----------------------------------------------------------------------------
class CWater
{
public:
  // public constants
  static float kHeight;

  // ctor
  CWater();

  // dtor
  ~CWater();

  /* Draw
   * ----
   * Draws the water surface.
   */
  void Draw() const ;

  /* Update
   * ------
   * Updates the position, behavior, etc. of the entire water surface after a
   * given delta time.
   */
  void Update(int msecs);

  /* Touch
   * -----
   * Touches the water surface at the given point. If small is true, a small
   * ripple is generated; otherwise, a large ripple is generated.
   */
  void Touch(float x, float y, bool small);

  /* GetHeight
   * ---------
   * Returns the height of the water surface at the given point.
   */
  inline float GetHeight(float x, float y) const;

  /* GetNormal
   * ---------
   * Returns the normal vector at the given point.
   */
  inline const CVector3& GetNormal(float x, float y) const;

protected:
  // state type for drawing mesh
  enum RenderType { kRTexture, kREnvMap, kRRefract, kRWireframe, kRNone };

  // constants
  static int	kFineVerts;
  static float	kFineSquare;
  static float	kWaveXPhase;
  static float	kWaveYPhase;
  static int	kSmallRippleSize;
  static int	kLargeRippleSize;

  // helper methods
  void InitVertexArrays() const;
  void InitHeightField();
  void UpdateWaves(int msecs);
  void UpdateRipples(int msecs);
  void PrecomputeRipples();
  void DrawMesh(RenderType render) const;
  void DrawNormals() const;
  void GenerateNormals();
  void Texture() const;
  void EnvMap() const;
  void RefractCoords(int row, int col) const;
  void FresnelTerm(int row, int col) const;

  inline float	ColToX(int col) const;
  inline float	RowToY(int row) const;
  inline int	XToCol(float x) const;
  inline int	YToRow(float y) const;


  // members
  int cur, prev;
  int gridSize;
  float squareSize;

  float **heights[2];		// parallel arrays so that i can
  CVector3 **normals;		// flip btw the heights arrays
  CVector3 **faces[2];		// face normals
  float **smallRipple;		// precomputed ripple heightfields
  float **largeRipple;
};




// -----------------------------------------------------------------------------
// INLINE METHODS
// -----------------------------------------------------------------------------

inline float CWater::GetHeight(float x, float y) const
{
  return heights[cur][YToRow(y)][XToCol(x)];
}

inline const CVector3& CWater::GetNormal(float x, float y) const
{
  return normals[YToRow(y)][XToCol(x)];
}


/* ColToX, RowToY, XToCol, YToRow
 * --------------
 * Convert between row/column and x/y coordinates in world space.
 */
inline float CWater::ColToX(int col) const
{
  return (col - gridSize / 2) * squareSize;
}
inline float CWater::RowToY(int row) const
{
  return (row - gridSize / 2) * squareSize;
}

inline int CWater::XToCol(float x) const
{
  return static_cast<int>(x / squareSize + (gridSize / 2));
}
inline int CWater::YToRow(float y) const
{
  return static_cast<int>(y / squareSize + (gridSize / 2));
}

#endif	// _WATER_H
