/* File: water.cpp
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 * CS248 Fall 2001
 * HW3 - Video Game
 *
 * Defines the CWater class. For more information, see water.h.
 */

#include <GL/glut.h>
#include <assert.h>
#include <string>
#include <math.h>
#include "settings.h"
#include "vector3.h"
#include "water.h"
#include "texture.h"


#ifndef M_PI
#define M_PI 3.14159265	// because vc++'s math.h doesn't define M_PI
#endif


// *** LOOK AT THIS!!! ***
// I'm including ripple.cpp, refract.cpp, and fresnel.cpp. I really wanted to
// separate them out, both literally and conceptually, from the rest of the
// drawing code. Also, water.cpp was getting big, but that wasn't as big a deal.
//
// ripple.cpp defines RippleFilter and MakeRipple.
// refract.cpp defines CWater::RefractCoords.
// fresnel.cpp defines CWater::FresnelTerm.
//
// *** LOOK AT THIS!!! ***
#include "ripple.cpp"
#include "refract.cpp"
#include "fresnel.cpp"


// -----------------------------------------------------------------------------
// GLOBALS
// -----------------------------------------------------------------------------
extern bool		gDrawNormals;
extern bool		gWaves;
extern bool		gTexture;
extern bool		gEnvMap;
extern bool		gRefract;
extern bool		gFresnel;
extern bool		gTransparent;
extern bool		gWireframe;



// -----------------------------------------------------------------------------
// INITIALIZE STATIC MEMBERS
// -----------------------------------------------------------------------------
// public
float CWater::kHeight		= 0; //CSettings::GetFloat("Water", "Height");

// private
int		CWater::kFineVerts	= 0; //CSettings::GetInt("Water", "FineVerts");
float	CWater::kFineSquare	= 0; //CSettings::GetFloat("Water", "FineSquare");
float	CWater::kWaveXPhase	= 0; //CSettings::GetFloat("Water", "WaveXPhase");
float	CWater::kWaveYPhase	= 0; //CSettings::GetFloat("Water", "WaveYPhase");
int	CWater::kSmallRippleSize = 0; //CSettings::GetInt("Water", "SmallRippleSize");
int	CWater::kLargeRippleSize = 0; //CSettings::GetInt("Water", "LargeRippleSize");


// -----------------------------------------------------------------------------
// PUBLIC METHODS
// -----------------------------------------------------------------------------

CWater::CWater()
{
  // init "constants" (damn vc++ can't do init order right)
  kHeight		= CSettings::GetFloat("Water", "Height");

  // private
  kFineVerts	= CSettings::GetInt("Water", "FineVerts");
  kFineSquare	= CSettings::GetFloat("Water", "FineSquare");
  kWaveXPhase	= CSettings::GetFloat("Water", "WaveXPhase");
  kWaveYPhase	= CSettings::GetFloat("Water", "WaveYPhase");
  kSmallRippleSize = CSettings::GetInt("Water", "SmallRippleSize");
  kLargeRippleSize = CSettings::GetInt("Water", "LargeRippleSize");

  // set resolution
  gridSize = kFineVerts;
  squareSize = kFineSquare;

  // allocate heights, normals, and faces arrays
  int i, r;

  for (i = 0; i < 2; i++) {
	heights[i] = new float *[gridSize];
	faces[i] = new CVector3 *[gridSize + 1];
	assert(heights[i] && faces[i]);
  }
  normals = new CVector3 *[gridSize];
  assert(normals);

  for (r = 0; r < gridSize + 1; r++) {
	for (i = 0; i < 2; i++) {
	  faces[i][r] = new CVector3[gridSize + 1];
	  assert(faces[i][r]);
	  if (r != gridSize) {
		heights[i][r] = new float[gridSize];
		assert(heights[i][r]);
	  }
	}
	normals[r] = new CVector3[gridSize];
	assert(normals[r]);
  }


  // allocate ripple arrays
  smallRipple = new float *[kSmallRippleSize];
  largeRipple = new float *[kLargeRippleSize];
  assert(smallRipple && largeRipple);

  for (r = 0; r < kSmallRippleSize; r++) {
	smallRipple[r] = new float[kSmallRippleSize];
	assert(smallRipple[r]);
  }
  for (r = 0; r < kLargeRippleSize; r++) {
	largeRipple[r] = new float[kLargeRippleSize];
	assert(largeRipple[r]);
  }


  // init
  InitHeightField();
  PrecomputeRipples();


  // set cur and prev pointers
  cur = 0;
  prev = 1;
}


CWater::~CWater()
{
  // deallocate heights, normals, and faces arrays
  int i, r;

  for (r = 0; r < gridSize + 1; r++) {
	for (i = 0; i < 2; i++) {
	  delete[] faces[i][r];
	  if (r != gridSize) 
		delete[] heights[i][r];
	}

	delete[] normals[r];
  }

  for (i = 0; i < 2; i++) {
	delete[] heights[i];
	delete[] faces[i];
  }
//  delete[] normals;


  // deallocate ripple arrays
  for (r = 0; r < kSmallRippleSize; r++)
	delete[] smallRipple[r];
  for (r = 0; r < kLargeRippleSize; r++)
	delete[] largeRipple[r];

  delete[] smallRipple;
  delete[] largeRipple;
}


/* Right now I draw the height field as a bunch of triangle strips,  one for each
 * column, with immediate mode glVertex calls. Eventually this will get converted
 * to vertex arrays because they're significantly faster.
 */
// TODO: convert this to vertex arrays!
// TODO: could I store the vectors for each point so I can avoid all the
// floating point multiplies?
void CWater::Draw() const
{
  static const CTexture refract(CSettings::GetString("Floor", "TexFile").c_str());

  glDisable(GL_LIGHTING);

  // draw normals?
  if (gDrawNormals)
	DrawNormals();

  if (gWireframe) {
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);
	DrawMesh(kRWireframe);
	return;
  }

  if (gTransparent) {
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_BLEND);
  }

  // draw with environment map
  if (gEnvMap) {
	EnvMap();
	DrawMesh(kREnvMap);
  }

  // draw with regular texture
  if (gTexture) {
	Texture();
	DrawMesh(kRTexture);
  }

  // draw with refraction.
  if (gRefract) {
	static float color[4]	=
	  { 1.0, 1.0, 1.0, CSettings::GetFloat("Water", "RefractAlpha") };
	glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
	glColor4fv(color);
	refract.Bind();
	DrawMesh(kRRefract);
  }

  // otherwise, draw with flat shading
  if (!gTexture && !gEnvMap && !gRefract) {
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);
	glColor3f(1.0, 1.0, 1.0);
	DrawMesh(kRNone);
  }
}


void CWater::Update(int msecs)
{
  // update procedural mesh stuff
  if (gWaves)
	UpdateWaves(msecs);
  UpdateRipples(msecs);

  // get normals
  GenerateNormals();
}


void CWater::Touch(float x, float y, bool small)
{
  int row = YToRow(y), col = XToCol(x);

  if (row < 0 || row >= gridSize || col < 0 || col >= gridSize)
	return;

  // add in ripple
  int	size		= (small) ? kSmallRippleSize : kLargeRippleSize;
  float	**ripple	= (small) ? smallRipple : largeRipple;

  // NOTE: r and c are indices into the precomputed ripple, hr and hc are
  // indices into the actual water
  for (int r = 0, hr = row + r - size / 2; r < size; r++, hr++) {
	// are we off the edge?
	if (hr < 0)
	  continue;
	if (hr >= gridSize)
	  break;

	for (int c = 0, hc = col + c - size / 2; c < size; c++, hc++) {
	  // are we off the edge?
	  if (hc < 0)
		continue;
	  if (hc >= gridSize)
		break;

	  heights[cur][hr][hc] += ripple[r][c];
	}
  }
}


// -----------------------------------------------------------------------------
// PRIVATE METHODS
// -----------------------------------------------------------------------------


/* InitVertexArrays
 * ----------------
 * Sets up OpenGL to treat the heightfield, normals, and other pertinent data
 * as vertex arrays. 
 *
 * NOTE: not used right now.
 */
void CWater::InitVertexArrays() const
{
  // turn on vertex arrays
  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_NORMAL_ARRAY);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);

  // tell OpenGL which arrays to use
  //glVertexPointer
}


/* InitHeightField
 * ---------------
 * Zeroes out the heights and face values.
 */
void CWater::InitHeightField()
{
  // initialize heights and face normals to 0
  for (int i = 0; i < 2; i++) {
	for (int r = 0; r < gridSize; r++) {
	  memset(heights[i][r], 0, gridSize * sizeof(float));
	  memset(faces[i][r], 0, gridSize * sizeof(CVector3));
	}
  }
}


/* UpdateWave
 * ----------
 * Updates waves that flow across the heightfield. The waves are generated by a
 * sum of sine functions.
 */
void CWater::UpdateWaves(int msecs)
{
  static float phase = 0;
  float x, y;

  // update phase shift
  phase += (M_PI * msecs / 1000);
  if (phase > 2 * M_PI)
	phase -= 2 * M_PI;


  // procedurally modify heights
  x = ColToX(0);

  for (int c = 0; c < gridSize; c++) {
	y = RowToY(0);

	for (int r = 0; r < gridSize; r++) {
	  heights[cur][r][c] += sin(phase + M_PI * x / kWaveXPhase) / 100;
	  heights[cur][r][c] += cos(phase + M_PI * y / kWaveYPhase) / 100;
	  y += squareSize;
	}

	x += squareSize;
  }
}


/* UpdateRipples
 * -------------
 * Updates ripples that are caused by disturbances in the water surface. The
 * RippleFilter function is defined in ripple.cpp.
 */
void CWater::UpdateRipples(int msecs)
{
  // apply filter
  RippleFilter(heights[cur], heights[prev], gridSize);

  // swap buffers
  int temp = prev;
  prev = cur;
  cur = temp;
}


/* PrecomputeRipples
 * -----------------
 * Precomputes the heightfields for two ripples, one large and one small.
 */
void CWater::PrecomputeRipples()
{
  static const float kSmallRippleDepth =
	CSettings::GetFloat("Water", "SmallRippleDepth");
  static const float kLargeRippleDepth =
	CSettings::GetFloat("Water", "LargeRippleDepth");

  MakeRipple(smallRipple, kSmallRippleSize, kSmallRippleDepth);
  MakeRipple(largeRipple, kLargeRippleSize, kLargeRippleDepth);
}


/* DrawMesh
 * --------
 * Draws the actual water. This may be called multiple times (for multiple
 * passes) with the regular texture, the environment map, the refraction, etc.
 *
 * If refract is true, RefractCoords is called before drawing each vertex in
 * order to specify the refracted texture coordinates.
 */
// TODO: put water vertices into a vertex array
// TODO: try passing &normals[...] to glNormal3fv

void CWater::DrawMesh(RenderType render) const
{
  float x, nextX, y;

  x = ColToX(0);
  nextX = x + squareSize;

  for (int col = 0; col < gridSize - 1; col++) {
	if (render == kRWireframe) {
	  glLineWidth(1.0);
	  glColor4f(1.0, 1.0, 1.0, 1.0);
	  glBegin(GL_LINE_STRIP);
	}
	else
	  glBegin(GL_TRIANGLE_STRIP);

	y = RowToY(0);
	for (int row = 0; row < gridSize; row++) {

	  // do any per-vertex calculations
	  if (render == kRRefract)
		RefractCoords(row, col);
	  else if (gFresnel && (render == kREnvMap || render == kRTexture))
		FresnelTerm(row, col);

	  // draw first vert
	  glNormal3f(normals[row][col].x, normals[row][col].y, normals[row][col].z);
	  glVertex3f(x, y, heights[cur][row][col]);

	  // next column
	  int c2 = col + 1;

	  // do any per-vertex calculations
	  if (render == kRRefract)
		RefractCoords(row, c2);
	  else if (render == kREnvMap && gFresnel)
		FresnelTerm(row, c2);

	  // draw second vert
	  glNormal3f(normals[row][c2].x, normals[row][c2].y, normals[row][c2].z);
	  glVertex3f(nextX, y, heights[cur][row][c2]);

	  // increment y
	  y += squareSize;
	}

	glEnd();

	// increment x, nextX
	x = nextX;
	nextX = x + squareSize;
  }
}


/* DrawNormals
 * -----------
 * Draws all of the normal vectors for the water surface mesh.
 */
void CWater::DrawNormals() const
{
  float x, y;

  glDisable(GL_LIGHTING);

  glBegin(GL_LINES);
  glColor3ub(255, 0, 0);

  x = ColToX(0);

  for (int c = 0; c < gridSize; c++) {
	y = RowToY(0);

	for (int r = 0; r < gridSize; r++) {
	  glVertex3f(x, y, heights[cur][r][c]);
	  glVertex3f(x + normals[r][c].x, y + normals[r][c].y,
				 heights[cur][r][c] + normals[r][c].z);
	  y += squareSize;
	}

	x += squareSize;
  }

  glEnd();
  glEnable(GL_LIGHTING);
}


/* GenerateNormals
 * ---------------
 * Generates the normals for the water surface. This is stupid and brute-force
 * right now.
 *
 * Could I pre-compute a bunch of normals for a quantized set of heights?
 * sounds like way too much memory....
 * TODO: check this out
 *
 * NOTE: right now, I go through the height field and calculate the face normals
 * for every triangle. (gridSize * gridSize * 2 tris). then I go through and
 * average the face normals to get the vertex normals.
 *
 * so...say you have a subset of the water surface mesh that looks like this:
 *
 *			   column
 *			  c    c+1
 *			+----+----+
 *		 r	| \ 1| \ 1|
 *			|0 \ |0 \ |
 *	row		+----+----+
 *			| \ 1| \ 1|
 *		r+1	|0 \ |0*\ |
 *			+----+----+
 *
 * To get the normal for the triangle with the asterisk next to it, read the
 * faces array like this:
 *		faces[0][r + 1][c + 1]
 *
 *
 * NOTE: the array of face normals is gridSize + 1 on each side. the edge rows
 * are permanently zero. this makes summing the normals faster since I don't
 * have to check for edges.
 */
void CWater::GenerateNormals()
{
  CVector3 across(squareSize, 0, 0),
		   diag(squareSize, -squareSize, 0),
		   down(0, -squareSize, 0);
  int c, r;

  // calculate face normals
  for (c = 1; c < gridSize; c++) {
	for (r = 1; r < gridSize; r++) {
	  across.z = heights[cur][r - 1][c] - heights[cur][r - 1][c - 1];
	  diag.z = heights[cur][r][c] - heights[cur][r - 1][c - 1];
	  down.z = heights[cur][r][c - 1] - heights[cur][r - 1][c - 1];

	  faces[0][r][c] = down.Cross(diag);
	  faces[1][r][c] = diag.Cross(across);
	}
  }

  // clear vertex normals
  //memset(normals, 0, sizeof(normals));

  // EXPERIMENTAL: copy face normals into outside edge faces
  //memcpy(&faces[gridSize], &faces[gridSize - 1], sizeof(CVector3) * gridSize * 2);

  // get vertex normals by summing face normals
  for (c = 0; c < gridSize; c++) {
	for (r = 0; r < gridSize; r++) {
	  CVector3 &n = normals[r][c];
	  CVector3 &f1 = faces[0][r][c], &f2 = faces[1][r][c],
			   &f3 = faces[0][r][c + 1], &f4 = faces[1][r + 1][c],
			   &f5 = faces[0][r + 1][c + 1], &f6 = faces[1][r + 1][c + 1];

	  n = f1 + f2 + f3 + f4 + f5 + f6;

	  // TODO: benchmark this
	  // written this way to outsmart the dumb c++ compiler
	  //n.x = f1.x + f2.x + f3.x + f4.x + f5.x + f6.x;
	  //n.y = f1.y + f2.y + f3.y + f4.y + f5.y + f6.y;
	  //n.z = f1.z + f2.z + f3.z + f4.z + f5.z + f6.z;

	  n.Normalize();
	}
  }
}


/* Texture
 * -------
 * Sets up the water for drawing with its regular texture.
 */
void CWater::Texture() const
{
  static const CTexture tex(CSettings::GetString("Water", "TexFile").c_str());
  static float x[4]		= { 0.1, 0.0, 0.0, 0.0 };
  static float y[4]		= { 0.0, 0.1, 0.0, 0.0 };
  static float diffuse[4]= { 1.0, 1.0, 1.0,
							 CSettings::GetFloat("Water", "TexAlpha") };

  tex.Bind();

  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, diffuse);
  glColor4fv(diffuse);

  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexGenfv(GL_S, GL_OBJECT_PLANE, x);
  glTexGenfv(GL_T, GL_OBJECT_PLANE, y);
  glEnable(GL_TEXTURE_GEN_S);
  glEnable(GL_TEXTURE_GEN_T);
}


/* EnvMap
 * -------
 * Sets up a spherical environment map.
 */
void CWater::EnvMap() const
{
  static const CTexture envmap(
	CSettings::GetString("Water", "EnvMapFile").c_str());
  static float color[4]	=
	{ 1.0, 1.0, 1.0, CSettings::GetFloat("Water", "EnvMapAlpha") };

  envmap.Bind();
	
  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
  glColor4fv(color);

  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
  glEnable(GL_TEXTURE_GEN_S);
  glEnable(GL_TEXTURE_GEN_T);
}



