/* File: CTexture.cpp
 * Author: Ryan Barrett (rbarrett@stanford.edu)
 * --------------------
 * A simple, small texture class. For more information, see CTexture.h.
 */

#include <stdio.h>
#include <math.h>
#include <assert.h>
#include "texture.h"


// ----------------------------------------------------------------------------
// PUBLIC METHODS
// ----------------------------------------------------------------------------

CTexture::CTexture(const char *rawfile)
{
	FILE *file;
	int len, size;
	unsigned char *pixels;

	// open RAW file
	file = fopen(rawfile, "rb");
	assert(file);

	// get size of image
	fseek(file, 0, SEEK_END);
	len = ftell(file);
	fseek(file, 0, SEEK_SET);
	assert(len % 3 == 0);

	// load texture data from RAW file
	pixels = new unsigned char[len];
	assert(pixels);

	for (int i = 0; i < len; i += 3) {
		if (fread(&pixels[i], sizeof(char), 3, file) != 3)
			assert(false);
	}

	// set texture
	size = static_cast<int>(sqrt(len / 3));
	assert(sqrt(len / 3) == size);

	SetTexture(size, pixels);

	// clean up
	delete [] pixels;
	fclose(file);
}


CTexture::CTexture(const int size, const unsigned char *pixels)
{
	SetTexture(size, pixels);
}


void CTexture::Bind() const
{
  glBindTexture(GL_TEXTURE_2D, mTexture);
}


// ----------------------------------------------------------------------------
// PRIVATE METHODS
// ----------------------------------------------------------------------------

void CTexture::SetTexture(const int size, const unsigned char *pixels)
{
	// set size
	assert(size == 8 || size == 16 || size == 32 || size == 64 || size == 128 ||
		   size == 256 || size == 512);
	mSize = size;

	// set up texture
	glGenTextures(1, &mTexture);
	glBindTexture(GL_TEXTURE_2D, mTexture);
	glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, 3, mSize, mSize, 0, GL_RGB,
		GL_UNSIGNED_BYTE, pixels);
}



CTexture::~CTexture()
{
	glDeleteTextures(1, &mTexture);
}


