/* File: ripple.cpp
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 * CS248 Fall 2001
 * HW3 - Video Game
 *
 * Defines the ripple functions.
 */

#include <string.h>		// for memset


/* RippleFilter
 * ------------
 * Applies the ripple filter to the heightfield in cur. Assumes the the previous
 * heights are stored in prev and the size of the heightfields is size (i.e.
 * both dimensions of the cur and prev arrays).
 *
 * NOTE: this is an inner loop, so i've tried to optimize it pretty heavily. I
 * think it can be even faster though.
 */
void RippleFilter(float **cur, float **prev, const int size)
{
  static const float kSmooth = CSettings::GetFloat("Water", "RippleSmooth");
  static const float kDamp = CSettings::GetFloat("Water", "RippleDamp");

  for (int c = 1; c < size - 1; c++) {
	for (int r = 1; r < size - 1; r++) {
	  // get filtered value
	  float filtered = cur[r - 1][c] + cur[r][c - 1] +
					   cur[r + 1][c] + cur[r][c + 1];
	  // apply smoothing
	  filtered *= kSmooth;

	  // add in acceleration (value from two frames ago)
	  filtered -= prev[r][c];

	  // apply damping and store
	  prev[r][c] = filtered * kDamp;
	}
  }
}


/* MakeRipple
 * ----------
 * Precomputes the heightfield for a ripple of the given depth and stores it in
 * the provided heightfield. The depth corresponds (roughly) to the depth that
 * the water is displaced before it begins to move back into the hole. The size
 * parameter is the size of the array (both dimensions).
 *
 * NOTE: depth should be *positive*! (i.e. the absolute value of the depth of
 * the ripple.)
 */
void MakeRipple(float **heights, int size, float depth)
{
  static const int kPrecompFilters =
	CSettings::GetInt("Water", "PrecompFilters");
  static const float kSquareSize =
	CSettings::GetFloat("Water", "FineSquare");
  float **prev;
  int i, r;

  assert(size >= 4);
  assert(depth > 0);
  assert(kPrecompFilters % 2 == 0);

  // allocate array to store previous heights
  prev = new float *[size];
  assert(prev);
  for (i = 0; i < size; i++) {
	prev[i] = new float[size];
	assert(prev[i]);
  }


  // start ripple
  for (r = 0; r < size; r++) {
	memset(heights[r], 0, size * sizeof(float));
	memset(prev[r], 0, size * sizeof(float));
  }


  // apply filter
  for (i = 0; i < kPrecompFilters; i++) {
	heights[size / 2][size / 2] -= kSquareSize * depth;
	RippleFilter(heights, prev, size);

	// switch arrays
	float **temp = heights;
	heights = prev;
	prev = temp;
  }

  // deallocate prev array
  for (r = 0; r < size; r++)
	delete[] prev[r];
  delete[] prev;
}
