/* File: piece.cpp
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 * CS248 Fall 2001
 * HW3 - Video Game
 *
 * Defines the CPiece class. For more information, see piece.h.
 *
 * CITATION: The code for drawing a torus was taken from the online Red Book
 * (OpenGL Programming Guide, Addison Wesley, blah blah, etc.) at:
 *
 * http://ask.ii.uib.no/ebt-bin/nph-dweb/dynaweb/SGI_Developer/OpenGL_PG/
 */

#include <GL/glut.h>
#include <iostream>
#include <assert.h>
//#include <plib/sl.h>
#include "settings.h"
#include "vector3.h"
#include "board.h"
#include "tictactoe.h"
#include "water.h"
#include "floor.h"
#include "texture.h"
#include "piece.h"


#ifndef M_PI
#define M_PI 3.14159265	// because vc++'s math.h doesn't define M_PI
#endif


// -----------------------------------------------------------------------------
// GLOBAL VARIABLES
// -----------------------------------------------------------------------------
extern CTicTacToe	gGame;
extern CWater		gWater;
//extern slScheduler	gSL;
extern bool			gSharpEdges;
extern bool			gWireframe;
extern bool			gSound;


// -----------------------------------------------------------------------------
// HELPER FUNCTION PROTOTYPES
// -----------------------------------------------------------------------------
void DrawListAt(GLuint list, const CVector3& pos);

void SharpQuadStrip(float x1, float y1, float z1,
					float x2, float y2, float z2,
					float nx, float ny, float nz);


// -----------------------------------------------------------------------------
// INITIALIZE STATIC MEMBERS
// -----------------------------------------------------------------------------
const float CPiece::kInitVelocity	= 1;	// units per second
const float CPiece::kWaterVelocity	= 1;
const float CPiece::kAirAccel		= 2.5;	// units per second^2
const float CPiece::kSize			= CBoard::kSquareSize * .8;
const float CPiece::kDepth			= CBoard::kSquareSize * .25;

unsigned int	CPieceX::sDLSharp	= 0;
unsigned int	CPieceX::sDLSmooth	= 0;
const float		CPieceX::kColor[4]	= { 1.0, 1.0, 1.0, 1.0 };		// white

unsigned int	CPieceO::sDispList	= 0;
const float		CPieceO::kColor[4]	= { 1.0, 1.0, 1.0, 1.0 };		// white


// -----------------------------------------------------------------------------
// CPIECE METHODS
// -----------------------------------------------------------------------------

CPiece::CPiece(const CVector3& pos) : pos(pos), normal(0, 0, 1)  // points up
{
  velocity = kInitVelocity;
  falling = true;
  playedSplash = false;
}

void CPiece::Update(int msecs)
{
  static const float kFloatFactor = CSettings::GetFloat("Piece", "FloatFactor");
//  static slSample splash(const_cast<char *>
//						 (CSettings::Get<string>("Piece", "DropFile").c_str()),
//						 &gSL);
  float waterHeight = gWater.GetHeight(pos.x, pos.y);

  if (falling) {
	// it's in the air, accelerate it due to gravity
	velocity += kAirAccel * msecs / 1000;
	pos.z -= velocity * msecs / 1000;

	// did it hit the water?
	if (pos.z < waterHeight) {
	  gWater.Touch(pos.x, pos.y, false);
	  falling = false;
	  gGame.PieceLanded();	// tell game to continue
	}

	// preemptively play sound to get it to sync
	if (!playedSplash && gSound && pos.z - CBoard::kStartZ * 2 / 3 < waterHeight) {
	  // play splash sound
//	  gSL.playSample(&splash);
	  playedSplash = true;
	}

  }
  else {		// it's in the water
	// update the piece's height
	pos.z += (waterHeight - pos.z) * kFloatFactor;

	// update the piece's normal
	const CVector3& waterNormal = gWater.GetNormal(pos.x, pos.y);
	normal = normal * (1 - kFloatFactor) + waterNormal * kFloatFactor;
  }
}


/* Orient
 * ------
 * If the piece is on the water's surface, the modelview matrix is oriented
 * so that the piece is perpendicular to the water surface normal. (If it's
 * falling, it points straight up.)
 */
void CPiece::Orient() const
{
  glTranslatef(pos.x, pos.y, pos.z);		// translate to origin
  //glRotatef(90, normal.x, normal.y, 0);		// rotate
  glRotatef(asin(normal.y) * 180 / M_PI, 1, 0, 0);
  glRotatef(asin(normal.x) * 180 / M_PI, 0, 1, 0);
  glTranslatef(-pos.x, -pos.y, -pos.z);		// translate back
}


/* SharpHelper
 * -----------
 * Calls SharpQuadStrip with the points (x y 0) and (x y CPiece::kDepth). The z
 * coordinate of the normal is 0.
 */
inline void CPiece::SharpHelper(float x, float y, float nx, float ny)
{
  SharpQuadStrip(x, y, -kDepth / 3, x, y, kDepth * 2 / 3, nx, ny, 0);
}


// -----------------------------------------------------------------------------
// CPIECEX METHODS
// -----------------------------------------------------------------------------
CPieceX::CPieceX(const CVector3& pos) : CPiece(pos)
{
  if (sDLSmooth == 0 || sDLSharp == 0)
	CreateDispList();

  assert(sDLSmooth != 0 && sDLSharp != 0);
}


void CPieceX::Draw() const
{
  glPushMatrix();
  Orient();

  if (gWireframe) {
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);
	glColor4f(1.0, 0.0, 0.0, 1.0);
  }

  if (gSharpEdges)
	DrawListAt(sDLSharp, pos);
  else
	DrawListAt(sDLSmooth, pos);

  glPopMatrix();
}


void CPieceX::CreateDispList()
{
  static const CTexture tex(CSettings::GetString("Piece", "XTexFile").c_str());

  GLuint list = glGenLists(2);
  assert(list != 0);

  sDLSharp = list;
  sDLSmooth = list + 1;

  // create sharp edged display list
  glNewList(sDLSharp, GL_COMPILE);

  tex.Bind();
  DrawTop();
  DrawSidesSharp();

  glEndList();


  // create smooth shaded display list
  glNewList(sDLSmooth, GL_COMPILE);

  /*
  glEnable(GL_TEXTURE_2D);
  glColor3f(1.0, 1.0, 1.0);
  glDisable(GL_BLEND);

  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glEnable(GL_TEXTURE_GEN_S);
  glEnable(GL_TEXTURE_GEN_T);
  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, kColor);
  */

  tex.Bind();
  DrawTop();
  DrawSidesSmooth();

  glEndList();
}


/* DrawTop
 * -------
 * Draws the top of the X. This is not called to actually draw, it's only called when
 * the display list is being created.
 */
void CPieceX::DrawTop()
{
  const float h	= kSize * .5,	// half of kSize
			  q	= kSize * .25;	// quarter "
			  //e = kSize * .125;	// eighth  "

  glBegin(GL_QUADS);

  glNormal3f(0, 0, 1.0);		// top of the X, surface normals are straight up

  // lower left to upper right
  glVertex3f(-h, -h, kDepth * 2 / 3);
  glVertex3f(-q, -h, kDepth * 2 / 3);
  glVertex3f(h, h, kDepth * 2 / 3);
  glVertex3f(q, h, kDepth * 2 / 3);
  
  // lower right to upper left
  glVertex3f(q, -h, kDepth * 2 / 3);
  glVertex3f(h, -h, kDepth * 2 / 3);
  glVertex3f(-q, h, kDepth * 2 / 3);
  glVertex3f(-h, h, kDepth * 2 / 3);

  glEnd();
}  

/* DrawSidesSharp
 * --------------
 * Draws the sides of the X with sharp edges. (Each face should have its own normal,
 * so this draws each face separately and the shared vertices have distinct normals
 * for each face.) This is not called to actually draw, it's only called when the
 * display list is being created.
 */
void CPieceX::DrawSidesSharp()
{
  const float h	= kSize * .5,	// half of kSize
			  q	= kSize * .25,	// quarter "
			  e = kSize * .125;	// eighth  "
  const float nx = .8, ny = .6;	// absolute values of coords for diagonal normals

  glBegin(GL_QUADS);

  // starting at the lower left corner 
  SharpHelper(-h, -h, 0, 0);		// startup call, doesn't draw a quad
  SharpHelper(-e, 0, -nx, ny);
  SharpHelper(-h, h, -nx, -ny);
  SharpHelper(-q, h, 0, -1);
  SharpHelper(0, e, nx, ny);
  SharpHelper(q, h, -nx, ny);

  // now we're at the upper right corner
  SharpHelper(h, h, 0, -1);
  SharpHelper(e, 0, nx, -ny);
  SharpHelper(h, -h, nx, ny);
  SharpHelper(q, -h, 0, -1);
  SharpHelper(0, -e, -nx, -ny);
  SharpHelper(-q, -h, nx, -ny);
  SharpHelper(-h, -h, 0, -1);

  glEnd();
}


/* DrawSidesSmooth
 * ---------------
 * Draws the sides of the X with rounded edges. This uses GL_QUAD_STRIP so that each
 * vertex only has one normal. The geometry is the same, but the normals are shared
 * so that there are no sharp transitions between shading for the faces. This is
 * easier to write code for, but DrawSidesSharp is more realistic and no more
 * expensive at runtime.
 */
#define TOP_AND_BOTTOM	for (z = -kDepth / 3; z <= kDepth * 2 / 3; z += kDepth)

void CPieceX::DrawSidesSmooth()
{
  const float h	= kSize * .5,	// half of kSize
			  q	= kSize * .25,	// quarter "
			  e = kSize * .125;	// eighth  "
  float z;

  glBegin(GL_QUAD_STRIP);

  // starting at the lower left corner 
  TOP_AND_BOTTOM { glNormal3f(-q, -e, 0);		glVertex3f(-h, -h, z); }
  TOP_AND_BOTTOM { glNormal3f(-1, 0, 0);		glVertex3f(-e, 0, z); }
  TOP_AND_BOTTOM { glNormal3f(-q, e, 0);		glVertex3f(-h, h, z); }
  TOP_AND_BOTTOM { glNormal3f(e, q, 0);			glVertex3f(-q, h, z); }
  TOP_AND_BOTTOM { glNormal3f(0, 1, 0);			glVertex3f(0, e, z); }
  TOP_AND_BOTTOM { glNormal3f(-e, q, 0);		glVertex3f(q, h, z); }

  // now we're at the upper right corner
  TOP_AND_BOTTOM { glNormal3f(q, e, 0);			glVertex3f(h, h, z); }
  TOP_AND_BOTTOM { glNormal3f(1, 0, 0);			glVertex3f(e, 0, z); }
  TOP_AND_BOTTOM { glNormal3f(q, -e, 0);		glVertex3f(h, -h, z); }
  TOP_AND_BOTTOM { glNormal3f(-e, -q, 0);		glVertex3f(q, -h, z); }
  TOP_AND_BOTTOM { glNormal3f(0, -1, 0);		glVertex3f(0, -e, z); }
  TOP_AND_BOTTOM { glNormal3f(e, -q, 0);		glVertex3f(-q, -h, z); }
  TOP_AND_BOTTOM { glNormal3f(-q, -e, 0);		glVertex3f(-h, -h, z); }

  glEnd();
}


// -----------------------------------------------------------------------------
// CPIECEO METHODS
// -----------------------------------------------------------------------------
CPieceO::CPieceO(const CVector3& pos) : CPiece(pos)
{
  if (sDispList == 0)
	sDispList = CreateDispList();
  assert(sDispList != 0);
}


void CPieceO::Draw() const
{
  glPushMatrix();
  Orient();

  if (gWireframe) {
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);
	glColor4f(0.0, 0.0, 0.0, 1.0);
  }

  DrawListAt(sDispList, pos);

  glPopMatrix();
}


unsigned int CPieceO::CreateDispList()
{
  static const CTexture tex(CSettings::GetString("Piece", "OTexFile").c_str());

  // create display list
  GLuint list = glGenLists(1);
  glNewList(list, GL_COMPILE);

  /*
  glEnable(GL_TEXTURE_2D);
  glColor3f(1.0, 1.0, 1.0);
  glDisable(GL_BLEND);

  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexGenfv(GL_S, GL_OBJECT_PLANE, x);
  glTexGenfv(GL_T, GL_OBJECT_PLANE, y);
  glEnable(GL_TEXTURE_GEN_S);
  glEnable(GL_TEXTURE_GEN_T);
  */
  //glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, kColor);

  tex.Bind();
  glutSolidTorus(CPiece::kSize * .2, CPiece::kSize * .4, 16, 25);

  glEndList();
  return list;
}



// -----------------------------------------------------------------------------
// HELPER FUNCTIONS
// -----------------------------------------------------------------------------
/* DrawListAt
 * ----------
 * Draws a display list at a given place. (The modelview matrix is translated to
 * the given position and then translated back after the display list is called.)
 */
void DrawListAt(GLuint list, const CVector3& pos)
{
  assert(glIsList(list) == GL_TRUE);

  // translate to pos
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glTranslatef(pos.x, pos.y, pos.z);
 
  // call display list
  glCallList(list);

  // pop matrix stack
  glPopMatrix();
}


/* SharpQuadStrip
 * --------------
 * I wrote this helper because OpenGL's GL_QUAD_STRIP primitive doesn't work for
 * objects with sharp edges. However, it is useful because you don't have to specify
 * vertices twice.
 *
 * SharpQuadStrip draws a quad with GL_QUADS using the two given points and
 * the last two points passed in. The first call to SharpQuadStrip stores
 * the points passed in and does nothing.
 *
 * To reset SharpQuadStrip to a new quad strip, pass (0 0 0) for the normal
 * vector.
 *
 * NOTE: SharpQuadStrip does *not* check that the normal vector is
 * normalized!
 *
 * NOTE: SharpQuadStrip does *not* call glBegin(GL_QUADS) or glEnd()! Make
 * sure you make those calls before and after you call SharpQuadStrip!
 */
void SharpQuadStrip(float x1, float y1, float z1,
					float x2, float y2, float z2,
					float nx, float ny, float nz)
{
  //static CVector3 lastpt1(pt1), lastpt2(pt2);	// only initialized on the first call
  static float lastx1 = x1, lasty1 = y1, lastz1 = z1,
			   lastx2 = x2, lasty2 = y2, lastz2 = z2;

  if (nx == 0 && ny == 0 && nz == 0)  // first call
	return;

  //glNormal3f(normal.x, normal.y, normal.z);
  //glVertex3f(lastpt1.x, lastpt1.y, lastpt1.z);
  //glVertex3f(lastpt2.x, lastpt2.y, lastpt2.z);
  //glVertex3f(pt2.x, pt2.y, pt2.z);
  //glVertex3f(pt1.x, pt1.y, pt1.z);
  glNormal3f(nx, ny, nz);
  glVertex3f(lastx1, lasty1, lastz1);
  glVertex3f(lastx2, lasty2, lastz2);
  glVertex3f(x2, y2, z2);
  glVertex3f(x1, y1, z1);

  lastx1 = x1;
  lasty1 = y1;
  lastz1 = z1;
  lastx2 = x2;
  lasty2 = y2;
  lastz2 = z2;
}

