/* File: main.cpp
 * Author: Ryan Barrett (rbarret@stanford.edu)
 * --------------------
 * CS248 Fall 2001
 * HW3 - Video Game
 *
 * The main file for the game. Contains main() and lots of GLUT callbacks and not
 * much else. :P
 *
 * CITATIONS:
 * - The stlini library for reading and writing .ini files. Written by Robert
 *   Kesterson. You can get it at:
 *		http://robertk.com/source/
 * - Some of the GLUT initialization (like the code in main()) was copied from
 *   Matt Ginzton's glut_example program.
 *
 */

#include <GL/glut.h>
//#include <plib/sl.h>	// for sound
#include <iostream>
#include <assert.h>
#include <time.h>
#include <stdlib.h>
#include <math.h>
#include "settings.h"
#include "floor.h"
#include "tictactoe.h"
#include "water.h"
#include "raindrop.h"
#include "widget/wwindow.h"
#include "widget/wscrolledwindow.h"
#include "widget/witem.h"
#include "widget/wcheckbox.h"


#ifndef M_PI
#define M_PI 3.14159265	// because vc++'s math.h doesn't define M_PI
#endif


// -----------------------------------------------------------------------------
// CONSTANTS
// -----------------------------------------------------------------------------
// init static members of CSettings
// moved here from settings.cpp as a hack to get vc++'s init order right
const string	CSettings::kFilename("tictactoe.ini");
INIFile			CSettings::sFile;
bool			CSettings::sLoaded = false;

// graphics stuff
static const float	kNearPlane		= 1;
static const float	kFarPlane		= 100;
static float		kViewDistance	= 0; //CSettings::GetInt("Game", "ViewDistance");

// raindrop stuff
static int			kNumRaindrops	= 0; //CSettings::GetInt("Rain", "NumRaindrops");

// time stuff
static const int	kClocksPerMsec	= CLOCKS_PER_SEC / 1000;
static const int	kMinUpdateMsecs	= 30;	// update rate is 30fps

// lighting stuff
static const float	kLightPos[4]	= { 0, 0, 50, 1 };
static const float	kAmbient[4]		= { .2, .2, .2, 1.0 };

// user input stuff
static const int	kDeltaAngle		= 2;	// mouse pixels per degree change
static int			kMinPhi			= 0; //CSettings::GetInt("Game", "MinPhi");



// -----------------------------------------------------------------------------
// GLOBAL VARIABLES
// -----------------------------------------------------------------------------
//CSettings	gSettings;		// the settings manager (not used, just a hack to get vc++ init order right)
//slScheduler gSL;			// the SL sound manager
CFloor		gFloor;			// the pool floor
CTicTacToe	gGame;			// the tic-tac-toe board
CWater		gWater;			// the water surface
CRaindrop	*gRaindrops;	// the raindrops
CVector3	gViewPos;		// the viewer position
CVector3	gViewDir;		// a normalized vector for the view orientation
int			gViewTheta;		// viewing angle in degrees (3D polar coordinates)
int			gViewPhi;
int			gWindowWidth	= 0; //CSettings::GetInt("Game", "WindowWidth");
int			gWindowHeight	= 0; //CSettings::GetInt("Game", "WindowHeight");
//slSample	gLoopSample(const_cast<char *>
//				 (CSettings::Get<string>("Rain", "LoopFile").c_str()), &gSL);

CWWindow	*gMenu = NULL, *gChildMenu = NULL, *gChildMenu2 = NULL;
int			gMouseX = 0, gMouseY = 0;
bool		gLeftButtonDown = false;

// debugging options
bool		gWaves			= false;
bool		gDrawNormals	= false;
bool		gDrawObjects	= true;
bool		gDrawWater		= true;
bool		gDrawRain		= true;
bool		gPrintFPS		= false;
bool		gSharpEdges		= false;
bool		gEnvMap			= true;
bool		gRefract		= false;
bool		gFresnel		= false;
bool		gTexture		= true;
bool		gTransparent	= true;
bool		gWireframe		= false;
//bool		gDrawMenu		= false;
bool		gSound			= true;



// -----------------------------------------------------------------------------
// PROTOTYPES
// -----------------------------------------------------------------------------
void InitGraphics();
void UpdateView();
void PrintFPS();
void Shutdown();

// widget ui stuff
void ShowMenu(int x = 100, int y = 100);
void GetMouse(int &x, int &y);
void WNewGameMenu(CWItem *item);	// callbacks
void WNewGame(CWItem *item);
void WMoveMenu(CWItem *item);
void WMoveMenu2(CWItem *item);
void WMove(CWItem *item);
void WOptionsMenu(CWItem *item);
void WOptionSet(CWItem *item);
void WQuit(CWItem *item);

// *** LOOK AT THIS!!! ***
// I'm including widgetcalls.cpp, which defines the widget ui functions above.
// I put these functions in their own file because they're sort of separate, and
// main.cpp was getting too big.
// *** LOOK AT THIS!!! ***
#include "widgetcalls.cpp"

// I'm also including extensions.cpp, which defines isExtensionSupported (for
// dealing with OpenGL extensions
//#include "extensions.cpp



// glut callbacks:
void Display();
void Reshape(GLint width, GLint height);
void Update();
void Keyboard(unsigned char key, int x, int y);
void Mouse(int button, int state, int x, int y);
void Motion(int x, int y);
void PassiveMotion(int x, int y);


// -----------------------------------------------------------------------------
// MAIN
// -----------------------------------------------------------------------------

int main(int argc, char **argv)
{
  // init "constants" (damn vc++ can't do init order right)
  kViewDistance	= CSettings::GetInt("Game", "ViewDistance");
  kNumRaindrops	= CSettings::GetInt("Rain", "NumRaindrops");
  kMinPhi		= CSettings::GetInt("Game", "MinPhi");
  gWindowWidth	= CSettings::GetInt("Game", "WindowWidth");
  gWindowHeight	= CSettings::GetInt("Game", "WindowHeight");


  // set exit handler
  atexit(Shutdown);

  // seed random number generator
  srand(time(NULL));

  // GLUT Window Initialization:
  glutInit (&argc, argv);
  glutInitWindowSize (gWindowWidth, gWindowHeight);
  glutInitDisplayMode ( GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH);
  glutCreateWindow ("CS248 HW3 - Ryan Barrett");
  //glutFullScreen();

  // Register callbacks:
  glutDisplayFunc(Display);
  glutReshapeFunc(Reshape);
  glutIdleFunc(Update);
  glutKeyboardFunc(Keyboard);
  glutMouseFunc(Mouse);
  glutMotionFunc(Motion);
  glutPassiveMotionFunc(PassiveMotion);

  // create raindrops
  gRaindrops = new CRaindrop[kNumRaindrops];

  InitGraphics();
  gViewTheta = 0;
  gViewPhi = 0;
  UpdateView();

  // start sound
//  gLoopSample.autoMatch(&gSL);
//  gSL.loopSample(&gLoopSample);

  // initialize widget ui stuff
  CWidget::sMouseCallback = GetMouse;

  // start game
  gGame.NewGame(true, true);

  // !!!
  // get extensions
  // !!!

  const GLubyte *exts = glGetString(GL_EXTENSIONS);
  if (!exts)
	printf("no extensions!\n");
  else
	printf("%s", exts);


  return 0;


  // Turn the flow of control over to GLUT
  glutMainLoop ();
  return 0;
}


// -----------------------------------------------------------------------------
// GLUT CALLBACKS
// -----------------------------------------------------------------------------

/* Display
 * -------
 * The display callback given to GLUT - called when the scene needs to be redrawn.
 */
void Display()
{
  // clear frame buffer and depth buffer
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  if (gPrintFPS)
	PrintFPS();

  glEnable(GL_DEPTH_TEST);
  glEnable(GL_CULL_FACE);

  // draw
  if (!gRefract && !gWireframe)
	gFloor.Draw();
  if (gDrawObjects)
	gGame.Draw();
  if (gDrawWater)
	gWater.Draw();

  if (gDrawRain) {
	glDisable(GL_BLEND);
	glDisable(GL_CULL_FACE);
	for (int i = 0; i < kNumRaindrops; i++)
	  gRaindrops[i].Draw();
  }

  if (gMenu)
	CWWindow::DrawAll();

  glutSwapBuffers();
}


/* Reshape
 * -------
 * The reshape callback given to GLUT - called when the OpenGL window is resized.
 * Right now, this immediately asserts.
 * TODO: find out how to disallow window resizing.
 */
void Reshape(GLint width, GLint height)
{
  gWindowWidth = width;
  gWindowHeight = height;

  CWidget::sScreenWidth = gWindowWidth;
  CWidget::sScreenHeight = gWindowHeight;

  glViewport(0, 0, gWindowWidth, gWindowHeight);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(65.0, static_cast<float>(gWindowWidth) / gWindowHeight,
				 kNearPlane, kFarPlane);
  glMatrixMode(GL_MODELVIEW);
}


/* Update
 * ------
 * The update callback given to GLUT - called when GLUT/OpenGL is idle and has
 * nothing to do.
 *
 * The timer code is pretty ugly. Definitely not modular. I should write a Timer
 * class...but since this is the only place I'll have code for calculating time
 * deltas, it's not a high priority.
 */
void Update()
{
  static clock_t lastTick = clock();
  clock_t tick;
  int diff;		// msecs passed since last update

  // get difference between timestamps
  tick = clock();
  assert(tick != -1 && lastTick != -1);
  diff = (tick - lastTick) / kClocksPerMsec;

  // time to update?
  if (diff >= kMinUpdateMsecs) {
	gGame.Update(diff);
	gWater.Update(diff);

	if (gDrawRain) {
	  for (int i = 0; i < kNumRaindrops; i++)
		gRaindrops[i].Update(diff);
	}

	lastTick = tick;
  }

  // update sound
//  gSL.update();

  // display
  glutPostRedisplay();
}


/* Keyboard
 * --------
 * The keyboard callback given to GLUT - called when the user presses an ASCII
 * key.
 */
void Keyboard(unsigned char key, int x, int y)
{
  switch (key) {
  case 27:             // ESCAPE key
	Shutdown();
	exit(0);
	break;

  case '!':		// shift-1
	gGame.NewGame(true, true);
	break;

  case '@':		// shift-2
	gGame.NewGame(false, true);
	break;

  case '#':		// shift-3
	gGame.NewGame(true, false);
	break;

  case '$':		// shift-4
	gGame.NewGame(false, false);
	break;

  case 'n': case 'N':
	gDrawNormals = !gDrawNormals;
	break;

  case 'o': case 'O':
	gDrawObjects = !gDrawObjects;
	break;

  case 'w':
	gDrawWater = !gDrawWater;
	break;

  case 'W':
	gWaves = !gWaves;
	break;
	
  case 's':
	gSound = !gSound;
//	if (gSound)
//	  gSL.resumeSample(&gLoopSample);
//	else
//	  gSL.pauseSample(&gLoopSample);
	break;

  case 'S':
	gSharpEdges = !gSharpEdges;
	break;

  case 'c': case 'C':	// return camera to center
	gViewTheta = 0;
	UpdateView();
	break;

  case 'e': case 'E':
	gEnvMap = !gEnvMap;
	break;

  case 'r':
	gRefract = !gRefract;
	UpdateView();
	break;

  case 'R':
	gDrawRain = !gDrawRain;
	break;

  case 'F':
	gFresnel = !gFresnel;
	break;

  case 'f':
	gPrintFPS = !gPrintFPS;
	break;

  case 't':
	gTexture = !gTexture;
	break;

  case 'T':
	gTransparent = !gTransparent;
	break;

  case 'm':
	ShowMenu();
	break;

  default:
	// is it a move?
	int pos = key - '1';

	if (pos >= 0 && pos <= 8)
	  gGame.Move(pos / 3, pos % 3);

	break;

  }
}


/* Mouse
 * -----
 * Called when a mouse button is pressed or released.
 */
void Mouse(int button, int state, int x, int y)
{
  // viewing is left button only
  if (button == GLUT_LEFT_BUTTON) {
	gLeftButtonDown = (state == GLUT_DOWN);
  }

  // widget ui interaction is right button only
  if (button == GLUT_RIGHT_BUTTON) {

	if (state == GLUT_DOWN) {
	  if (!gMenu)
		ShowMenu(x, y);
	  else if (!CWidget::SendMouseDown()) {
		// mouse click not handled, kill window
		//if (gChildMenu)
		//  DEL(gChildMenu)
		//else
		  DEL(gMenu)
	  }
	}

	if (state == GLUT_UP)
	  CWidget::SendMouseUp();
  }
}


/* Motion
 * ------
 * Called when the mouse moves within the window (only while a button is pressed).
 */
void Motion(int x, int y)
{
  static int lastx = -1, lasty = -1;

  // set the global mouse cursor position
  PassiveMotion(x, y);

  if (!gLeftButtonDown)
	return;

  if ((lastx == -1 && lasty == -1) || abs(x - lastx) > 20 || abs(y - lasty) > 20) {
	lastx = x;
	lasty = y;
  }

  // update angles
  gViewPhi -= (y - lasty) / kDeltaAngle;
  gViewTheta += (x - lastx) / kDeltaAngle;

  // clip angles
  if (gViewPhi > 0)
	gViewPhi = 0;
  if (gViewPhi < kMinPhi)
	gViewPhi = kMinPhi;
  if (gViewTheta < 0)
	gViewTheta += 360;
  if (gViewTheta > 360)
	gViewTheta -= 360;

  // update view
  UpdateView();

  // set last mouse cursor point
  lastx = x;
  lasty = y;
}


/* PassiveMotion
 * -------------
 * Called when the mouse moves within the window (whether or not a button is
 * pressed).
 */
void PassiveMotion(int x, int y)
{
  gMouseX = x;
  gMouseY = y;
}


// -----------------------------------------------------------------------------
// OTHER FUNCTIONS
// -----------------------------------------------------------------------------

/* InitGraphics
 * ------------
 * Initializes the OpenGL engine, including rendering options and the projection
 * and modelview matrices.
 */
void InitGraphics()
{
  // Initialize OpenGL graphics state
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);
  glShadeModel(GL_SMOOTH);

  // line stuff (for raindrops)
  glLineWidth(CSettings::GetFloat("Rain", "Width"));

  // culling
  glEnable(GL_CULL_FACE);
  glCullFace(GL_BACK);
  glFrontFace(GL_CCW);			// winding order counterclockwise

  // texture stuff
  glEnable(GL_TEXTURE_2D);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);

  // set up projection transformation
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(65.0, (float)gWindowWidth / gWindowHeight, kNearPlane, kFarPlane);

  // set up lighting
  glEnable(GL_LIGHTING);

  // ambient
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, kAmbient);

  // diffuse
  glLightfv(GL_LIGHT0, GL_POSITION, kLightPos);
  glEnable(GL_LIGHT0);
  //glEnable(GL_NORMALIZE);		// because its code is probably faster than mine
}


/* UpdateView
 * ----------
 * Resets the modelview matrix to the current view (given by gViewPhi and
 * gViewTheta.
 */
void UpdateView()
{
  static const int maxRefractPhi = CSettings::GetInt("Game", "MaxRefractPhi");

  // set pos and dir vectors
  gViewPos.x = kViewDistance * cos(gViewTheta * M_PI / 180);
  gViewPos.y = kViewDistance * sin(gViewTheta * M_PI / 180);
  gViewPos.z = kViewDistance * cos(gViewPhi * M_PI / 180);

  gViewDir = gViewPos;	// since the viewer is always looking at the origin
  gViewDir.Normalize();

  // set modelview matrix
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  gluLookAt(0, 0, kViewDistance, 0, 0, 0, 0, 1, 0);
  glRotatef(gViewPhi, 1, 0, 0);
  glRotatef(gViewTheta, 0, 0, 1);

  // should we turn off refraction?
  if (gRefract && gViewPhi <= maxRefractPhi)
	gRefract = false;
}


/* PrintFPS
 * --------
 * If the FPS debugging setting is turned on, this function is called every time
 * the game draws a frame. It counts the frames that are printed until a second has
 * gone by, and prints the fps measurement to the screen.
 *
 * NOTE: this is buggy. I probably won't fix it. :P
 */
void PrintFPS()
{
  static int frames = 0, diff = 0;
  static clock_t lastTick = clock();
  clock_t tick;

  // get difference between timestamps
  tick = clock();
  assert(tick != -1 && lastTick != -1);
  diff += (tick - lastTick);
  lastTick = tick;

  frames++;

  // print fps
  if (diff > kClocksPerMsec * 1000) {
	cout << frames << " fps" << endl;
	frames = 0;
	diff = 0;
  }
}


/* Shutdown
 * --------
 * Called when the game exits. Right now, all this does is dump the ini file to
 * disk.
 *
 * NOTE: commented out for now.
 */
void Shutdown()
{
  //CSettings::Save();

  //delete[] gRaindrops;
}



